/*++
sinh -- the hyperbolic sine

sinh(x)

x - expression
++*/

sinh:=
proc(x)
  option noDebug;
  local f, y;
begin
  if args(0) = 0 then error("no arguments given")
  elif x::dom::sinh <> FAIL then return(x::dom::sinh(args()))
  elif args(0) <> 1 then error("wrong no of args") 
  end_if;
  
  case type(x)
  of DOM_INT do
  of DOM_RAT do
     if x < 0 then return(-sinh(-x)) end_if;
     break;
  of DOM_FLOAT do
     return(sinh::float(x));
  of DOM_COMPLEX do
     if domtype(op(x,1)) = DOM_FLOAT or
        domtype(op(x,2)) = DOM_FLOAT then
        return(sinh::float(x))
      end_if;
      if op(x,1) = 0 then return(sin(-x*I)*I) end_if;
      break;
  of DOM_SET do
  of "_union" do
     return(map(x, sinh))
  of "_intersect" do
  of "_minus" do
     return(Dom::ImageSet(sinh(`#z`), `#z`, x))
  end_case;

  if not testtype(x,Type::Arithmetical) then
    /* generic handling of sets */
    if testtype(x, Type::Set) then
      return(Dom::ImageSet(sinh(#x), #x, x));
    end_if;

    error("argument must be of 'Type::Arithmetical'")
  end_if;

  case type(x)
  of "_mult" do
       f:= op(x, nops(x));
       if testtype(f, Type::Real) then
          if f < 0 then return(-sinh(-x)) end_if;
       elif type((f:=x/I/PI))=DOM_INT then return(0) // sinh(n*I*PI)=0 
       elif type(2*f)=DOM_INT then return((-1)^(f-1/2)*I)
       end_if;
       f:= x/PI/I;
       if contains({DOM_RAT, DOM_INT}, domtype(f)) then
         if f < 0 then return(-sinh(-x)) end_if;
         if f < 1/2 then break end_if;
         if f < 1 then return(sinh((1-f)*PI*I)) end_if;
         if f < 2 then return(-sinh((2-f)*PI*I)) end_if;
         return(sinh((f-2*(floor(f) div 2))*PI*I));
       end_if;
       f:= x/I;
       if type(f) = "arg" then  // x = sinh(I*arg(y)) = I*Im(y)/abs(y)
         return(I*Im(op(f))/abs(op(f))):
       end_if:

      break;
  of "_plus" do
       // due to performance: move this code to sinh::simplify
       // if sysorder(-x,x)=TRUE then return(-sinh(-x)) else break end_if

       // handle sinh(y + integer*PI) -> (-1)^integer*sinh(y)
       // We could also do sinh(y + PI/2) -> -cosh(y).
       // At the moment, we decide not to do it.
       if has(x, PI) then
          for y in x do
              f:= y/PI/I;
              if testtype(f, DOM_INT) and f<>0 then
                 return((-1)^f*sinh(x-y))
              end_if;
          end_for:
       end_if:
       break;
  // cf Table 4.5.60 p. 85 of Abramowitz and Stegun 
  of "arcsinh" do return(op(x))
  of "arccosh" do return(sqrt(op(x)^2-1))
  of "arctanh" do return(op(x)/sqrt(1-op(x))/sqrt(1+op(x)))
  //of "arccoth" do return(sqrt(1/(op(x)^2-1)))
  of "arccoth" do 
       x:= 1/op(x): // note that op(x) <> 0, because arccoth(0) is
                    // simplified to I*PI/2
       return(x/sqrt(1-x)/sqrt(1+x))
  end_case;

  procname(x)
end_proc:

sinh := prog::remember(sinh, 
  () -> [property::depends(args()), DIGITS, slotAssignCounter("sinh")]):

sinh(0):= 0:
sinh(I*PI):= 0:
sinh(I*PI/2):= I:
sinh(I*PI/3):= I*(3^(1/2))/2:
sinh(I*PI/4):= I*(2^(1/2))/2:
sinh(I*PI/  5):= I*1/4*2^(1/2)*(5-5^(1/2))^(1/2):
sinh(I*PI*2/5):= I*1/4*2^(1/2)*(5+5^(1/2))^(1/2):
sinh(I*PI/6):= I*1/2:
sinh(I*PI/  8):= I*((2-2^(1/2))^(1/2))/2:
sinh(I*PI*3/8):= I*((2+2^(1/2))^(1/2))/2:
sinh(I*PI/  10):= I*1/4*(5^(1/2)-1):
sinh(I*PI*3/10):= I*1/4*(5^(1/2)+1):
sinh(I*PI/  12):= I*(6^(1/2)-2^(1/2))/4: // simpler th. ((2-3^(1/2))^(1/2))/2: 
sinh(I*PI*5/12):= I*(2^(1/2)+6^(1/2))/4: // = ((2+3^(1/2))^(1/2))/2:

sinh( infinity):= infinity:
sinh(-infinity):=-infinity:

sinh:= funcenv(sinh, op(specfunc::sinh, 2)):
sinh::info := "sinh -- the hyperbolic sine":
sinh::print := "sinh":
sinh::type := "sinh":
sinh::float := specfunc::sinh:

sinh::hull := DOM_INTERVAL::sinh@hull:

sinh::inverse := "arcsinh":

sinh::diff :=
  proc(f)
    local op1;
  begin
    op1 := op(f, 1);
    diff(op1, args(2..args(0))) * cosh(op1)
  end_proc:

sinh::expand :=
    loadproc(sinh::expand, pathname("STDLIB","EXPAND"),"sinh"):

sinh::conjugate := 
    loadproc(sinh::conjugate, pathname("STDLIB","CONJ"),"sinh"):

sinh::rectform :=
    loadproc(sinh::rectform, pathname("STDLIB","RECTFORM"),"sinh"):

sinh::series :=
    loadproc(sinh::series, pathname("SERIES"),"sinh"):

sinh::simplify :=
    loadproc(sinh::simplify, pathname("STDLIB","SIMPLIFY"),"sinh"):

sinh::Re := loadproc(sinh::Re, pathname("STDLIB","RE"),"sinh"):

sinh::Im := loadproc(sinh::Im, pathname("STDLIB","IM"),"sinh"):

sinh::"transform::laplace":=
    loadproc( sinh::"transform::laplace",
        pathname("TRANS","LAPLACE"), "L_sinh"):

sinh::Content := stdlib::genOutFunc("Csinh", 1):

// end of file 
