/*++
tanh -- the hyperbolic tangens

tanh(x)

x - expression
++*/

tanh:=
proc(x)
  option noDebug;
  local f, y;
begin
  if args(0) = 0 then error("no arguments given")
  elif x::dom::tanh <> FAIL then return(x::dom::tanh(args()))
  elif args(0) <> 1 then error("wrong no of args") 
  end_if;
  
  case type(x)
  of DOM_INT do
  of DOM_RAT do
     if x < 0 then return(-tanh(-x)) end_if;
     break;
  of DOM_FLOAT do
     return(tanh::float(x));
  of DOM_COMPLEX do
     if domtype(op(x,1)) = DOM_FLOAT or
        domtype(op(x,2)) = DOM_FLOAT then
         return(tanh::float(x))
     end_if;
     if op(x,1) = 0 then return(tan(-x*I)*I) end_if;
     break;
  of DOM_SET do
  of "_union" do
     return(map(x, tanh))
  of "_intersect" do
  of "_minus" do
     return(Dom::ImageSet(tanh(`#z`), `#z`, x))
  end_case;

  if not testtype(x,Type::Arithmetical) then
    /* generic handling of sets */
    if testtype(x, Type::Set) then
      return(Dom::ImageSet(tanh(#x), #x, x));
    end_if;

    error("argument must be of 'Type::Arithmetical'")
  end_if;

  case type(x)
  of "_mult" do
      f:= op(x, nops(x));
      if testtype(f, Type::Real) then
          if f < 0 then return(-tanh(-x)) end_if;
      elif type((f:=x/I/PI))=DOM_INT then return(0) // tanh(k*I*PI)=0 
      elif type(2*f)=DOM_INT then
         //return((-1)^(f-1/2)*I*infinity)
           error("singularity");
      end_if;
      f:= x/I/PI;
      if contains({DOM_RAT, DOM_INT}, domtype(f)) then
       if f < 0 then return(-tanh(-x)); end_if;
       if f < 1/2 then break; end_if;
       if f = 1/2 then error("singularity"); end_if;
       if f < 1 then return(-tanh((1-f)*PI*I)) end_if;
       if f < 2 then return(tanh((f-1)*PI*I)) end_if;
       return(tanh((f-floor(f))*PI*I));
      end_if:
      f:= x/I;
      if type(f) = "arg" then  // x = tanh(I*arg(z)) = I*Im(z)/Re(z)
        x:= Re(op(f)); 
        if iszero(x) then
           error("singularity")
        else
           return(I*Im(op(f))/x):
        end_if;
      end_if:
      break;
  of "_plus" do
      // due to performance: move this code to tanh::simplify
      // if sysorder(-x,x)=TRUE then return(-tan(-x)) else break end_if

      // handle tanh(x + integer*PI*I) -> tanh(y)
      // We could also do tanh(x + PI/2*I) -> -coth(y).
      // At the moment, we decide not to do it.
      if has(x, PI) then
        for y in x do
           f:= y/PI/I;
           if testtype(f, DOM_INT) and f<>0 then
              return(tanh(x-y))
           end_if;
        end_for:
      end_if:
      break;
    // Table 4.5.60 p. 85 of Abramowitz and Stegun 
  of "arctanh" do return(op(x))
  of "arccoth" do return(1/op(x))
  of "arcsinh" do return(op(x)/sqrt(1+op(x)^2))
  of "arccosh" do return(sqrt(op(x)^2-1)/op(x))

  end_case;

  procname(x)
end_proc:

tanh := prog::remember(tanh, 
  () -> [property::depends(args()), DIGITS, slotAssignCounter("tanh")]):

tanh(0):= 0:
tanh(I*PI):= 0:
tanh(  I*PI/3 ):= I*(3^(1/2)):
tanh(  I*PI/4 ):= I:
tanh(  I*PI/5 ):= I*(5 - 2*5^(1/2))^(1/2):
tanh(2*I*PI/5 ):= I*(1 + 5^(1/2)) * (2*5^(1/2) + 10)^(1/2)/4:
tanh(  I*PI/6 ):= I*(3^(1/2))/3:
tanh(  I*PI/8 ):= I*(2^(1/2)-1):
tanh(3*I*PI/8 ):= I*(2^(1/2)+1):
tanh(  I*PI/10):= I*1/5*5^(1/2)*(5 - 2*5^(1/2))^(1/2):
tanh(3*I*PI/10):= I*1/5*5^(1/2)*(5 + 2*5^(1/2))^(1/2):
tanh(  I*PI/12):= I*(2-3^(1/2)):
tanh(5*I*PI/12):= I*(2+3^(1/2)):
tanh(I*   PI/24):= I*(-2 + 2^(1/2) - 3^(1/2) + 2^(1/2)*3^(1/2)):
tanh(I* 5*PI/24):= I*(-2 - 2^(1/2) + 3^(1/2) + 2^(1/2)*3^(1/2)):
tanh(I* 7*PI/24):= I*( 2 - 2^(1/2) - 3^(1/2) + 2^(1/2)*3^(1/2)):
tanh(I*11*PI/24):= I*( 2 + 2^(1/2) + 3^(1/2) + 2^(1/2)*3^(1/2)):

tanh(infinity):=1:
tanh(-infinity):=-1:

tanh:= funcenv(tanh, op(specfunc::tanh, 2)):
tanh::info := "tanh -- the hyperbolic tangent":
tanh::print := "tanh":
tanh::type := "tanh":
tanh::float :=specfunc::tanh:

tanh::hull := DOM_INTERVAL::tanh@hull:

tanh::inverse := "arctanh":

tanh::undefined :=
    loadproc(tanh::undefined, pathname("STDLIB","DISCONT"), "tanh"):
tanh::realDiscont := {}:
tanh::complexDiscont :=
    loadproc(tanh::complexDiscont, pathname("STDLIB","DISCONT"), "tanh"):

tanh::diff :=
  proc(f)
    local op1;
  begin
    op1 := op(f, 1);
    diff(op1, args(2..args(0))) * (1 - tanh(op1)^2)
  end_proc:

tanh::expand :=
loadproc(tanh::expand, pathname("STDLIB", "EXPAND"), "tanh"):

tanh::conjugate :=
    loadproc(tanh::conjugate, pathname("STDLIB","CONJ"), "tanh"):

tanh::rectform :=
    loadproc(tanh::rectform, pathname("STDLIB","RECTFORM"), "tanh"):

tanh::series :=
  proc(f,x,n,dir)
  begin
    Series::series(sinh(f),x,n,dir)/Series::series(cosh(f),x,n,dir)
  end_proc:

tanh::simplify :=
    loadproc(tanh::simplify, pathname("STDLIB","SIMPLIFY"), "tanh"):

tanh::Re := loadproc(tanh::Re, pathname("STDLIB","RE"), "tanh"):

tanh::Im := loadproc(tanh::Im, pathname("STDLIB","IM"), "tanh"):

tanh::Content := stdlib::genOutFunc("Ctanh", 1):

// end of file 
