/*
whittakerM(a,b,z) is, with whittakerW(a,b,z), a solution of the Whittaker ODE :
y''(z)+(-1/4+a/z+(1/4-b^2)/z^2)y(z)=0
It can be expressed with the KummerM function (solution of the ODE
zy''(z)+(B-z)y'(z)-Ay(z)) which is equal to the Gauss hypergeometric function
1F1(A,B,z) :
whittakerW(a,b,z)=exp(-1/2*z)*z^(1/2+b)*1F1(A,B,z)
with A=1/2+b-a and B=1+2*b.

Some particular cases are implemented for particular values of a and b, due
to the particular cases of the 1F1.

Reference : Abramowitz and Stegun (A&S), Chap. 13
*/

whittakerM:=
proc(a, b, z)
  local fa, fb, fz, c, rc, k, f, rb, A, B, rA, rB;
begin
  if args(0) <> 3 then
    error("wrong number of arguments")
  elif z::dom::whittakerM <> FAIL then
    return( z::dom::whittakerM(args()) )
  end_if:

  if not (testtype(a, Type::Arithmetical) and
          testtype(b, Type::Arithmetical) and
          testtype(z, Type::Arithmetical)) then
    error("arguments must be of 'Type::Arithmetical'")
  end_if;

  fa:= domtype(float(a));
  fb:= domtype(float(b));
  fz:= domtype(float(z));
  if (fa=DOM_FLOAT or fa=DOM_COMPLEX) and 
     (fb=DOM_FLOAT or fb=DOM_COMPLEX) and 
     (fz=DOM_FLOAT or fz=DOM_COMPLEX) and 
     has([map((op(a),op(b),op(z)),type)],DOM_FLOAT) then
      return(whittakerM::float(a,b,z))
  end_if:

  // The leading term of the expansion around z=0 is
  // whittakerM(a,b,z) = z^(1/2 + b)*(1 + O(z))
  if iszero(z) and 
    (fb=DOM_COMPLEX or fb=DOM_FLOAT) then
    if iszero(b + 1/2) then
       error("whittakerM is not defined for these parameter values");
    elif Re(b) + 1/2 <= 0 then
       error("singularity")
    else
       return(0)
    end_if;
  end_if:

  // a+b+1/2=0 <=> A=B and  1F1(A,A,z)=exp(z) :
  if iszero(a+b+1/2) and not(iszero(z)) then
    return(z^(1/2+b)*exp(1/2*z))
  end_if:

  // if A=1/2+b-a is a negative integer then 1F1(A,B,z) can be expressed
  // in terms of Laguerre polynomials (c.f. A&S 13.6.9):
  c:=-1/2-b+a; 
  rc:=round(c);
  if domtype(rc)=DOM_INT and rc >= 0 and iszero(frac(c)) and
    not(iszero(z)) then 
    f:=pochhammer(1+2*b,rc);
    if iszero(f) then
      error("whittakerM is not defined for these parameter values");
    else
      return(exp(-z/2)*z^(b+1/2)*rc!/f*laguerreL(rc, 2*b, z));
    end_if:
  end_if:

  // B=2A <=> a=0 and 1F1(A,2A,z) can be expressed in terms of modified
  // bessel function (c.f. A&S 13.6.3) :
  if iszero(a) then
    rb:=round(1+b);
    if not(iszero(1+b) or (domtype(rb)=DOM_INT and rb < 0 and iszero(frac(1+b))))
      and not(iszero(z)) then
        return(eval(gamma(b+1)*4^b*sqrt(z)*besselI(b,z/2)))
    end_if;
  end_if:

  // If 0<=B<=A and A and B integers then
  // 1F1(A,B,z)=exp(z)*sum_(k=0..n-m)((m-n)_k(-z)^k/k!(m)_k) :
  A:=1/2+b-a; B:=1+2*b;
  rA:=round(A); 
  rB:=round(B);
  if domtype(rA)=DOM_INT and rA > 0 and iszero(frac(A)) and 
     domtype(rB)=DOM_INT and rB > 0 and iszero(frac(B)) and rB<=rA then
    f:=_plus(pochhammer(rB-rA,k)*(-z)^k/fact(k)/pochhammer(rB,k) $ k=0..rA-rB);
    return(exp(z/2)*z^(1/2+b)*f):
  end_if:

  //  for all the remaining cases when 1+2b is zero or a negative integer, the
  //  1F1 is not defined so we return undefined :
  rb:=round(1+2*b);
  if (fa=DOM_COMPLEX or fa=DOM_FLOAT) and 
     domtype(rb)=DOM_INT and rb <= 0 and iszero(frac(1+2*b)) then
      error("whittakerM is not defined for these parameter values"):
  end_if:

  procname(a, b, z)        
end_proc:

whittakerM := prog::remember(whittakerM, 
  () -> [property::depends(args()), DIGITS, slotAssignCounter("whittakerM")]):

whittakerM := funcenv(whittakerM):
whittakerM::type := "whittakerM":
whittakerM::print := "whittakerM":
whittakerM::info := "whittakerM(a,b,z) -- confluent hypergeometric function,
solution of the Whittaker differential equation.":

whittakerM::float :=
proc()
local a, b, z, fa, fb, fz, rb, c, rc, k, f, fl, A, B, rA, rB;
begin
  a:=args(1); b:=args(2); z:=args(3);
  fa:=float(a); fb:=float(b); fz:=float(z);
  if not((type(fa)=DOM_FLOAT or type(fa)=DOM_COMPLEX) and
         (type(fb)=DOM_FLOAT or type(fb)=DOM_COMPLEX) and
         (type(fz)=DOM_FLOAT or type(fz)=DOM_COMPLEX)) then
    return(whittakerM(fa,fb,fz))
  end_if:

  if iszero(z) then
    // The leading term of the expansion around z=0 is
    // whittakerM(a,b,z) = z^(1/2 + b)*(1 + O(z))
    if iszero(b + 1/2) then
       error("whittakerM is not defined for these parameter values");
    elif Re(b) + 1/2 <= 0 then
       error("singularity")
    else
       return(float(0))
    end_if;
  end_if:

  if iszero(a+b+1/2) then
    return(float(z^(1/2+b)*exp(1/2*z)))
  end_if:

  c:=-1/2-b+a; 
  rc:=round(c);
  if domtype(rc)=DOM_INT and rc >= 0 and iszero(frac(c)) and
    not(iszero(z)) then 
    f:=pochhammer(1+2*b,float(rc));
    if iszero(f) then
      error("whittakerM is not defined for these parameter values");
    else
      return(exp(-fz/2)*fz^(b+1/2)*gamma(rc+1)/f*laguerreL(rc, 2*b, fz));
    end_if:
  end_if:

  if iszero(a) then
    rb:=round(1+b);
    if not(domtype(rb)=DOM_INT and rb <= 0 and iszero(frac(1+b))) then
        return(float(gamma(b+1)*4^b*sqrt(z)*besselI(b,z/2)))
    end_if;
  end_if:

  A:=1/2+b-a; B:=1+2*b;
  rA:=round(A); rB:=round(B);
  if domtype(rA)=DOM_INT and rA > 0 and iszero(frac(A)) and 
     domtype(rB)=DOM_INT and rB > 0 and iszero(frac(B)) and rB<=rA then
    f:=_plus(pochhammer(rB-rA,k)*(-z)^k/fact(k)/pochhammer(rB,k) $ k=0..rA-rB);
    return(float(exp(z/2)*z^(1/2+b)*f)):
  end_if:

  rb:=round(1+2*b);
  if domtype(rb)=DOM_INT and rb<=0 and iszero(frac(1+2*b)) then
     error("whittakerM is not defined for these parameter values");
  end_if:

  if traperror((fl:=float(exp(-1/2*z)*z^(1/2+b)*hypergeom([1/2+b-a],[1+2*b],z),args(4..args(0)))))=0 then
    fl
  else
    error("numeric exception")
  end_if;
end_proc:


// derivative of whittakerM is given in A&S 13.4.32.
// Combined with 13.4.29 one has also the relation :
//  z*M'(a,b,z)=(-1/2*z+a)*M(a,b,z)+(1/2+b-a)*M(a-1,b,z)
whittakerM::diff :=
proc(e, x)
local a, b, f, Select;
begin
  if has({op(e,1),op(e,2)}, [args(2..args(0))]) then hold(diff)(args())
  elif args(0)>2 then diff(diff(e,x),args(3..args(0)))
  else
    a:=op(e,1); b:=op(e,2); f:=op(e,3);

    Select:= X -> _lazy_or(_lazy_and(type(X) = "_plus",
           _lazy_and(select([op(X)], testtype, Type::NegRat) = [],
                   not(Select(select(_plus(op(X)), testtype, DOM_COMPLEX))))),
         testtype(Re(X), Type::PosRat) ):

    if Select(a) then
        ((-1/2+a/f)*whittakerM(a,b,f)+1/f*(1/2-a+b)*whittakerM(a-1,b,f))
         *diff(f,x)
    else
      ((1/2-a/f)*whittakerM(a,b,f)+1/f*(1/2+a+b)*whittakerM(a+1,b,f))*diff(f,x)
    end_if:
  end_if:
end_proc:


whittakerM::series :=
proc(a, b, z)
local ser;
begin
  if traperror((
        ser:=series(exp(-1/2*z)*z^(1/2+b)*hypergeom([1/2+b-a],[1+2*b],z),args(4..args(0)))))=0 
        and not type(ser) = "series" 
        and not has(ser, hypergeom)
    then
      return(ser)
  else
    return(Series::unknown(whittakerM(a,b,z),args(4..args(0))))
  end_if;
end_proc:


whittakerM::Content := stdlib::genOutFunc("CwhittakerM", 3):
