/*
whittakerW(a,b,z) is, with whittakerM(a,b,z), a solution of the Whittaker ODE :
y''(z)+(-1/4+a/z+(1/4-b^2)/z^2)y(z)=0
It can be expressed with the KummerM function (solution of the ODE
z*y''(z)+(B-z)y'(z)-Ay(z)) ::
whittakerW(a,b,z)=exp(-1/2*z)*z^(1/2+b)*KummerM(A,B,z)
with A=1/2+b-a and B=1+2*b.

Some particular cases are implemented for particular values of a and b, due
to the particular cases of the KummerM.

Reference : Abramowitz and Stegun (A&S), Chap. 13
*/

alias(isinteger = (x -> (iszero(frac(x)) and domtype(float(x)) = DOM_FLOAT))):

whittakerW:=
proc(a, b, z)
  local fa, fb, fz, kum;
begin
  if args(0) <> 3 then
    error("wrong number of arguments")
  elif z::dom::whittakerW <> FAIL then
    return( z::dom::whittakerW(args()) )
  end_if:

  if not (testtype(a, Type::Arithmetical) and
          testtype(b, Type::Arithmetical) and
          testtype(z, Type::Arithmetical)) then
    error("arguments must be of 'Type::Arithmetical'")
  end_if:

  fa:= domtype(float(a));
  fb:= domtype(float(b));
  fz:= domtype(float(z));
  if (fa=DOM_FLOAT or fa=DOM_COMPLEX) and 
     (fb=DOM_FLOAT or fb=DOM_COMPLEX) and 
     (fz=DOM_FLOAT or fz=DOM_COMPLEX) and 
     has([map((op(a),op(b),op(z)),type)],DOM_FLOAT) then
      return(whittakerW::float(a,b,z))
  end_if:

  if iszero(z) then
   // The series representation around z= 0 is whittakerW(a, b, z)
   // =   gamma( 2*b)/gamma( b-a+1/2)*z^(1/2-b) * (1 - a*z/(1-2*b) + ...)
   //   + gamma(-2*b)/gamma(-b-a+1/2)*z^(1/2+b) * (1 - a*z/(1+2*b) + ...)
   // The exponents 1/2-b and 1/2+b of z have a real part > 0 iff |Re(b)| < 1/2.
   // In this case, everything goes to zero as z -> 0. 
   // If |Re(b) > 1/2, there is a singularity as z -> 0, unless
   // the term is zero for any z, anyway, because we divide by
   // a singularity of gamma.
   if contains({DOM_FLOAT, DOM_COMPLEX}, fb) then 
    if abs(Re(b)) < 1/2 then
       return(0);
    elif abs(Re(b)) > 1/2 then
       if (not (domtype(b) = DOM_INT and b <= 0)) and
          domtype(b - a + 1/2) = DOM_INT and b - a + 1/2 <= 0 and
          Re(b) > -1/2 then
          return(0)
       elif (not (domtype(b) = DOM_INT and b >= 0)) and
          domtype(-b - a + 1/2) = DOM_INT and -b - a + 1/2 <= 0 and
          Re(b) < 1/2 then
          return(0)
       else
          error("singularity");
       end_if:
    elif iszero(Re(b) - 1/2) then
       if domtype(b - a + 1/2) = DOM_INT and b-a+1/2 <= 0 then
         return(0)
       elif iszero(b - 1/2) then  
         return(gamma(2*b)/gamma(b - a + 1/2));
       else
          error("singularity");
       end_if;
    elif iszero(Re(b) + 1/2) then
       if domtype(-b - a + 1/2) = DOM_INT and -b-a+1/2 <= 0  then
          return(0);
       elif iszero(b + 1/2) then  
         return(gamma(-2*b)/gamma(-b - a + 1/2));
       else
         error("singularity");
       end_if;
    else
       return(gamma( 2*b)/gamma( b - a + 1/2)
            + gamma(-2*b)/gamma(-b - a + 1/2)):
    end_if;
   end_if:
  end_if:

  if traperror((kum:=kummerU(1/2+b-a,1+2*b,z)))=0 and
     type(kum) <> "kummerU" then
      return(exp(-1/2*z)*z^(1/2+b)*kum);
  end_if:

  procname(a, b, z)        
end_proc:

whittakerW := prog::remember(whittakerW, 
  () -> [property::depends(args()), DIGITS, slotAssignCounter("whittakerW")]):

whittakerW := funcenv(whittakerW):
whittakerW::type := "whittakerW":
whittakerW::print := "whittakerW":
whittakerW::info := "whittakerW(a,b,z) -- confluent hypergeometric function,
solution of the Whittaker differential equation.":

whittakerW::float :=
proc(a, b, z)
local fa, fb, fz, fl;
begin
  fa:=float(a); fb:=float(b); fz:=float(z);
  if not((type(fa)=DOM_FLOAT or type(fa)=DOM_COMPLEX) and
         (type(fb)=DOM_FLOAT or type(fb)=DOM_COMPLEX) and
         (type(fz)=DOM_FLOAT or type(fz)=DOM_COMPLEX)) then
    return(whittakerW(fa,fb,fz))
  end_if:

  // The series representation around z= 0 is whittakerW(a, b, z)
  // =   gamma( 2*b)/gamma( b-a+1/2)*z^(1/2-b) * (1 - a*z/(1-2*b) + ...)
  //   + gamma(-2*b)/gamma(-b-a+1/2)*z^(1/2+b) * (1 - a*z/(1+2*b) + ...)
  // The exponents 1/2-b and 1/2+b of z have a real part > 0 iff |Re(b)| < 1/2.
  // In this case, everything goes to zero as z -> 0. 
  // If |Re(b) > 1/2, there is a singularity as z -> 0, unless
  // the term is zero for any z, anyway, because we divide by
  // a singularity of gamma.
  if iszero(z) then
    if abs(Re(b)) < 1/2 then
       return(float(0));
    elif abs(Re(b)) > 1/2 then
       if (not (isinteger(b) and b <= 0)) and
          isinteger(b - a + 1/2) and b - a + 1/2 <= 0 and
          Re(b) > -1/2 then
          return(float(0))
       elif (not (isinteger(b) and b >= 0)) and
          isinteger(-b - a + 1/2) and -b - a + 1/2 <= 0 and
          Re(b) < 1/2 then
          return(float(0))
       else
          error("singularity");
       end_if:
    elif iszero(Re(b) - 1/2) then
       if isinteger(b - a + 1/2) and b-a+1/2 <= 0 then
         return(float(0))
       elif iszero(b - 1/2) then  
         return(float(gamma(2*b)/gamma(b - a + 1/2)));
       else
          error("singularity");
       end_if;
    elif iszero(Re(b) + 1/2) then
       if isinteger(-b - a + 1/2) and -b-a+1/2 <= 0  then
          return(float(0));
       elif iszero(b + 1/2) then  
         return(float(gamma(-2*b)/gamma(-b - a + 1/2)));
       else
          error("singularity");
       end_if;
    else
       return(float( gamma( 2*b)/gamma( b - a + 1/2)
                   + gamma(-2*b)/gamma(-b - a + 1/2))):
    end_if;
  end_if:
  if traperror((fl:=float(exp(-1/2*z)*z^(1/2+b)*
                          kummerU(1/2+b-a,1+2*b,z)
                         ,args(4..args(0)))))=0 then
    return(fl)
  else
    error("numeric exception, division by zero")
  end_if;
end_proc:


// derivative of whittakerW is given in A&S 13.4.33.
// Combined with 13.4.31 one has also the relation :
//  z*W'(a,b,z)=(-1/2*z+a)*W(a,b,z)-(1/2+b-a)*(-1/2+a+b)*W(a-1,b,z)
whittakerW::diff :=
proc(e, x)
local a, b, f, Select;
begin
  if has({op(e,1),op(e,2)}, [args(2..args(0))]) then hold(diff)(args())
  elif args(0)>2 then diff(diff(e,x),args(3..args(0)))
  else
    a:=op(e,1); b:=op(e,2); f:=op(e,3);

    Select:= X -> _lazy_or(_lazy_and(type(X) = "_plus",
           _lazy_and(select([op(X)], testtype, Type::NegRat) = [],
                   not(Select(select(_plus(op(X)), testtype, DOM_COMPLEX))))),
         testtype(Re(X), Type::PosRat) ):
    
    if Select(a) then
        ((-1/2+a/f)*whittakerW(a,b,f)-1/f*(-1/4+a-a^2+b^2)*whittakerW(a-1,b,f))
          *diff(f,x)
    else  
      ((1/2-a/f)*whittakerW(a,b,f)-1/f*whittakerW(a+1,b,f))*diff(f,x)
    end_if:
  end_if:
end_proc:


whittakerW::series :=
proc()
local a, b, z, ser;
begin
  a:=args(1); b:=args(2); z:=args(3);
  if traperror((ser:=series(exp(-1/2*z)*z^(1/2+b)*kummerU(1/2+b-a,1+2*b,z),args(4..args(0)))))=0 
     and not type(ser)="series" 
     and not has(ser,kummerU) then
    return(ser)
  else
    return(Series::unknown(whittakerW(a,b,z),args(4..args(0))))
  end_if;
end_proc:

whittakerW::Content := stdlib::genOutFunc("CwhittakerW", 3):
