/*

Call:          stats::exponentialCDF(a, b)

Parameter(s):  a, b - arithmetical expressions

Returns: a procedure

------------------------------------*/

stats::exponentialCDF:= proc(a, b)
local fa, fb;
option escape;
begin
  if args(0) <> 2 then 
     error("expecting two arguments")
  end_if:

  // ------------- check a -------------
  fa:= float(a):
  if domtype(fa) = DOM_COMPLEX then
     error("the location parameter must be real");
  end_if;

  // ------------- check b -------------
  fb:= float(b):
  if domtype(fb) = DOM_FLOAT and fb <= 0 then
     error("the scale parameter must be positive"):
  end_if;
  if domtype(fb) = DOM_COMPLEX then
     error("the scale parameter must be real");
  end_if;

  //-------------------------------
  // return the following procedure
  //-------------------------------
  proc(x)
  local aa, bb, fa, fb, fx,
        amx, famx, fbamx, absfbamx
       ;
  save DIGITS;
  begin
    if args(0) <> 1 then
       error("one argument expected") 
    end_if:

    // ------------- check a -------------
    aa:= context(a):
    fa:= float(aa):
    if domtype(fa) = DOM_COMPLEX then
       error("the location parameter must be real");
    end_if;
 
    // ------------- check b -------------
    bb:= context(b);
    fb:= float(bb):
    if domtype(fb) = DOM_FLOAT and fb <= 0 then
       error("the scale parameter must be positive"):
    end_if;
    if domtype(fb) = DOM_COMPLEX then
       error("the scale parameter must be real");
    end_if;
 
    // ------------- check x -------------
    fx:= float(x);
    if domtype(fx) = DOM_COMPLEX then
       error("expecting a real argument");
    end_if;

    amx := aa - x;

    // --------- float evaluation ? ----------
    // If the argument x is a real or complex float, 
    // then apply float to all parameters.
    if domtype(x) = DOM_FLOAT and
       domtype(fa) = DOM_FLOAT and
       domtype(fb) = DOM_FLOAT then
       famx:= float(amx);
       if famx >= 0 then
            return(float(0))
       else fbamx:= fb*famx;
            absfbamx:= specfunc::abs(fbamx):
            // avoid numerical instability of
            // 1 - exp(b*(a-x)) due to cancellation,
            // if b*(a-x) is small
            // Use 
            //  1 - exp(fbamx) = - fbamx - fbamx^2/2! - ...
            // for stabilization
            if absfbamx > 1/10
            then  // we do not loose more than
                  // one significant digit due to
                  // cancellation in 1 - exp(fbamx)
                  return(1 - exp::float(fbamx));
            elif absfbamx > 10.0^(-DIGITS/2) then
                 // Boost DIGITS to boos relative
                 // precision of exp(fbamx) to
                 // avoid cancellation problems
                 DIGITS:= round(DIGITS*3/2);
                 return(1 - exp::float(fbamx));
            elif absfbamx > 10.0^(-DIGITS) then
                 // use 2 terms of the Taylor approximation
                  return(-fbamx - fbamx^2/2)
            else // absbfamx <= 10.0^(-DIGITS)
                 // The leading term of the Taylor approximation suffices
                  return(-fbamx)
            end_if;
       end_if;
    end_if;

    // Now, the symbolic part starts.
    // Do no use floating point conversion, but
    // work with the original values aa, bb, x

    // First, avoid the costly 'is' call, if possible:

    case domtype(amx)
    of DOM_INT do
    of DOM_RAT do
        if amx >= 0 then
             return(0);
        else return(1 - exp(bb*(amx)));
        end_if;
    of DOM_FLOAT do
        if amx >= 0 then
             return(float(0));
        else return(float(1 - exp(bb*(amx))));
        end_if;
    end_case;

    // ---- can side condition x >= a be decided? ----
    // ---- return an explicit result if possible ----

    // do test x >= a as well as x <= a, not just x > a, x <= a !
    if is(amx >= 0) = TRUE then
       if domtype(x) = DOM_FLOAT then
            return(float(0));
       else return(0);
       end_if;
    end_if;

    if is(amx <= 0) = TRUE then
       if domtype(x) = DOM_FLOAT then
            return(float(1 - exp(bb*amx)));
       else return(1 - exp(bb*amx));
       end_if;
    end_if;

    // Here, x - a >= 0 could not be decided.
    return(hold(stats::exponentialCDF)(aa, bb)(x));
  end_proc:
end_proc:
