/* --------------------------------------------------
Call:        stats::geometricCDF(p)

Parameters:  p  - an arithmetical expression representing
                  real numbers 0 <= p <= 1

Returns:  a function

Details:
  - The geometric distribution describes the number of
    Bernoulli trials with success probability p
    up to and including the first success
  - f:= stats::geometricCDF(p) can be called in the
    form f(x) with an arithmetical expression x representing
    a real number.

     x symbolic             --> symbolic call
     x float and p symbolic --> symbolic expression
     x float and p numeric  --> float
-------------------------------------------------- */

stats::geometricCDF:=proc(p)
local fp;
option escape;
begin
  if args(0) <> 1 then
     error("expecting one arguments")
  end_if:

  // check the parameters at the time when the
  // function is generated

  // ------------- check p -------------
  fp:= float(p):
  if domtype(fp) = DOM_FLOAT then
     if fp < 0 or fp > 1 then
        error("the 'probability parameter' p must satisfy 0 <= p <= 1"):
     end_if;
  end_if;
  if domtype(fp) = DOM_COMPLEX then
     error("the 'probability parameter' must be real");
  end_if;

  //-------------------------------
  // return the following procedure
  //-------------------------------

  proc(x)
  local pp, fp, q, fx, flx, k;
  begin

    // -------------------------------------------------
    // Return
    // P(X <= x) = sum( p*q^(k - 1), k = 1..floor(x))
    //           =  1 - q^floor(x)
    // where q = 1 - p and x >= 1
    // -------------------------------------------------

    if args(0) <> 1 then
       error("expecting one argument")
    end_if:

    // ---------------------------------------------
    // DO NOT MESS UP THE ORDERING OF THE FOLLWING 
    // CHECKS! YOU WILL CHANGE THE FUNCTIONALITY!
    // ---------------------------------------------

    // double check the parameter p. It might have
    // changed since this procedure was created:

    // ------------- check p -------------
    pp:= context(p):
    fp:= float(pp):
    q:= 1 - pp; 
    if domtype(fp) = DOM_FLOAT then
       if fp < 0 or fp > 1 then
          error("the 'probability parameter' p must satisfy 0 <= p <= 1"):
       end_if;
    end_if;
    if domtype(fp) = DOM_COMPLEX then
       error("the 'probability parameter' must be real");
    end_if;

    // -------------- check x ---------------
    if x = -infinity then return(0); end_if;
    if x =  infinity then return(1); end_if;
    fx:= float(x):
    flx:= floor(x);
    if domtype(fx) = DOM_COMPLEX then
       error("expecting a real argument"):
    end_if;

    // ---------------------------------------------
    // special cases: p = 0 and p = 1
    // Note that 0^floor(x) will not simplify to 0
    // if x is symbolic ;-(
    // ---------------------------------------------
    if iszero(pp) then
       if domtype(x) = DOM_FLOAT then
            return(float(0));
       else return(0);
       end_if;
    end_if;

    if iszero(q) then
       if (domtype(flx) = DOM_INT and flx < 1) or (is(x < 1) = TRUE) then
          if domtype(x) = DOM_FLOAT then
              return(float(0));
          else return(0);
          end_if;
       end_if;
       if (domtype(flx) = DOM_INT and flx >= 1) or (is(x >= 1) = TRUE) then
          if domtype(x) = DOM_FLOAT then
              return(float(1));
          else return(1);
          end_if;
       end_if;
       return(hold(stats::geometricCDF)(pp)(x));
    end_if;

    // ---------------------------------------------
    // the symbolic part using properties
    // ---------------------------------------------
    if domtype(fx) <> DOM_FLOAT then
       // x is symbolic. Nothing can be done
       // if we do not know whether x < 1
       if is(x < 1) = TRUE then
          return(0);
       end_if;
       if is(x >= 1) = TRUE then
          return(1 - q^flx):
       end_if;
       // we cannot decide whether x < 0. Return symbolically:
       return(hold(stats::geometricCDF)(pp)(x));
    end_if;

    // --------------------------------
    // Now we know that x is numerical
    // --  x can be converted to a float, i.e., 
    //     trunc(x) produces an integer
    // --  p can be symbolic or numeric
    // --------------------------------

    if flx < 1 then
       if domtype(x) = DOM_FLOAT then
            return(float(0))
       else return(0)
       end_if;
    end_if;

    // -------------------------------------------
    // Now we know that x is numerical and x >= 1
    // ------------------------------------------

    // produce a floating point result? 
    // If so, convert pp and q to floats for speed

    if domtype(x) = DOM_FLOAT then
         q:= float(q):
         pp:= fp;
         // pp and q are floats, so the following
         // should be fast. Still, we need to 
         // stabilize the result numerically:
         // For small flx, do the summation,
         // because it's numerically stable: 
         if flx <= 10 then
              return(float(pp*_plus(q^k $ k = 0..flx - 1)))
         else // here is danger of numerical underflow
              // if q << 1. If q^flx < 10^(-DIGITS) then
              // there is no need to compute q^flx.
              // I.e., check flx*ln(q) < -DIGITS*ln(10),
              // ln(10) =  2.3. Replace 2.3 by 3.0:
              if domtype(q) = DOM_FLOAT and
                 flx*ln::float(q) < -3.0*DIGITS then
                   return(float(1));
              else // q may be symbolic or
                   // numeric and sufficiently large
                   return(float(1 - q^flx));
              end_if;
         end_if;
    else // x is symbolic: return a symbolic expression
         return(1 - q^flx);
    end_if;
  end_proc:
end_proc:
