/* ----------- -------------------------------------------------------------

stats::betaRandom -- produce a generator of beta distributed random numbers

Call(s):
    stats::betaRandom(a, b, <Seed = n>)

Parameters:
   a, b -- the shape parameters: arithmetical expressions
           representing positive real numbers
      n -- the seed: an integer

Options:
     With Seed = n, the generator returned by stats::betaRandom
     is initialized with the seed n. With this option, the
     parameters a, b must both be convertible to positive
     real floating point numbers.

     Two generators created with the same seed produce the
     same sequence of numbers.

!!! There is also the undocumented syntax
    stats::betaRandom(a, b, frandom)
    stats::betaRandom(a, b, frandom(Seed = n))
    This is needed by binomialRandom(n, p, Seed = n) which
    has to call betaRandom with an initialized UC(0, 1) generator
!!!

Returns: a procdure of type DOM_PROC

Details:
 - stats::betaRandom produces a generator of beta deviates
   with shape parameter a, b.
 - After f:= stats::betaRandom(a,b), the call f() returns a random
   floating point number.
 - If a or b cannot be converted to positive real floats,
   then betaRandom(a, b) returns a symbolic call.
 - If a or b cannot be converted to positive real floats
   then betaRandom(a, b, Seed = n) raises an error.
 - The generators produced by betaRandom do not react
   to the global variable SEED!
 - For efficiency, it is recommended to create random sequences
   via
   >> f:= stats::XXXRandom(a, b)
   >> f() $ i = 1..10
   rather than via
   >>  stats::XXXRandom(a, b)() $ i = 1..10
   Also note that
   >>  stats::XXXRandom(a, b, Seed = n)() $ i = 1..10
   does not produce a random sequence, because
   in each call a freshly initialized generator
   would be created, all of them producing the same number

Examples:
 >> f:= stats::betaRandom(1, 2)
 >> f(), f()
                   0.441896254, 0.2074173878
-----------------------------------------------------------------*/

stats::betaRandom:= proc(a,b)
local fa, fb, r, y1, y2;
option escape;
begin
  if args(0)< 2 then
     error("expecting at least two arguments")
  end_if:
  if args(0)> 3 then
     error("expecting no more than three arguments")
  end_if:

  // ------------- check a -------------
  fa:= float(a):
  if domtype(fa) = DOM_FLOAT and fa <= 0 then
     error("the first shape parameter must be positive"):
  end_if;

  // ------------- check b -------------
  fb:= float(b):
  if domtype(fb) = DOM_FLOAT and fb <= 0 then
     error("the second shape parameter must be positive"):
  end_if;

  // -----------  check option Seed = s ---------------
  if args(0)=3 then
       if {domtype(args(3))} minus {DOM_PROC, DOM_FUNC_ENV} = {} then
            // a UC(0,1) generator frandom (DOM_FUNC_ENV) 
            // or frandom(seed) (DOM_PROC) was passed as 
            // third argument:
            r:= args(3):
       elif type(args(3))<>"_equal" then
            error("the 3rd argument must be of the form 'Seed = integer' or 'Seed = CurrentTime'"):
       elif op(args(3),1)<>Seed then
            error("the 3rd argument must be of the form 'Seed = integer' or 'Seed = CurrentTime'"):
       elif domtype(op(args(3),2))<>DOM_INT and op(args(3),2)<>CurrentTime then
            error("the 3rd argument must be of the form 'Seed = integer' or 'Seed = CurrentTime'"):
       else r:= frandom(op(args(3), 2));
       end_if;
  else r:= frandom:
  end_if:

  // ------ unevaluated return ? --------
  if domtype(fa)<>DOM_FLOAT or
     domtype(fb)<>DOM_FLOAT
  then
    if args(0) = 2 then
         return(procname(args()));
    else // do not accept symbolic a and/or b in conjunction
         // with Seed = n, because otherwise the following would
         // happen:
         // delete a, b: f:= stats::XXXRandom(a, b, Seed = 1):
         // a:= 1: b:= 2: f(), f(), f()
         //     -1.506518279, -1.506518279, -1.506518279
         error("the shape parameters must be numerical ".
               "if 'Seed = ...' is specified"):
    end_if;
  end_if:

  //------------------------------------
  y1:= stats::gammaRandom(a, 1, r):
  y2:= stats::gammaRandom(b, 1, r):

  //-------------------------------
  // return the following procedure
  // See: Knuth, Seminumerical Algorithms, Vol. 2
  //-------------------------------
  proc()
  local x1, x2;
  begin
    x1:=y1():
    x2:=y2():
    if iszero(x1 + x2) then
       return(fa/(fa + fb));
    end_if:
    return(x1/(x1 + x2));
  end_proc:
end_proc:
