/* -------------------------------------------
Calls:
   stats::empiricalRandom(x1, x2, ..., <Seed = s>)
   stats::empiricalRandom([x1, x2, ...], <Seed = s>)
   stats::empiricalRandom(s, <c>, <Seed = s>)
   stats::empiricalRandom(s, <[c]>, <Seed = s>)

Parameters: x1, x2, .. : the statistical data: numerical values,
                         must be convertible to floats (i.e., no
                         symbols allowed)
            s: a stats::sample object
            c: a column index of s

Example: 
     r:= stats::empiricalRandom(0, 2, PI, 1, 3, Seed = 1234):
     r() $ k = 1..12
               3, 1, 0, 3, 2, PI, PI, 0, PI, PI, 1, 1

     r:= stats::empiricalRandom([0, 2, PI, 1, 3], Seed = 1234):
     r:= stats::empiricalRandom(stats::sample([0, 2, PI, 1, 3]), Seed = 1234):
     r:= stats::empiricalRandom(stats::sample([0, 2, PI, 1, 3]), 1, Seed = 1234):
     r:= stats::empiricalRandom(stats::sample([0, 2, PI, 1, 3]), [1], Seed = 1234):
------------------------------------------- */
stats::empiricalRandom:=proc(data)
local seed, dummy, quantile, r;
option escape;
begin
  if args(0) < 1 then
     error("expecting at least one argument")
  end_if:
  // -----------  separate and check option Seed = s ---------------
  [seed, data, dummy]:= split([args()], has, Seed):
  if seed <> [] then
    seed:= seed[1];
    if type(seed) <> "_equal" then
       error("the Seed argument must be of the form 'Seed = integer' or 'Seed = CurrentTime'"):
    end_if:
    if domtype(op(seed,2)) <> DOM_INT and op(seed,2)<>CurrentTime then
       error("the Seed argument must be of the form 'Seed = integer' or 'Seed = CurrentTime'"):
    end_if:
    seed:= op(seed, 2);
  end_if:
  //------------------------------------------------------------
  // stats::getdata accepts "all_data" and "numeric_only".
  // Use "all_data", because with "numeric_only" exact numerical
  // expressions such as sqrt(2), PI etc. would be converted to
  // floats. However, we do need the original input data.
  //------------------------------------------------------------
  data:= stats::getdata(testargs(), "all_data", 1, op(data)):
  if domtype(data) = DOM_STRING then
       error(data)
  end_if:
  if testargs() then
     if data = [] then
        error("empty sample"):
     end_if:
     if nops(select(map(data, float), testtype, DOM_FLOAT)) <> nops(data) then
        error("some data could not be converted to floats")
     end_if:
  end_if:

  //produce an uc(0,1) random generator: 
  if seed <> [] then
    r:=frandom(seed):
  else r:=frandom:
  end_if:

  quantile:= stats::empiricalQuantile(data);
  //-------------------------------
  // return the following procedure
  //-------------------------------
  return(proc() begin quantile(r()) end_proc);
end_proc:
