/* -------------------------------------------
Info: Random generator for picking random 
      elements from a finite data sample

Calls:
   stats::finiteRandom([x1, x2, ..], [p1, p2, ..], <Seed = s>)
   stats::finiteRandom([[x1, p1], [x2, p2], ..], <Seed = s>)
   stats::finiteRandom(s, <c>, <Seed = s>)
   stats::finiteRandom(s, <[c]>, <Seed = s>)

Parameters: 
       x1, x2, .. -- arbitrary MuPAD objects. 
       p1, p2, .. -- probability values: symbols or positive
                     numerical values. In case of numerical values,
                     they  must add up to 1
       s          -- a sample of type stats::sample
       c1, c2     -- a column index: a positive integer.
                     Column c1 of s provides the data x1, x2, ..
                     Column c2 of s provides the data p1, p2, ..
                     There is no need to pass a column index,
                     if s has only two columns

Example: 
     r:= stats::finiteRandom([0, x, 3, PI], [0.1, 0.2, 0.4, 0.3], Seed = 1234):
     r() $ k = 1..12

                 PI, 3, x, 3, 3, PI, PI, 0, PI, PI, x, x

     r:= stats::finiteRandom([[0, 0.1], [x,0.2], [3,0.4], [PI, 0.3]], Seed = 1234):
     r() $ k = 1..12
                 PI, 3, x, 3, 3, PI, PI, 0, PI, PI, x, x

------------------------------------------- */
stats::finiteRandom:=proc(data)
local seed, dummy, p, quantile, r;
option escape;
begin
  if args(0) < 1 then
     error("expecting at least one argument")
  end_if:
  // -----------  separate and check option Seed = s ---------------
  if has([args()], Seed) then
    [seed, data, dummy]:= split(args(), has, Seed):
  else
    seed:= null();   
    data:= args();
  end_if;
  if seed <> null() then
    if type(seed) <> "_equal" then
       error("the Seed argument must be of the form 'Seed = integer' or 'Seed = CurrentTime'"):
    end_if:
    if domtype(op(seed,2)) <> DOM_INT and op(seed,2) <> CurrentTime then
       error("the Seed argument must be of the form 'Seed = integer' or 'Seed = CurrentTime'"):
    end_if:
    seed:= op(seed, 2);
  end_if:
  //------------------------------------------------------------
  data:= stats::getdata(testargs(), "anything", 2, data):
  if domtype(data) = DOM_STRING then
       error(data)
  end_if:

  if testargs() then
     if has([data], []) then
        error("empty sample"):
     end_if:
  end_if:

  //----------------------------------------------------
  // Now, data = [x1, x2, ..], [p1, p2, ..];
  //----------------------------------------------------

  p:= [data][2]; // = [p1, p2, ..]
  assert(domtype(p) = DOM_LIST);

  // Call eliminateDuplicates as in CDF/PF/Quantile? 
  // There is no need to eliminate duplicates in the
  // random generator!

  if map({op(p)}, domtype@float) <> {DOM_FLOAT} then
       error("there are symbolic probabilities");
  elif specfunc::abs(float(_plus(op(p)) - 1)) > 10^(2-DIGITS) then
       error("the probabilities do not add up to 1");
  end_if;

  quantile:= stats::finiteQuantile(data);

  //produce an uc(0,1) random generator: 
  if seed <> null() then
       r:=frandom(seed):
  else r:=frandom:
  end_if:

  //-------------------------------
  // return the following procedure
  //-------------------------------
  return(proc() begin quantile(r()) end_proc);
end_proc:
