/* ---------------------------------------------
Call:        stats::geometricRandom(p)

Parameters:  p  - an arithmetical expression representing
                  real numbers 0 <= p <= 1

Returns:  a function (the random generator)
------------------------------------------------*/

stats::geometricRandom:= proc(p)
local fp, q, fq, r;
option escape;
begin
  if args(0) < 1 then
     error("expecting at least one argument")
  end_if:
  if args(0) > 2 then
     error("expecting no more than two arguments")
  end_if:

  // check the parameter at the time when the
  // function is generated. For XXXRandom, there
  // is no need to double check later, because
  // no procedure is returned if p is not numerical

  // ------------- check p -------------
  fp:= float(p):
  if domtype(fp) = DOM_FLOAT then
     if fp < 0 or fp > 1 then
        error("the 'probability parameter' p must satisfy 0 <= p <= 1"):
     end_if;
  end_if;
  if domtype(fp) = DOM_COMPLEX then
     error("the 'probability parameter' must be real");
  end_if;

  // -----------  check option Seed = s ---------------
  if args(0)=2 then
    if type(args(2))<>"_equal" then
       error("the 2nd argument must be of the form 'Seed = integer' or 'Seed = CurrentTime'"):
    end_if:
    if op(args(2),1)<>Seed then
       error("the 2nd argument must be of the form 'Seed = integer' or 'Seed = CurrentTime'"):
    end_if:
    if domtype(op(args(2),2))<>DOM_INT and op(args(2),2)<>CurrentTime then
       error("the 2nd argument must be of the form 'Seed = integer' or 'Seed = CurrentTime'"):
    end_if:
  end_if:

  // ------ unevaluated return ? --------
  if domtype(fp) <> DOM_FLOAT
  then
    if args(0) = 1 then
         return(procname(args()));
    else // do not accept symbolic p in conjunction
         // with Seed = s, because otherwise the following would
         // happen:
         // delete a, b: f:= stats::XXXRandom(a, b, Seed = 1):
         // a:= 1: b:= 2: f(), f(), f()
         //     -1.506518279, -1.506518279, -1.506518279
         error("the 'probability parameter' must be numerical ".
               "if 'Seed = ...' is specified"):
    end_if;
  end_if:

  // ------ special case p = 0 --------
  if iszero(p) then
     return(() -> infinity);
  end_if;

  // ------ special case p = 1 --------
  q:= 1 - p:
  if iszero(q) then
     return(() -> 1);
  end_if;
  fq:= float(q):

  //------------------------------------
  // produce an uc(0,1) random generator: 
  //------------------------------------
  if args(0)=2 then
       r:= frandom(op(args(2),2)):
  else r:= frandom:
  end_if:

  //-------------------------------
  // return the following procedure
  //-------------------------------
  proc()
  local x;
  begin
     // -------------------------------------
     // geometric(p) = ln(UC(0, 1))/ln(1 - p)
     // -------------------------------------
     x:= r():
     if not iszero(x) then
        return(ceil(ln::float(x)/ln(fq)));
     end_if;
     // if x = 0.0 then return the mean
     return(ceil(1/fp));
  end_proc:
end_proc:
