/*

stats::lognormalRandom -- produce a generator of log-normally
                          distributed random numbers

Call(s): 
    stats::lognormalRandom(m, v, <Seed = n>)

Parameters:
      m -- the mean: a real number
      v -- the variance: a positive real number
      n -- the seed: an integer

Options: 
     With Seed = n, the generator returned by stats::lognormalRandom
     is initialized with the seed n. With this option, the 
     parameters m, v must both be convertible to real floating
     point numbers.

     Two generators created with the same seed produce the
     same sequence of numbers.

Returns: a procedure of type DOM_PROC

This code simply forwards to stats::normalRandom.

-----------------------------------------------------------------*/

stats::lognormalRandom:= 
proc(m, v)
  local fm, fv, r;
  option escape;
begin
  if args(0) < 2 then
    error("expecting at least two arguments")
  end_if:
  if args(0) > 3 then 
    error("expecting no more than three arguments")
  end_if:

  // ------------- check m -------------
  fm:= float(m):
  if domtype(fm) = DOM_COMPLEX then
    error("the mean must be real");
  end_if;

  // ------------- check v -------------
  fv:= float(v):
  if domtype(fv) = DOM_FLOAT and fv <= 0 then
    error("the variance must be positive"):
  end_if;
  if domtype(fv) = DOM_COMPLEX then
    error("the variance must be real");
  end_if;

  // -----------  check option Seed = s ---------------
  if args(0)=3 then
    if type(args(3))<>"_equal" then
      error("the 3rd argument must be of the form 'Seed = integer' or 'Seed = CurrentTime'"):
    end_if:
    if op(args(3),1)<>Seed then
      error("the 3rd argument must be of the form 'Seed = integer' or 'Seed = CurrentTime'"):
    end_if:
    if domtype(op(args(3),2))<>DOM_INT and op(args(3),2)<>CurrentTime then
      error("the 3rd argument must be of the form 'Seed = integer' or 'Seed = CurrentTime'"):
    end_if:
  end_if:

  // ------ unevaluated return ? --------
  if domtype(fm) <> DOM_FLOAT or
     domtype(fv) <> DOM_FLOAT
  then
    if args(0) = 2 then
      return(procname(args()));
    else // do not accept symbolic m and/or v in conjunction
         // with Seed = n, because otherwise the following would
         // happen:
         // delete m, v: f:= stats::lognormalRandom(m, v, Seed = 1):
         // m:= 0: v:= 1: f(), f(), f()
         //     -1.506518279, -1.506518279, -1.506518279
         error("mean and variance must be numerical ".
               "if 'Seed = ...' is specified"):
    end_if;
  end_if:

  r:= stats::normalRandom(args()):
  return(() -> exp(r(args())));
end_proc:
