//this procedure computes poisson deviates 
// with parameter mu >= 0;
//for given a,  f:=poissonRandom(a) calls the generator; f() returns the
//random numbers;

/* ---------------------------------------------
Walter Oevel, 13.7.01
  - ueberarbeitet
------------------------------------------------*/

stats::poissonRandom:= proc(m)
local fm, y;
option escape;
begin
  if args(0)< 1 then
     error("expecting at least one argument")
  end_if:
  if args(0)> 2 then
     error("expecting no more than two arguments")
  end_if:

  // ------------- check m -----------------
  fm:= float(m):
  if domtype(fm) = DOM_FLOAT and fm < 0 then
     error("the mean must be >= 0"):
  end_if;
  if domtype(fm) = DOM_COMPLEX then
       error("the mean must be real"):
  end_if;

  // -----------  check option Seed = s ---------------
  if args(0)=2 then
    if type(args(2))<>"_equal" then
       error("the 2nd argument must be of the form 'Seed = integer' or 'Seed = CurrentTime'"):
    end_if:
    if op(args(2),1)<>Seed then
       error("the 2nd argument must be of the form 'Seed = integer' or 'Seed = CurrentTime'"):
    end_if:
    if domtype(op(args(2),2))<>DOM_INT and op(args(2),2)<>CurrentTime then
       error("the 2nd argument must be of the form 'Seed = integer' or 'Seed = CurrentTime'"):
    end_if:
  end_if:

  // ------ unevaluated return ? --------
  if domtype(fm) <> DOM_FLOAT
  then
    if args(0) = 1 then
         return(procname(args()));
    else // do not accept symbolic n and/or p in conjunction
         // with Seed = s, because otherwise the following would
         // happen:
         // delete a, b: f:= stats::XXXRandom(a, b, Seed = 1):
         // a:= 1: b:= 2: f(), f(), f()
         //     -1.506518279, -1.506518279, -1.506518279
         error("the mean must be numerical ".
               "if 'Seed = ...' is specified"):
    end_if;
  end_if:

  // ------ special case m = 0 --------
  if iszero(m) then
     return(() -> 0);
  end_if;

  //------------------------------------
  // produce an uc(0,1) random generator: 

  if args(0)=2 then
     y:=frandom(op(args(2),2)):
  else
     y:=frandom:
  end_if:

  //-------------------------------
  // return the following procedure
  //-------------------------------
  proc()
  local r;
  begin

    // r = random generator for Poisson(m) variates.
    // Recursively calls Poisson(m') (and others)
    // where m' is approximately 7/8*m

    // See D. Knuth, Seminumerical Algorithms, p 137

    r:= proc(fm)
    local s, k, tmp, kmax, sy, X;
    name stats::poissonRandom::recursive;
    begin

      // for small m, use a direct quantile
      // computation using an UC(0, 1) number y(): 
      if fm < 20 then
        sy := y()*exp::float(fm):
        s := float(1);
        if s >= sy then
           return(0);
        end_if;
        tmp:= float(1);
        kmax:= 2*DIGITS*max(1/fm,fm);
        for k from 1 to kmax do
            tmp:= tmp* fm/k;
            s := s + tmp;
            if s >= sy then
               return(k);
            end_if;
        end_for;
        // return the mean if we arrive here
        return(m);
      end_if;

      // For large m, reduce m recursively.
      // The resulting runtime is O(log(m))
      // rather than the O(m) of the direct
      // quantile method above.
      // In the following, the obscure factor 
      // 7/8 is recommend by Ahrens and Dieter.
      // See Knuth for further information.
      tmp:= trunc(7/8*fm):
      X:= stats::gammaRandom(tmp, 1, y)();
      if X < fm then
           return(tmp + r(fm - X)())
      else return(stats::binomialRandom(tmp - 1, fm/X, y)())
      end_if;
    end_proc:

    // call the recursive poissonRandom

    r(fm);

  end_proc:
end_proc:
