//--------------------------------------------------------------
// stats::correlationMatrix -- convert a covariance matrix
//                             to a correlation matrix
//
// Calls:  stats::correlationMatrix(covarianceMatrix)
//
// Parameters:
//    covarianceMatrix - a positive definite symmetric square
//                       matrix of category Cat::Matrix or an
//                       array
// Retuns:  a symmetric square matrix of the same dimension 
//          and type as the input matrix
//          FAIL is returned if at least one of the diagonal
//          elements of the covariance matrix is zero. 
//
// Details:
//   Given a positive definite matrix covMat, the corresponding
//   correlation matrix corMat is given by
//        corMat[i, j] = covMat[i,j]/sqrt(covMat[i,i])/sqrt(covMat[j,j]);
//   If covMat is positive definite, the correponding correlation
//   matrix is well defined (all diagonal elements of covMat are > 0).
//   Its entries along the diagonal are 1 (1.0, repectively) and
//   all entries lie between -1 and 1. 
//
//   This routine is a useful utility for converting the
//   covariance matrix returned by stats::linReg to the
//   corresponding correlation matrix
//------------------------------------------------------------

stats::correlationMatrix:= proc(covarianceMatrix)
local T, m, n, s, i, j;
begin
  if args(0) < 1 then
    error("expecting at least one argument")
  end_if;
  //--------------------------------------------------------
  if covarianceMatrix::dom::hasProp(Cat::Matrix) = TRUE then
       T:= covarianceMatrix::dom:
       // convert the matrix to a an array for speed
       if T::constructor = Dom::/*Sparse*/Matrix then
            // special implementation for sparse matrices
            [m, n]:= covarianceMatrix::dom::matdim(covarianceMatrix):
            if m <> n then
               error("expecting a square matrix");
            end_if:
            s:= [covarianceMatrix[i, i] $ i=1..n]:
            if contains(s, 0) > 0 or contains(s, float(0)) > 0 then
               // the covarianceMatrix is not positive definite:
               // some diagonal elements are zero.
               return(FAIL);
            end_if;
            s:= map(s, _invert@sqrt):
            s:= covarianceMatrix::dom(m, n, s, Diagonal):
            // multiply with this diagonal matrix 
            // from the left and from the right
            return(s*covarianceMatrix*s);
       else covarianceMatrix:= expr(covarianceMatrix):
       end_if:
  else T:= DOM_ARRAY:
  end_if;
  //--------------------------------------------------------
  if domtype(covarianceMatrix) <> DOM_ARRAY then
     error("expecting an array or a matrix of category 'Cat::Matrix'");
  end_if;
  //--------------------------------------------------------
  [m, n]:= [ op(covarianceMatrix, [0, 2, 2]),
             op(covarianceMatrix, [0, 3, 2]) ]:
  if m <> n then
     error("expecting a square matrix");
  end_if:
  //--------------------------------------------------------
  s:= [covarianceMatrix[i, i] $ i=1..n]:
  if contains(s, 0) > 0 or contains(s, float(0)) > 0 then
     // the covarianceMatrix is not positive definite:
     // some diagonal elements are zero.
     return(FAIL);
  end_if;

  s:= map(s, sqrt):

  for i from 1 to n do
    for j from 1 to n do
       covarianceMatrix[i, j]:= covarianceMatrix[i, j] / s[i] / s[j];
    end_for:
  end_for:
  //--------------------------------------------------------------
  // return a matrix/an array of the same type as the input matrix
  if T = DOM_ARRAY then
       return(covarianceMatrix);
  else return(T::create(covarianceMatrix)):
  end_if:
end_proc:
