//
// stats::cutoff - discard outliers
//
// stats::cutoff(data, alpha, <index>)
//
//   data can be a list of values,
//               a list of lists,
//               or a stats::sample
//   0 <= alpha < 1/2
//   index must be given if data
//         is not a list of values
//         and determines the index
//         to analyze.
//
// returns a copy of data with elements removed

stats::cutoff := 
proc(data, alpha, index)
  local minval, maxval, quantiles, data2, a, types;
begin
  a := float(alpha);
  if domtype(a) <> DOM_FLOAT then
    return(procname(args()));
  end_if;

  if a <= 0 then
    return(data);
  end_if;
  if a >= 1/2 then
    if domtype(data) = stats::sample then
      return(stats::sample());
    else
      return([]):
    end_if:
  end_if;

  if not testtype(data, DOM_LIST) and
     not testtype(data, stats::sample) then
    return(procname(args()));
  end_if;

  if args(0) > 2 then
    data2 := map(coerce(data, DOM_LIST), op, index);
  else
    data2 := data;
  end_if;

  if not testtype(data2, DOM_LIST) then
    error("bad data specification");
  end_if;

  types := map({op(data2)}, domtype@float);
  if types <> {DOM_FLOAT} then
    if DOM_COMPLEX in types then
      error("complex values not allowed");
    end_if;
    return(procname(args()));
  end_if;

  quantiles := stats::empiricalQuantile(data2);
  minval := quantiles(alpha);
  maxval := quantiles(1-alpha);

  if args(0) > 2 then
    data := map(data,
                l -> if l[index] < minval or
                        l[index] > maxval then
                       null()
                     else
                       l
                     end_if)
  else
    data := map(data,
                v -> if v < minval or
                        v > maxval then null()
                     else v end_if)
  end_if;
end_proc:

