/*---------------------------------------------------------------
This is a utility function for stats::csGOFT.
It computes a 'cell partitioning' 
[[b0, b1], [b1, b2], ..., [b.(k-1), b.k]]
where b.i = quantile(i/k). 

Call:    stats::equiprobableCells(k, quantile <, NoWarning>)

Parameters:
   k        -- the number of cells: a positive integer
   quantile -- a procedure representing the quantile function
               of a statistical distribution.
               Typically, a quantile function provided by
               the stats-library such as stats::normalQuantile(m, v)
               (with numerical values m, v).
               Alternatively, a user-defined function.
               It will be called in the form quantile(i/k)
               with i = 0..k and should produce a numerical
               real value or +/- infinity.
Option:
   NoWarning -- suppress warning messages

Return: a list of k cells, given by
             [[b0, b1], [b1, b2], ..., [b.(k-1), b.k]]
        where b.i = float(quantile(float(i/k))), i = 0..k. 
       

Details:
  --  This function serves for creating equiprobable
      cells that may be passed to the chi square test
      implemented in stats::csGOFT. Cf. Example 1.
  --  Warnings are issued if b.i are not 
      real or +/- infinity
  --  Warnings are issued if not b0 <= b1 <= .. <= b.k,
      i.e., if the procedure 'quantile' does not behave
      like a quantile function should do.
  --  Warnings are issued if b.(i - 1) = b.i (the i-th
      cell [b.(i-1), b.i] is zero. This may happen
      for discrete distributions and a requested 
      number of cells that exceeds the number of
      possible discrete values the random variable
      can attain. Cf. Example 3.
  --  If the quantile function corresponds to a
      discrete distribution, the returned cells
      are not equiprobable (in general, an equiprobable
      partitioning does not exist in the discrete case)

-----------------------------------------------------------------
Example 1:
----------
2.1.0 > cells:= stats::equiprobableCells(4, stats::normalQuantile(0, 1))

[[-infinity, -0.6744897502], [-0.6744897502, 0.0], [0.0, 0.6744897502],

   [0.6744897502, infinity]]

// The cells are equiprobable w.r.t. normalCDF(0, 1)
2.1.0 > cdf:= stats::normalCDF(0,1):
2.1.0 > map(cells, cell -> cdf(cell[2]) - cdf(cell[1]))

                            [0.25, 0.25, 0.25, 0.25]
`
// The cell partitioning 'cells' can be passed to
// the chi-square goodness-of-fit test stats::csGOFT:
2.1.0 > r:= stats::normalRandom(0, 1):
2.1.0 > data:= [r() $ i = 1..100]:
2.1.0 > stats::csGOFT(data, cells, CDF = cdf)

                       [1.839999999, 0.3937305889, 25.0]

-----------------------------------------------------------------
Example 2:
----------
2.1.0 > r:= frandom(0): 
2.1.0 > data:= [r() $ k = 1..100]:
2.1.0 > quantile:= stats::empiricalQuantile(data):
2.1.0 > cells:= stats::equiprobableCells(4, quantile)

       [[0.004636546443, 0.2407624965], [0.2407624965, 0.4666761034],

          [0.4666761034, 0.7269049446], [0.7269049446, 0.9717924569]]

// Since the empirical distribution is discrete, the cells are not truly
// equiprobable:
2.1.0 > cdf:= stats::empiricalQuantile(data):
2.1.0 > map(cells, cell -> cdf(cell[2]) - cdf(cell[1]))

            [0.2361259501, 0.2026188153, 0.229066485, 0.2895955861]

-----------------------------------------------------------------
Example 3:
----------
2.1.0>  quantile:= stats::binomialQuantile(4, 1/2):
2.1.0 > cells:= stats::equiprobableCells(5, quantile)

          [[0.0, 1.0], [1.0, 2.0], [2.0, 2.0], [2.0, 3.0], [3.0, 4.0]]

// There is one empty cell [2.0, 2.0] with zero probability:
2.1.0 > cdf:= stats::binomialCDF(4, 1/2):
2.1.0 > map(cells, cell -> cdf(cell[2]) - cdf(cell[1]))

                        [0.25, 0.375, 0.0, 0.25, 0.0625]

-----------------------------------------------*/

stats::equiprobableCells :=proc(k, quantile)
local nowarning, cells, i, real;
begin
  if args(0) < 2  then
     error("expecting at least two arguments")
  end_if:
  if args(0) > 3  then
     error("expecting no more than three arguments")
  end_if:
  if args(0) = 3 then
     if args(3) = NoWarning then
        nowarning:= TRUE;
     else error("unknown option ".expr2text(args(3)).", expecting 'NoWarning'");
     end_if;
  else nowarning:= FALSE;
  end_if;

  if domtype(k) <> DOM_INT or k < 1 then
     error("the first argument (the number of cells) must be a positive integer"):
  end_if;

  case domtype(quantile)
  of DOM_PROC do 
  of DOM_FUNC_ENV do 
     break;
  otherwise
     error("the second argument (the quantile function) must be a procedure");
  end_case;

  cells:= [float(quantile(float(i/k))) $ i = 0..k]: 

  if nowarning then
       // convert to a list of cells [[b0, b1], [b1, b2], ...]];
       cells:= [[cells[i], cells[i + 1]] $ i = 1.. k];
       return(cells);
  end_if;

  // Without the option NoWarning, check that the partitioning
  // is suitable for stats::csGOFT. If not, issue a warning:

  if {op(map(cells, domtype@float))} minus {DOM_FLOAT, stdlib::Infinity} <> {} then
     warning("the quantile function must produce " .
             "numerical real values or +/- infinity");
       real:= FALSE;
  else real:= TRUE;
  end_if;


  for i from 1 to k do
      if cells[i] = cells[i+1] then
         warning("the ".output::ordinal(i)." cell has zero length. Maybe a ".
                 "discrete distribution was specified and the requested " .
                 "number of cells is too large.");
         break;
      end_if;
      if real and cells[i+1] < cells[i] then
         warning("something's wrong with the quantile function: it does " .
         "not seem to be a monotonically increasing function");
         break;
      end_if;
  end_for;

  // convert to a list of cells [[b0, b1], [b1, b2], ...]];
  cells:= [[cells[i], cells[i + 1]] $ i = 1.. k];
  cells:= subs(cells, [RD_INF = infinity, RD_NINF = -infinity]);

  return(cells);

end_proc:
