/*-----------------------------------------------------------------------------
  The Hodrick Prescott Filter (HP-Filter) is the most popular method to
  separate a time series into a trend component plus a cyclical component.

  The computation in the following routine makes use of the penta-diagonal 
  structure of the coefficient-matrix. 
 
Parameters:
  data  : time series (as list of numerical values)
  lambda: smoothing parameter; 
          recommended values are:
           monthly data:  lambda = 100000 .. 140000
           quaterly data: lambda = 1600
           annual data:   lambda = 6 .. 14

Return Value: the trend component of the time series (as a list)
              construct the cyclical component as

               trend:= stats::hodrickPrescottFilter(x, lambda):
               cyclic:= [x[i] - trend[i] $ i = 1..nops(x)]:

References:

   Robert Hodrick and Edward C. Prescott: 
       "Postwar U.S. Business Cycles.  An empirical investigation", 
        Journal of Money, Credit and Banking, 1997

   Agustin Maraval and Ana del Rio:
       "Time Aggregation and the Hodrick-Prescott-Filter",
       Banco de Espana, 2001.

Example:
    monthly:= i -> i/1000                  // the trend
          +     cos(i * 2*float(PI)/120.0) // the trend, period = 10 years
          + 0.7*cos(i * 1.12*2*float(PI))  // approx monthly cycle
          + 0.3*sin(i * 2.04*4*float(PI))  // approx bimonthly cycle
          + 0.2*cos(i * 1.01*6*float(PI))  // approx trimonthly cycle
    :
    // Monthly data, given for 120 month = 10 years
    x1:= [i/100 + monthly(i) $ i = 1..120]:
    x2:= [x1[i] + frandom() - 0.5 $ i = 1..nops(x1) ]: // additional noise
    trend:= stats::hodrickPrescottFilter(x2, 10^5):
    // for MuPAD 3.0
    plot(
        (plot::Line2d([i, x1[i]], [i+1, x1[i+1]], Color = RGB::Black),
         plot::Line2d([i, x2[i]], [i+1, x2[i+1]], Color = RGB::Red),
         plot::Line2d([i, trend[i]], [i+1, trend[i+1]], Color = RGB::DarkRed),
         plot::Line2d([i, x1[i] - trend[i]], [i+1, x1[i+1]-trend[i+1]], Color = RGB::DarkBlue),
         plot::Line2d([i, x2[i] - trend[i]], [i+1, x2[i+1]-trend[i+1]], Color = RGB::DarkGreen)
        ) $ i = 1..nops(x1) -1);
    // for MuPAD > 3.2
    plot(
         plot::Listplot([[i, x1[i]] $ i = 1..nops(x1)], Color = RGB::Black),
         plot::Listplot([[i, x2[i]] $ i = 1..nops(x1)], Color = RGB::Red),
         plot::Listplot([[i, trend[i]] $ i = 1..nops(x1)], Color = RGB::DarkRed),
         plot::Listplot([[i, x1[i] - trend[i]] $ i = 1..nops(x1)], Color = RGB::DarkBlue),
         plot::Listplot([[i, x2[i] - trend[i]] $ i = 1..nops(x1)], Color = RGB::DarkGreen)
        );
             
-----------------------------------------------------------------------------*/
stats::hodrickPrescottFilter:= proc(x, lambda)
local n, a, b, c, data, solvePentaDiagonal;
begin
   // -----------------------------------------------------
   // utility function for solving a penta-diagonal linear equation 
   // A*x = y:
   //
   //   (a[1] b[1] c[1]   0    ...    0  ) (x[1])   (y[1])
   //   (b[1] a[2] b[2] c[2]    0    ... ) (x[2])   (y[2])
   //   (c[1] b[2] a[3] b[3]  c[3]   ... ) (x[3]) = (y[3])
   //   ( ..   ..   ..   ..    ..     .. ) ( .. )   ( .. )
   //   (  0   ..   .. c[n-2] b[n-1] a[n]) (x[n])   (y[n])
   //
   // Syntax:  solvePentaDiagonal(a, b, c, x)
   // Return Value: solution vector x as a list
   // -----------------------------------------------------
   solvePentaDiagonal := proc(a : DOM_LIST, 
                              b : DOM_LIST, 
                              c : DOM_LIST, 
                              y : DOM_LIST) : DOM_LIST
   local n, i,
         h1, h2, h3, h4, h5, 
         hh1, hh2, hh3, hh4, hh5, 
         tmp, tmpb, tmpc;
   begin
    n:= nops(y):
    if n < 2 then 
       if iszero(a[1]) then
          error("matrix not invertible");
       end_if;
       return([y[1]/a[1]]); 
    end_if;
    if nops(b) < n then
       b:= b.[0 $ n - nops(b)]; // now, nops(b) = n
    end_if;
    if nops(c) < n then
       c:= c.[0 $ n - nops(c)]; // now, nops(c) = n
    end_if;
    if nops(a) <> n or
       nops(b) <> n or
       nops(c) <> n then
       error("mismatch of data lengths");
    end_if;

    // Initialization
    [h1, h2, h3, h4, h5, 
     hh1, hh2, hh3, hh4, hh5, 
     tmp, tmpb, tmpc]:= [0 $ 13]:
    
    for i from 1 to n do
        tmp:= a[i] - h1*h4 - hh2*hh5;
        if iszero(tmp) then
           error("matrix not invertible"):
        end_if;
        tmpb:= b[i];
        hh1:= h1;
        h1:= (tmpb - h2*h4)/tmp;
        b[i]:= h1;
        tmpc:= c[i];
        hh2:= h2;
        h2:= tmpc/tmp;
        c[i]:= h2;
        a[i]:= (y[i] - hh3*hh5 - h3*h4)/tmp;
        hh3:= h3;
        h3:= a[i];
        h4:= tmpb - hh1*h5;
        hh5:= h5;
        h5:= tmpc;
    end_for:
    h2:= 0;
    h1:= a[n];
    y[n]:= h1;
    for i from n downto 1 do
        y[i]:= a[i] - b[i]*h1 - c[i]*h2;
        h2:= h1;
        h1:= y[i];
    end;
    return(y);
  end_proc:


  //--------------------------------------
  // main program of hodrickPrescottFilter
  //--------------------------------------
   data:= stats::getdata(testargs(), "all_data",
                         1, args(1..args(0) - 1)):

   if domtype(data) = DOM_STRING then
      // stats::getdata returns a string with an error
      // message if something went wrong
      error(data);
   end_if:

   // now, the data are a list
   data:= float(data):
   n:= nops(data):

   if n = 1 then
     return(data);
   end_if;

   lambda:= float(lambda):
   a:= [ 6*lambda + 1 $ n]:
   b:= [-4*lambda $ n-1, 0]:
   c:= [   lambda $ n-2, 0, 0]:

   a[1]:=      lambda+1;
   a[2]:=    5*lambda+1;
   a[n-1]:=  5*lambda+1;
   a[n]:=      lambda+1;
   b[1]:=   -2*lambda;
   b[n-1]:= -2*lambda;

   solvePentaDiagonal(a, b, c, data);
end_proc:
