/*--

stats::meanTest -- Hypothesis testing

Call: 

stats::meanTest(l, v, dist)

Parameter:

l	-- list
v 	-- value
dist	-- distribution (optional)

Synopsis:

stats::meanTest(l, v, dist) determine how good of a guess is 'v' for the
mean of values of the list 'l' using the distribution 'dist'. If no 
distribution is specified, the T distribution is used.

Examples:

Determine how good of a guess is 5 for the mean of x, where 10
experiments gave 1, -2, 3, -4, 5, -6, 7, -8, 9, 10.

>> l := [1, -2, 3, -4, 5, -6, 7, -8, 9, 10]:

>> float(stats::meanTest(l, 5));

		5.756660091e-2

>> float(stats::meanTest(l, 5, stats::normal));

		4.058346175e-2

See also:

stats::mean, stats::stdev, stats::normal, stats::Tdist

--*/

stats::meanTest := proc(l, v, dist)
   local sigma, x;
begin
   sigma:= stats::stdev(l,Sample);
   if iszero(sigma) then return(FAIL) end_if:
   x := (stats::mean(l)-v)/(sigma/sqrt(nops(l)));
   if args(0) = 3 then
      case dist 
      of stats::Tdist do return(stats::Tdist(x, nops(l)-1)); 
                         break;
      of stats::normal do return(stats::normal(x, 0, 1)); 
                          break;
      otherwise
         error("third argument must be stats::normal or stats::Tdist");
      end_case;
   else
      stats::Tdist(x, nops(l)-1)
   end_if
end_proc:
