//-----------------------------------------------------------------------
// combine::ln(ee): handles the call combine(ee, ln). Tries to rewrite
// k*ln(a)+l*ln(b) ==> ln(a^k*b^l) whenever possible.
//
// Descends recursively into expressions of type +, *, ^, but not
// into arguments of other function calls. In particular, arguments
// of exp are not combined.
// Properties are taken into account.
//
// Implemented rules:
//
// k*ln(a) --> ln(a^k) if
// - k in R and a > 0
// - -2 <= k <= 2 and Re(a) > 0
// - -1 < k <= 1 or (k = 1 and not a < 0)
// This is done first.
//
// k*ln(a) + k*ln(b) --> k*ln(a*b) if a > 0 or b > 0
// k*ln(a) - k*ln(b) --> k*ln(a/b) if a > 0 or b > 0 or a < b
// k*ln(a) - k*ln(b) --> -k*ln(b/a) if b < a

//
// Not implemented: the latter two rules if Re(a) > 0 and Re(b) > 0
//
// Note:
// - 2*ln(x) + ln(4) will NOT be combined, except if is(x < 0) = TRUE
// - products of logarithms, such as ln(2)*ln(3), are not combined
//

combine::ln:=
proc(ee, options)
  local e, l, r, kn, ks, selectFirst, s, i, j,
  k, split_product, arguments1, arguments2, reals1, reals2, positiveArguments;
begin

  // shortcut: do nothing if there are no logarithms
  if not has(ee, hold(ln)) then
    return(ee)
  end_if;

  // recursively combine all operands
  e := ee;
  if contains({DOM_EXPR, DOM_SET, DOM_LIST, DOM_ARRAY}, domtype(e)) then
    e := stdlib::mapEvalChanges(e, combine::ln, options)
  elif domtype(e) = DOM_POLY then
    e := mapcoeffs(e, combine::ln, options)
  end_if;

   // shortcut: finished if there are no logarithms left
  if not has(e, hold(ln)) then
    return(e)
  end_if;

  // handle k*ln(x) --> ln(x^k)
  if type(e) = "_mult" then

    // extract ln factors
    l := split([op(e)], testtype, "ln");

    if nops(l[1]) = 0 then
      return(e)
    end_if;

    // split real factors in the coefficient
    kn := split(l[2], testtype, Type::Real);
    if options[IgnoreAnalyticConstraints] then
      // treat all factors like reals
      ks:= _mult(op(kn[2]));
      kn := _mult(op(kn[1]));
      r:= 1
    else
      ks := split(kn[2], is, Type::Real);
      r := _mult(op(ks[2].ks[3]));
      ks := _mult(op(ks[1]));
      kn := _mult(op(kn[1]));
      if kn*ks = 1 then
        // no real factors in the coefficient; we can do nothing
        return(e)
      end_if;
    end_if;
    
    l := map(l[1], op);
    // now e = r * kn * ks * _mult(ln(l[i]) $ i = 1..nops(l))
    // kn is real numeric, ks is real symbolic.
    
    // auxilliary procedure: return the index of the first
    // element in the list l for which f returns TRUE
    // if none exists, return FAIL
    selectFirst :=
    proc(l, f)
      local i;
    begin
      if options[IgnoreAnalyticConstraints] then
        if nops(l) = 0 then
          FAIL
        else
          1
        end_if
      else
        for i from 1 to nops(l) do
          if f(l[i]) = TRUE then
            return(i)
          end_if
        end_for;
        FAIL
      end_if
    end_proc;

    // is there a logarithm with positive argument?
    // prevent expansions n^k for integers n, k with k >= Pref::autoExpansionLimit()
    if (domtype(kn) = DOM_INT or domtype(kn) = DOM_RAT) and
      specfunc::abs(kn) >= Pref::autoExpansionLimit() then
      s:= selectFirst(l, z -> bool(domtype(z) <> DOM_INT and
                                   domtype(z) <> DOM_RAT) and
                      is(z, Type::NonNegative))
    else
      s := selectFirst(l, z -> is(z, Type::NonNegative))
    end_if;
    if s <> FAIL then
      // pull the real factors as exponents into the first
      // logarithm with positive argument
      // but take care that this does not become too large
      if combine::isSmall(l[s], kn*ks)
        then
        l := subsop(l, s = l[s]^(kn*ks))
      else
        r:= r*kn*ks
      end_if;
      return(_mult(r, op(map(l, ln))))
    end_if;

    // try the rules for -1 <= k <= 1 and -2 <= k <= 2
    
    // first try k = kn*ks
    if ks <> 1 then
      if options[IgnoreAnalyticConstraints] then
        s:= 1
      elif is(kn*ks, Type::Interval([-1], [1])) = TRUE then
        if is(kn*ks > -1) = TRUE then
          s := 1
        else
          s := selectFirst(l, z -> not (is(z < 0)));
        end_if;
      elif is(kn*ks, Type::Interval([-2], [2])) = TRUE then
        s := selectFirst(l, z -> is(Re(z) > 0));
      end_if;
      if s <> FAIL then
        l := subsop(l, s = l[s]^(kn*ks));
        return(_mult(r, op(map(l, ln))))
      end_if;
    end_if;
    
    // try k = kn
    if kn <> 1 and -1 <= kn and kn <= 1 then
      if kn <> -1 then
        s := 1
      else
        s := selectFirst(l, z -> not (is(z < 0)));
      end_if;
    elif kn <> 1 and -2 <= kn and kn <= 2 then
      s := selectFirst(l, z -> is(Re(z) > 0));
    end_if;
    if s <> FAIL then
      l := subsop(l, s = l[s]^kn);
      kn := 1;
      s := FAIL;
    end_if;

    // finally, try k = ks
    // we don't have to do this for IgnoreAnalyticConstraints
    if not options[IgnoreAnalyticConstraints] and ks <> 1 then
      if is(ks, Type::Interval([-1], [1])) = TRUE then
        if is(ks > -1) = TRUE then
          s := 1
        else
          s := selectFirst(l, z -> not (is(z < 0)));
        end_if;
      elif is(ks, Type::Interval([-2], [2])) = TRUE then
        s := selectFirst(l, z -> is(Re(z) > 0));
      end_if;
      if s <> FAIL then
        l := subsop(l, s = l[s]^ks);
        ks := 1;
      end_if;
    end_if;
    
    return(_mult(r, kn, ks, op(map(l, ln))));
    
  end_if; // type(e) = "_mult"

  // not a sum ==> finished
  if type(e) <> "_plus" then
    return(e)
  end_if;

  // simple method for IgnoreAnalyticConstraints: since a*ln(b) has already been combined to
  // ln(b^a)
  // it suffices to make ln(a) + ln(b) to ln(a*b)

  if options[IgnoreAnalyticConstraints] then
    l:= split([op(e)], testtype, "ln");
    if nops(l[1]) = 0 then
      return(e)
    end_if;
    r:= ln(combine(_mult(op(map(l[1], op, 1))), options)) +
        _plus(op(l[2]));
    return(r)
  end_if;
  
   // auxilliary procedure: try to write y = k*ln(x) syntactically
   // return the list [k, x] if successful, and x otherwise
  split_product :=
  proc(y)
    local l;
  begin
    if type(y) = "_mult" then
      l := select(y, testtype, "ln");
      if l = 1 then
        return(y)
      elif type(l) = "ln" then
        return([y/l, op(l)])
      else // l is a product of logs; take the last one
        l := op(l, nops(l));
        return([y/l, op(l)])
      end_if;
    elif type(y) = "ln" then
      return([1, op(y)])
    else
      return(y)
    end_if
  end_proc;
       
  l := map([op(e)], split_product);
  l := split(l, testtype, DOM_LIST);
  r := _plus(op(l[2]));
  l := l[1];
   // Now e = r + _plus(l[i][1] * ln(l[i][2]) $ i = 1..nops(l))

  // handle k*ln(x) +- k*ln(y)
  while nops(l) > 0 do
    k := l[1][1];
    l := split(l, z -> bool(z[1] = k));
    arguments1 := map(l[1], op, 2);
    l := l[2];
    l := split(l, z -> bool(z[1] = -k));
    arguments2 := map(l[1], op, 2);
    l := l[2];
     // all entries x of arguments1 occur in the form k*ln(x) in e,
     // all entries x of arguments2 occur in the form -k*ln(x) in e

     // do nothing if there is only one term
     if nops(arguments1) + nops(arguments2) > 1 then

       // Extract positive arguments
       arguments1 := split(arguments1, is, Type::NonNegative);
       positiveArguments := _mult(op(arguments1[1]));
       arguments1 := arguments1[2].arguments1[3];
       arguments2 := split(arguments2, is, Type::NonNegative);
       positiveArguments := positiveArguments/_mult(op(arguments2[1]));
       arguments2 := arguments2[2].arguments2[3];

       // multiply all positive arguments to the first of the other args
       if nops(arguments1) > 0 then
         arguments1 := subsop(arguments1, 1 = arguments1[1]*positiveArguments)
       elif nops(arguments2) > 0 then
         arguments2 := subsop(arguments2, 1 = arguments2[1]/positiveArguments)
       else
         arguments1 := [positiveArguments];
       end_if;

       // we have arrived at k* (_plus(ln(A) $A in arguments1) - _plus(ln(B) $B in arguments2) )
       // try to find out whether ln(A) - ln(B) can be made into ln(A/B), for some A, B
       [reals1, arguments1, s]:= split(arguments1, is, Type::Real, Goal = TRUE);
       [reals2, arguments2, s]:= split(arguments2, is, Type::Real, Goal = TRUE);
       i:= 1;
       while i <= nops(reals1) do 
         j:= 1;
         while j <= nops(reals2) do 
           // can reals1[i] and reals2[j] be combined? 
           // if A < B then ln(A) - ln(B) = ln(A/B) 
           if is(reals1[i] <= reals2[j], Goal = TRUE) then
           // replace reals1[i] by reals1[i]/reals2[j]
             reals1[i]:= reals1[i]/reals2[j];
             delete reals2[j];
             // and start over with the same i
             i:= i-1;
             break 
           elif is(reals1[i] >= reals2[j], Goal = TRUE) then 
             // ln(A) - ln(B) = - (ln(B) - ln(A)) = - ln(B/A) 
             // thus put this into the list of ln's with negative sign 
             reals2[j]:= reals2[j]/reals1[i];
             delete reals1[i];
             // go on with the next i. i need not be increased as one element has been deleted
             i:= i-1;
             break 
           end_if; 
           j:= j+1
         end_while;
         i:= i+1
       end_while;  
       arguments1:= arguments1.reals1;
       arguments2:= arguments2.reals2;
       
     end_if;

     // add linear combination of the logarithms of the combined arguments
     // to the final result r
     r := _plus(r, op(map(arguments1, z -> k*ln(z))),
                   op(map(arguments2, z -> -k*ln(z))))
   end_while;

  return(r)

end_proc:

// end of file