//-----------------------------------------------------------------------
// combine::log implemented in analogy to
// combine::ln 
//
// combine::log(ee): handles the call combine(ee, log). Tries to rewrite
// k*log(b, x)+l*log(b, y) ==> log(b, x^k*y^l) whenever possible.
//
// Descends recursively into expressions of type +, *, ^, but not
// into arguments of other function calls. In particular, arguments
// of exp are not combined.
// Properties are taken into account.
//
// Implemented rules:
//
// k*log(b, a) --> log(b, a^k) if
// *) k in R and a > 0
// *) -2 <= k <= 2 and Re(a) > 0
// *) -1 < k <= 1 or (k = 1 and not a < 0)
// This is done first.
//
// k*log(b, x) + k*log(b, y) --> k*log(b, x*y) if x > 0 or y > 0
// k*log(b, x) - k*log(b, y) --> k*log(b, x/y) if x > 0 or y > 0
//
// Not implemented: the latter two rules if Re(x) > 0 and Re(y) > 0
//
// Note:
// - 2*log(b, x) + log(b, 4) will NOT be combined, except if is(x < 0) = TRUE
// - products of logarithms, such as log(b, 2)*log(b, 3), are not combined
//
combine::log :=
proc(ee, options)
  local e, l, r, kn, ks, selectFirst, s, b,
  k, split_product, arguments1, arguments2, positiveArguments;
begin

   // combine(.., log) implies combine(.., ln);
   if has(ee, ln) then
      ee:= combine(ee, ln, options);
   end_if;

   // shortcut: do nothing if there are no logarithms
   if not has(ee, log) then
     return(combine(ee, ln, options));
   end_if;

   // recursively combine all operands
   e := ee;
   if contains({DOM_EXPR, DOM_SET, DOM_LIST, DOM_ARRAY}, domtype(e)) then
     e := stdlib::mapEvalChanges(e, combine::log, options);
   elif domtype(e) = DOM_POLY then
     e := mapcoeffs(e, combine::log, options)
   end_if;

   // shortcut: finished if there are no logarithms left
   if not has(e, log) then
     return(e);
   end_if;

   // handle k*log(b, x) --> log(b, x^k)
   if type(e) = "_mult" then

      // extract ln factors
     l := split([op(e)], testtype, "log");
     
     if nops(l[1]) = 0 then
       return(e)
     end_if;

      // split real factors in the coefficient
     kn := split(l[2], testtype, Type::Real);
     if options[IgnoreAnalyticConstraints] then
      // treat all factors like reals
       ks:= _mult(op(kn[2]));
       kn := _mult(op(kn[1]));
       r:= 1
     else
       ks := split(kn[2], is, Type::Real);
       r := _mult(op(ks[2].ks[3]));
       kn := _mult(op(kn[1]));
       ks := _mult(op(ks[1]));
     end_if;
     
     if kn*ks = 1 then
       // no real factors in the coefficient; we can do nothing
       return(e)
     end_if;

     l := map(l[1], X -> [op(X)]);
     // now e = r * kn * ks * _mult(log(b, l[i]) $ i = 1..nops(l))
     // kn is real numeric, ks is real symbolic.

     //----------------------------------------------------
      // auxilliary procedure: return the index of the first
      // element in the list l for which f returns TRUE
      // if none exists, return FAIL
     selectFirst :=
     proc(l, f)
       local i;
     begin
       if options[IgnoreAnalyticConstraints] then
         if nops(l) = 0 then
           FAIL
         else
           1
         end_if
       else  
         for i from 1 to nops(l) do
           if f(l[i]) = TRUE then
             return(i)
           end_if
         end_for;
         FAIL
       end_if
     end_proc;
      //----------------------------------------------------

      // is there a logarithm with positive argument?
      // avoid expanding k*log(n) -> log(n^k)
     // for integers n and large k
     if (domtype(kn) = DOM_INT or domtype(kn) = DOM_RAT) and
       specfunc::abs(kn) > 100 then
       s:= selectFirst(l, z -> bool(domtype(z[2]) <> DOM_INT and
                        domtype(z[2]) <> DOM_RAT) and
                       is(z[2], Type::NonNegative))
     else
       s := selectFirst(l, z -> is(z[2], Type::NonNegative))
     end_if;
     
     if s <> FAIL then
       // pull the real factors as exponents into the first
       // logarithm with positive argument
        // but take care that this does not become too large
       if combine::isSmall(l[s][2], kn*ks) then
         l := subsop(l, s = [l[s][1], (l[s][2])^(kn*ks)]);
       else
         r:= r*kn*ks
       end_if;
       return(_mult(r, op(map(l, z -> log(z[1], z[2])))))
     end_if;

     // try the rules for -1 <= k <= 1 and -2 <= k <= 2

     // first try k = kn*ks
      if ks <> 1 then
        if options[IgnoreAnalyticConstraints] then
          s:= 1
        elif is(kn*ks, Type::Interval([-1], [1])) = TRUE then
          if is(kn*ks > -1) = TRUE then
            s := 1
          else
            s := selectFirst(l, z -> not (is(z[2] < 0)));
          end_if;
        elif is(kn*ks, Type::Interval([-2], [2])) = TRUE then
          s := selectFirst(l, z -> is(Re(z[2]) > 0));
        end_if;
        if s <> FAIL then
          l := subsop(l, s = [l[s][1], (l[s][2])^(kn*ks)]);
          return(_mult(r, op(map(l, z -> log(z[1], z[2])))))
        end_if;
      end_if;

      // try k = kn
      if kn <> 1 then
        if options[IgnoreAnalyticConstraints] then
          s:= 1
        elif -1 <= kn and kn <= 1 then
          if kn <> -1 then
            s := 1
          else
            s := selectFirst(l, z -> not (is(z[2] < 0)));
          end_if;
        elif -2 <= kn and kn <= 2 then
          s := selectFirst(l, z -> is(Re(z[2]) > 0));
        end_if;
      end_if;
      if s <> FAIL then
        l := subsop(l, s = [l[s][1], (l[s][2])^kn]);
        kn := 1;
        s := FAIL;
      end_if;

      // finally, try k = ks
      if ks <> 1 and is(ks, Type::Interval([-1], [1])) = TRUE then
        if is(ks > -1) = TRUE then
          s := 1
        else
          s := selectFirst(l, z -> not (is(z[2] < 0)));
        end_if;
      elif ks <> 1 and is(ks, Type::Interval([-2], [2])) = TRUE then
        s := selectFirst(l, z -> is(Re(z[2]) > 0));
      end_if;
      if s <> FAIL then
        l := subsop(l, s = [l[s][1], (l[s][2])^ks]);
        ks := 1;
      end_if;

      return(_mult(r, kn, ks, op(map(l, z -> log(z[1],z[2])))));

   end_if; // type(e) = "_mult"

   //-----------------------------------------------------------
   // not a sum ==> finished
   if type(e) <> "_plus" then
     return(e);
   end_if;
   //-----------------------------------------------------------

   //-----------------------------------------------------------
   // from here, e is of type "_plus":
   //-----------------------------------------------------------

   assert(type(e) = "_plus");

 // simple method for IgnoreAnalyticConstraints: since a*log(b) has already been combined to
  // log(b^a)
  // it suffices to make log(a) + log(b) to log(a*b)

  if options[IgnoreAnalyticConstraints] then
    l:= split([op(e)], testtype, "log");
    if nops(l[1]) = 0 then
      return(e)
    end_if;
    arguments1:= map(l[1], op, 1);
    if nops({op(arguments1)}) > 1 then
      // should we rewrite to ln here ?!
      return(e)
    else
      b:= arguments1[1]
    end_if;
    
    r:= log(b, combine(_mult(op(map(l[1], op, 2))), options)) +
        _plus(op(l[2]));
    return(r)
  end_if;


  
   //-----------------------------------------------------------
   // auxilliary procedure: try to write y = k*log(b, x) syntactically
   // return the list [k, [b, x]] if successful, and y otherwise
   split_product := proc(y)
     local l;
   begin
     if type(y) = "_mult" then
       l := select(y, testtype, "log");
       if l = 1 then
         return(y)
       elif type(l) = "log" then
         return([y/l, [op(l)]])
       else // l is a product of logs; take the last one
         l := op(l, nops(l));
         return([y/l, [op(l)]])
       end_if;
     elif type(y) = "log" then
       return([1, [op(y)]])
     else
       return(y)
     end_if
   end_proc;
   //-----------------------------------------------------------
       
   l := map([op(e)], split_product);
   l := split(l, testtype, DOM_LIST);
   r := _plus(op(l[2]));
   l := l[1];
   // Now e = r + _plus(l[i][1] * log(l[i][2][1], l[i][2][2]) $ i = 1..nops(l))

   // handle k*log(b, x) +- k*log(b, y)
   while nops(l) > 0 do
     k := l[1][1];
     b := l[1][2][1];
     l := split(l, z -> bool(z[1] = k and z[2][1] = b));
     arguments1 := map(l[1], op, [2, 2]);
     l := l[2];
     l := split(l, z -> bool(z[1] = -k and z[2][1] = b));
     arguments2 := map(l[1], op, [2, 2]);
     l := l[2];
     // all entries x of arguments1 occur in the form k*log(b,x) in e,
     // all entries x of arguments2 occur in the form -k*log(b,x) in e

     // do nothing if there is only one term
     if nops(arguments1) + nops(arguments2) > 1 then

       // Extract positive arguments
       arguments1 := split(arguments1, z -> is(z, Type::NonNegative));
       positiveArguments := _mult(op(arguments1[1]));
       arguments1 := arguments1[2].arguments1[3];
       arguments2 := split(arguments2, z -> is(z, Type::NonNegative));
       positiveArguments := positiveArguments/_mult(op(arguments2[1]));
       arguments2 := arguments2[2].arguments2[3];

       // multiply all positive arguments to the first of the other args
       if nops(arguments1) > 0 then
         arguments1 := subsop(arguments1, 1 = arguments1[1]*positiveArguments)
       elif nops(arguments2) > 0 then
         arguments2 := subsop(arguments2, 1 = arguments2[1]/positiveArguments)
       else
         arguments1 := [positiveArguments];
       end_if;

     end_if;

     // add linear combination of the logarithms of the combined arguments
     // to the final result r
     r := _plus(r, op(map(arguments1, z ->  k*log(b, z))),
                   op(map(arguments2, z -> -k*log(b, z))))
   end_while;

   return(r)

end_proc:

// end of file