/*++
conj.mu

	conjugate -- A function for computing the complex 
	             conjugate of an expression


	For computing the complex conjugate of an expression x,
	the method "conjugate" of the domain of x will be used.
	If such a method does not exist, then 'conjugate(x)' 
	will be returned.

	The basis types DOM_INT, DOM_RAT, DOM_FLOAT, DOM_COMPLEX
	and DOM_EXPR be an exception. These types are treated by
	the procedure 'conjugate'.
	For expressions of type DOM_EXPR the procedure 'conjugate'
	looks for a function attribut 'conjugate', which will
	be used for x if such exists. Otherwise either, if the
	function environment of x is one of the built-in environ-
	ments "_plus", "_mult" or "_power", x will be treated by
	the procedure 'conjugate' or the expression 'conjugate(x)'
	will be returned.
++*/

conjugate :=
proc(x)
  local t,i;
begin
  if args(0) = 0 then
    error("conjugate called without arguments")
  end_if;
  if (t:= x::dom::conjugate) <> FAIL then
    return( t(args()) )
  end_if;

  if args(0) <> 1 then error("wrong no of args") end_if;

  t := domtype(x);
  case t
    of DOM_POLY do
      return(mapcoeffs(x, conjugate));
    of DOM_LIST do
    of DOM_ARRAY do
    of DOM_SET do
    of DOM_TABLE do
      return(map(x, conjugate));
    of DOM_HFARRAY do
      return(hfa::conjugate(x));
    of DOM_INT      do
    of DOM_RAT      do
    of DOM_INTERVAL do
    of DOM_FLOAT    do return( x );
    of DOM_COMPLEX  do return( op(x,1) - op(x,2)*I );
    of DOM_EXPR     do
      t := op(x,0);
      if domtype(eval(%)) = DOM_FUNC_ENV and
        (i:= slot(eval(%),"conjugate")) <> FAIL then
        return( i( op(x) ) )
      elif domtype(eval(%)) = DOM_IDENT then
        // f(x)
        return( procname(args()) )
      end_if;
      
      if t = hold(_plus) or t = hold(_mult) then
        return( eval(t)(conjugate(op(x,i)) $ i=1..nops(x)) )
      end_if;
      /* x = a^b = exp(b*ln(a) unless a=0
         therefore conjugate(x) = conjugate(exp(b*ln(a)))
         = exp(conjugate(b*ln(a))) = exp(conjugate(b) * conjugate(ln((a))))
         Normally, we are not able to express conjugate(ln(a)) in closed
         form. However, for a that are not on the branch cut of ln,
         this simplifies to a^conjugate(b)
         and for integers b, it simplifies to conjugate(a)^b
         for a negative number a, conjugate(a^(1/2)) = -a^(1/2)
      */ 
      if t = hold(_power) then
        if (i:= is(op(x,1) < 0)) = FALSE then
          return( conjugate(op(x,1))^conjugate(op(x,2)) )
        elif i = TRUE and (op(x, 2) = 1/2 or op(x, 2) = -1/2) then
          return(-x)
        elif is(op(x, 2) in Z_) = TRUE then
          return(conjugate(op(x, 1))^op(x, 2) )
        else
          // too complicated
          // return(exp::expand(conjugate(op(x, 2))* conjugate(ln(op(x, 1)))))
          break
        end_if
      end_if
  end_case;
  
  if is(x,Type::Real)=TRUE then
    x
  else
    procname(x)
  end_if
end_proc:

conjugate:= funcenv
(conjugate,
// output                     
proc(x)
  local pretty_arg;
begin
  if PRETTYPRINT then
    pretty_arg := strprint(All, op(x));
    // wider than TEXTWIDTH?
    if pretty_arg[2] <> pretty_arg[4] then
      FAIL
    else
      stdlib::Exposed(_concat("_" $ pretty_arg[3], "\n",
                              pretty_arg[1]));
    end_if;
  else
    FAIL
  end_if;
end,
table( "type"="conjugate", "print"="conjugate", "conjugate"=id,
       "info"="conjugate(x) -- the complex conjugate of x" )
):

// 
// Further Function Attributes 
// 
conjugate::float :=  proc(x) begin
  x:= float(x);
  case domtype(x)
  of DOM_FLOAT do
  of DOM_COMPLEX do
     return(conjugate(x))
  otherwise
     return(hold(conjugate)(x))
  end_case;
end_proc:

conjugate::rectform :=
   loadproc(conjugate::rectform, pathname("STDLIB", "RECTFORM"), "conj"):

conjugate::Content :=  stdlib::genOutFunc("Cconjugate", 1):

conjugate::complexDiscont := {}:
