/*--
	lngamma/expand -- the function attribut "expand" for gamma

        implements the following rules for
        a of Type::Real and b of Type::PosInt:

0) lngamma(b) = ln(1 * ... * (b - 1))

1) lngamma(x + a) = lngamma(x + frac(a)) + _plus(ln(x + frac(a) + k) $ k = 0..ceil(a) - 1) if a > 0,
   lngamma(x + a) = lngamma(x + frac(a)) - _plus(ln(x + frac(a) - k) $ k = 0..ceil(a) - 1) if a < 0,
   (also for x = 0 or x of Type::Imaginary)
   
2) lngamma(b*x) = b*x*ln(b) - ln(b)/2 - (b-1)/2*ln(2)-(b-1)/2*ln(PI)
                 + _plus(lngamma(x + k/b) $ k = 0..b-1)
   (also for x = I)

3) lngamma(-x) = much too complicated, not implemented
--*/

lngamma::expand := prog::remember(
proc(g: "lngamma")
  local a, fa, b, r, j, e;
  name lngamma::expand;
begin
  // recursively expand the argument
  e := expand(op(g, 1));

  if domtype(e) = DOM_INT then // rule 0
    if e > 0 then
      return(ln(_mult(j $ j = 1 .. (e - 1))))
    else // should not happen, since already done in SPECFUNC/gamma.mu
      error("singularity")
    end_if
  end_if;

  r := 0;
  if type(e) = "_plus" or testtype(e, Type::Numeric) then // rule 1
    if type(e) = "_plus" then
      a := select(e, testtype, Type::Numeric);
    else
      a := e;
    end_if;
    fa := floor(Re(a));
    e := e - fa;
    if fa >= 0 then
      r := r + _plus( ln(e + j) $ j = 0 .. fa-1);
    else // fa < 0
      r := r + _plus(-ln(e + j) $ j = fa .. -1)
    end_if;
  end_if;

  if type(e) = "_mult" or 
     domtype(e/I) = DOM_INT then // rule 2
    if type(e) = "_mult" then
      b := select(e, testtype, Type::Numeric);
    else
      b := e;
    end_if;
    if domtype(b/I) = DOM_INT then
      b := b/I;
    elif domtype(b) <> DOM_INT then
      b := 1;
    end_if;
    e := e/b;
    if b > 0 then
      return(
          b*e*ln(b)
         -ln(b)/2 
         -(b-1)/2*ln(2)
         -(b-1)/2*ln(PI)
         + _plus(lngamma(e + j/b) $ j = 0 .. (b - 1))
         + r
       ):
    end_if;
  end_if;

  return(lngamma(e) + r);
end_proc, () -> [property::depends(args()), DIGITS]):

