// hypergeom::float(as,bs,z) where as = [a1,a2,...ap] bs = [b1,b2,...aq]

// This code assumes the parameters to be of reasonable range
// as there is very little in terms of asymptotic expansions for
// large parameters (see Luke vol. 1 section 4.7 for 1F1 case)
// However, the range for Z is unlimited.

// If p = q+1 there is a branch cut along [1,infinity), closure from below.
// Otherwise pFq is entire.
//
// Evaluation near |z|=1 is difficult, and can take a long time in some
// cases with this code.
//

alias(hypergeom_float=hypergeom::float):

alias(Hypergeom=specfunc::Hypergeom):

alias(ordinary=Hypergeom::ordinary):
alias(confluent=Hypergeom::confluent):
alias(flip=Hypergeom::flip):
alias(f2F1_reflect=Hypergeom::f2F1_reflect):
alias(typenegfloatint=Hypergeom::typenegfloatint):
alias(typenumcomplex=Hypergeom::typenumcomplex):
alias(sel=Hypergeom::sel):
alias(asympt=Hypergeom::asympt):
alias(asy_slater_cond=Hypergeom::asy_slater_cond):
alias(spf1=Hypergeom::spf1):
alias(spf2=Hypergeom::spf2):
//
alias(sexp=specfunc::exp):
alias(sabs=specfunc::abs):
alias(sgamma=specfunc::gamma):
alias(sfrac= specfunc::Hypergeom::frac);
//

hypergeom_float := proc()
    local b, as, bs, z, s, flag,i, j, p, q, a0, b0, n, nm1, bet, sc, zcrit,
          a1, a2;
begin
  if args(0) <> 3 then
    error(_concat("expecting 3 arguments, got ",args(0))):
   end_if:

   as := float(args(1)):
   bs := float(args(2)):
   z  := float(args(3)):
   if type(as) <> DOM_LIST or type(bs) <> DOM_LIST then
      error("invalid arguments"):
   end_if:

   // trivial case
   if iszero(z) then
      return(float(1)):
   end_if:
   // remove common terms (PBM 7.2.3  no. 7
   // IMPORTANT Note: this rule applies even when the common terms are
   // zero or negative integers 
   // (see e.g. Luke. Vol. 1, p. 41 section 3.1 (25)
   //for i from nops(as) by -1 to 1 do
   if as<>[] and bs<>[] then
          for i from nops(as) downto 1 step 1 do
              j:=contains(bs, as[i]):
              if j <> 0 then
                        as := subsop(as,i=null()):
                        bs := subsop(bs,j=null()):
             end_if:
          end_for:
   end_if:
   if not _and(op(map(as,typenumcomplex)),
               op(map(bs,typenumcomplex)),typenumcomplex(z))
    //------------------------------------------------------
    // Walter 25.3.04: replaced return(procname(..) by
    // return(hypergeom(..)). This is consistent with the
    // behavior of other specfuncs.
    // Voldemort's code  hypergeom::float(symbolic) --> hypergeom::float(symbolic)
    // causes havoc when trying to plot hypergeom.
    // This change fixes the plot problem
    //------------------------------------------------------
    // then return(procname(as,bs,z)):
      then return(hypergeom(as,bs,z)):
   end_if:
   p := nops(as):
   q := nops(bs):
   // handle polynomial and denominator singularity cases
   a0 := sel(as):
   b0 := sel(bs):
   if b0 > a0 then
      error("singularity encountered"):
      // Case when hypergeometric function reduces to a polynomial
   elif a0 > b0 and a0 > -infinity then
      ([flag,s]):=ordinary(as,bs,z):
      return(s):
//Special Cases:
   elif p = 0 and q = 0 then
      sexp(z):
// 1F0(a:z) = (1-z)^(-a) except at z=1
   elif p = 1 and q = 0 then
     if iszero(z-1) then
        if Re(as[1]) < 0 then
           return(float(0)):
        else
           error("singularity encountered"):
        end_if:
     else
        return(float((1-z)^(-as[1]))):
     end_if:
// Use 7.13.1 no. 1 of Prudnikok et al. (PBM) vol. 3
   elif p = 0 and q = 1 then
      float(sgamma(bs[1])*(z^((1-bs[1])/2))*besselI(bs[1]-1,2*sqrt(z))):
   elif p = 1 and q = 1 then 
      if iszero(Im(z)) and Re(z) > 0 and 
        type(float(as[1]))=DOM_FLOAT and 
        type(float(bs[1]))=DOM_FLOAT and
        iszero(sfrac(as[1])) and testtype(round(as[1]),Type::PosInt) and as[1] < 7 and 
        type(float(bs[1]))=DOM_FLOAT and
        ( not(iszero(sfrac(bs[1]))) or 
        (iszero(sfrac(bs[1])) and 
        not(testtype(round(bs[1]-2),Type::NegInt)))) then 

           // test for 7.11.1 no. 11 of PBM vol. 3
           // but derivative is limited at as[1]=6
           // c.f. issues of instability as discussed with Walter.
           // However, later on, we would be desirable to better
           // exploit this formula.
           nm1 := round(as[1])-1:
           s:= spf1(nm1,z,bs[1]):
           if s <> FAIL then
              return(float(((bs[1]-1)/sgamma(as[1]))*s));
           end_if;
        end_if;
        // The threshold between Taylor series and use of
        // asymptotics is set at 50. - see if this works.
        if sabs(z) > float(50) and not(typenegfloatint(as[1])) then 
           // try asymptotic form of 1F1:
           // estimate by empirical formula, maximum value
           // Digits for which asymptotics are successful
           if DIGITS < round(4+3*(sabs(as[1])-sabs(bs[1]))+sabs(z)/2) then
              ([flag,s]):=confluent(as[1],bs[1],z):
           else
              flag:=1:
           end_if:
           if not(iszero(flag)) then
              ([flag,s]):=ordinary(as,bs,z):
           end_if:
           return(s):
       else
           ([flag,s]):=ordinary(as,bs,z):       
           return(s):
       end_if:
       // if igamma would have a wider domain! The above and following identity
       // could be exploited more extensively!

    //==========================================================
    // The following code computes hypergeom([n, b], [], z) with
    // a positive integer n between 1 and 6 and negative real z 
    // using specfunc::Hypergeom::spf2 
    //==========================================================
    elif p = 2 and q = 0 and 
      iszero(Im(z)) and Re(z) < float(0) and 
      type(float(as[1]))=DOM_FLOAT and 
      type(float(as[2]))=DOM_FLOAT and 
        ( 
          ( as[1] < 8 and 
            iszero(sfrac(as[1])) and testtype(round(as[1]),Type::PosInt)
            and not(iszero(_mult((as[2]-i)$i=1..round(sabs(as[1]-1)))))) 
        or 
          ( as[2] < 8 and
            iszero(sfrac(as[2])) and testtype(round(as[2]),Type::PosInt) 
            and not(iszero(_mult((as[1]-i)$i=1..round(sabs(as[2]-1))))))
        ) then
      if iszero(sfrac(as[1])) and 
        testtype(round(as[1]),Type::PosInt) and 
        not(iszero(_mult((as[2]-i)$i=1..round(sabs(as[1]-1))))) then
          n:=round(as[1]-1):
          b:=as[2]:
      else
          n:=round(as[2]-1):
          b:=as[1]:  
      end_if:
      sc:=_mult((b-i)$i=1..n);
      float((((-1)^n)/(gamma(n+1)*sc))*spf2(n,-z,b)/(-z)^(2*n));
    else
      //------------------------------------------------
      // Walter 2.1.2006:
      // There are terrible convergence problems such as
      // hypergeom([1/5, 1], [6/5], 0.99999) (taking several
      // minutes) when evaluating close to the singularity
      // at z = 1. We use A&S 1972, p. 559:
      //   2F1([a1, a2], [b], z) = 2F1([a2, a1], [b], z) 
      // = 2F1([a1, b - a2], [b], z/(z -1)) / (1 - z)^a1
      // = 2F1([b - a1, a2], [b], z/(z -1)) / (1 - z)^a2
      // to transform the argument close to infinity to
      // make use of fast asymptotics
      if p = 2 and q = 1 and
      // specfunc::abs(z - 1) < 0.1 and 
         domtype(z) = DOM_FLOAT and 
         z < 1 and
         z > 0.9 then
         [a1, a2]:= as;
         [b]:= bs;
         return(hypergeom::float([b - a1, a2], [b], z/(z-1))/(1 - z)^a2);
      end_if;
      //----- end of Walter's change ------------------

      // Try asymptotic expansions and at last resort the Taylor
      // series expansion.
      // Determine critical value for z for which asymptotics should
      // be tried.
      bet:=q+1-p:
      zcrit:=100:
      // The higher q is w.r.t. to p, the better the chance of the
      // Taylor series converging and the worse are the chances of
      // the asymptotics...
      if bet > 1 then zcrit:=bet*100: end_if;
              
      // First try to see if Slater's theorem can handle this case:
      if (p = q and asy_slater_cond(as,bs,z)) then
         return(float(flip(as,bs,z))):        
      end_if:
      // Failing that, go for the nitty-gritty
      // If the asymptotics work, steal the result and RUN
      if sabs(z) > zcrit then
         ([flag,s]):=asympt(as,bs,p,q,z):
         if flag=0 then return(float(s)): end_if;
      end_if:
      if p <= q or p = q+1 and sabs(z) < 1 then
         ([flag,s]):=ordinary(as,bs,z):        
         return(s):                
      elif p=2 and q=1 and 
           not(iszero(z-1)) and 
           sabs(1-sabs(z)) < 1/10 and 
           sabs(1-sabs(1-z)) >= 1/100 then
          f2F1_reflect(as[1],as[2],bs[1],z):
      // Should probably implement A&S 15.3.7/13/14 for large |z|, but this
      // code will work (via Slater's theorem) for the time being.  Note that
      // the points +-1/2+-sqrt(3)/2*I form an invariant set under all of the
      // linear 2F1 transformations (hence the 2nd inequality above).
      /*
      elif p=2 and q=1 and 
           sabs(1-sabs(z)) < 1/10 and
           sabs(1-sabs(1-z)) < 1/100 then
           // we are close to one of the fixed points
           //  z = 1/2  +/-  sqrt(3)*I/2
           
           need some code for this special case

      */
      elif p > q+1 or (p = q+1 and sabs(z) > float(1)) then
          return(float(flip(as,bs,z))):        
      else  // p = q+1 and |z| = 1
        n := 1 - _plus(op(as)) + _plus(op(bs)):
        if iszero(z-1) and Re(n) > 1 or 
           not(iszero(z-1)) and Re(n) > 0 then
           // for 2F1 around z=1 use. 15.1.20 of A & S 
          if p=2 and q=1 and iszero(z-1) and 
             not(typenegfloatint(bs[1])) then
                return(float(gamma(bs[1])*gamma(n-1)/
                            (gamma(bs[1]-as[1])*
                             gamma(bs[1]-as[2])))):
          end_if:
          ([flag,s]):=ordinary(as,bs,z):                
          return(s):        
        else
          error("singularity encountered or cannot compute the result"):
        end_if:
      end_if:
  end_if:
end_proc:

hypergeom_float:= prog::remember(hypergeom_float, () -> [property::depends(args()), DIGITS]):

unalias(hypergeom_float):
unalias(Hypergeom):
unalias(ordinary):
unalias(confluent):
unalias(f2F1_reflect):
unalias(flip):
unalias(typenegfloatint):
unalias(typenumcomplex):
unalias(sel):
unalias(asy_slater_cond):
unalias(asympt):
unalias(spf1):
unalias(spf2):
unalias(sexp):
unalias(sabs):
unalias(sgamma):
unalias(sfrac):

