/*++
Im.mu

    Im -- A function for computing the imaginary part of
              an expression


    For computing the imaginary part of an expression x,
    the method "Im" of the domain of x will be used.
    If this method does not exist, then 'Im(x)' will be
    returned.

    The basis types DOM_INT, DOM_RAT, DOM_FLOAT, DOM_COMPLEX
    and DOM_EXPR be an exception. These types are treated by
    the procedure  'Im'.
    For expressions of type DOM_EXPR the procedure 'Im'
    looks for a function attribute 'Im', which will be used
    for x if such exist.
    Otherwise either, if the function environment of x is one
    of the built-in environments "_plus", "_mult" or "_power",
    x will be treated by the procedure 'Im' or the expression
    'Im(x)' will be returned respectivly.
++*/

Im := 
proc(x)
  name Im;
  local t, fx, multSplit, a, ai;
begin
  if (t:= x::dom::Im) <> FAIL then
    return( t(args()) )
  end_if;
  if args(0) <> 1 then
    error("wrong no of args")
  end_if;

  case (t:= domtype(x))
    of DOM_POLY do
      return(mapcoeffs(x, Im));
    of DOM_LIST do
    of DOM_ARRAY do
    of DOM_SET do
    of DOM_TABLE do
      return(map(x, Im));
    of DOM_HFARRAY do
      return(hfa::Im(x));
    of DOM_INT      do
    of DOM_RAT      do return(0)
    of DOM_FLOAT    do return(0.0)
    of DOM_COMPLEX  do return(op(x,2))
  end_case;

  if testtype(x,Type::Constant) then
    // is x real ?
    if traperror((fx := float(x))) = 0 and
      domtype(fx) = DOM_FLOAT then return(0)
      // is x an indexed identifier ?
    elif type(x) = "_index" then
      return(procname(x))
    end_if
  end_if;

  if t = DOM_EXPR then
    t:= eval(op(x,0));
    if domtype(t) = DOM_FUNC_ENV then
      a:= slot(t,"Im")(op(x));
      if a <> FAIL then return( a ) end_if
    elif domtype(t) = DOM_IDENT then
      // case f(x):
      return( procname(x) )
    end_if;

    multSplit :=
    proc(z)
      local t,a,b,ci,cr,d;
    begin
      // split product into real and imaginary parts if possible
      cr := 1; ci := 0;
      for t in op(z) do
        a := Re(t);
        // if has(a,Re) then return( FAIL ) end_if;
        b := Im(t);
        // if has(b,Im) then return( FAIL ) end_if;
        d := cr;
        cr := cr*a - ci*b;
        ci := d*b + ci*a
      end_for;
      if cr = undefined or ci = undefined then
        // may happen e.g. if 0*infinity was computed
        FAIL
      else
        [cr,ci]
      end_if
    end_proc:

    case t
      of _union do
      of _plus do
        return( map(x,Im) )
      of _mult do
        // find constant terms
        t:= split( x,testtype,Type::Real );
        if t[1] <> 1 and t[1] <> 1.0 then
          return( t[1]*Im(t[2]*t[3]) )
        end_if;

        // split product into real and imaginary parts if possible
        t := multSplit(x);
        if t <> FAIL and
          length(t[2]) <= 3/2*length(procname(x)) then
          // result is not much more complicated than the unevaluated call
          return( t[2] )
        end_if;

        // test if the expression is a symbolic complex number
        if (a:= contains( [op(x)],I )) <> 0 then
          // I*ex = (Re(ex)+Im(ex)*I)*I = Re(ex)*I-Im(ex)
          /* note: using 'contains', 1/I = -I will not be considered where
this rule would lead to a wrong result
*/
          return( Re( eval(subsop(x,a=1)) ) )
        else
          return( procname(x) )
        end_if
      of _power do
        if op(x,2) = 1/2 then
          case domtype( op(x,1) )
            of DOM_INT     do
            of DOM_RAT     do
              if op(x, 1) >= 0 then
                return(0)
              else
                return((-op(x,1))^(1/2))
              end_if
            of DOM_FLOAT   do
              // _power should have simplified this!
              if op(x, 1) >= 0 then
                return(0.0)
              else
                return((-op(x,1))^(1/2))
              end_if
            of DOM_COMPLEX do
              t := op(op(x,1));
              return(sign(op(t,2))*sqrt((-op(t,1)+sqrt(op(t,1)^2+op(t,2)^2))/2))
            otherwise
              if is(op(x,1),Type::Real)=TRUE then
                if is(op(x,1)>=0)=TRUE then
                  return(0)
                elif is(op(x,1)>=0)=FALSE then
                  return(sqrt(-op(x,1)))
                else
                  return(procname(x))
                end_if
              end_if
          end_case
        elif domtype(op(x,2)) = DOM_INT then
          case domtype(op(x,1))
            of DOM_INT      do
            of DOM_RAT      do
            of DOM_FLOAT    do return( 0 )
            otherwise
              a := Re(op(x,1));
              ai := Im(op(x,1));
              if has(a,{Re,Im}) or has(ai,{Re,Im}) then
                return( procname(x) )
              end_if;
              t := [a,ai];
              if op(x,2) > 0 and op(x, 2) < Pref::autoExpansionLimit()
                then
                t := multSplit( [(a+I*ai) $ op(x,2)] );
                if t = FAIL then
                  return( procname(x) )
                else
                  return( t[2] )
                end_if
              elif op(x,2) < -1 and
                op(x, 2) > -Pref::autoExpansionLimit() then
                t := multSplit( [(a+I*ai) $ -op(x,2)] );
                if t = FAIL then
                  return( procname(x) )
                else
                  return( -t[2]/(t[1]^2+t[2]^2))
                end_if
              elif op(x,2) = -1 then
                return( -t[2]/(t[1]^2+t[2]^2))
              end_if;
          end_case
        elif testtype( op(x,2),Type::Numeric ) then
          /* hold is *very* important, otherwise
             exp simplifies back to op(x,1)^op(x,2)
             and we get an infinite loop
           */
          t := Im(hold(exp)(op(x,2)*ln(op(x,1))));
          if type(t) <> "Im" then return( t ) end_if
        end_if
    end_case
  end_if;

  // uses information from assume
  if is(x in R_, Goal = TRUE) then
    return(0)
  end_if;
  if is(x,Type::Imaginary)=TRUE then
    return(x/I)
  end_if;
  procname(x)
end_proc:


Im := prog::remember(Im,
                     () -> [property::depends(args()), DIGITS]
                     ):



Im:= funcenv(Im,
             NIL,
             table( "type"="Im", "print"="Im", "Im"=0,
                   "Re"= proc()
                         begin
                           hold(Im)(args(1))
                         end_proc,
                   "float"= proc(x)
                              name floatIm;
                            begin
                              Im(float(x))
                            end_proc,
                   "info"="Im(x) -- the imaginary part of x" )
             ):

//
// Further Function Attributes
//

// minimal property
Im::minprop := Type::Real:

Im::rectform :=
   loadproc(Im::rectform, pathname("STDLIB","RECTFORM"), "Im"):

Im::series := loadproc(Im::series, pathname("SERIES"), "Im"):

Im::lazy :=
proc(x)
begin
  if indets(x) = {} then
    Im(x)
  else
    procname(args());
  end_if
end_proc:

// for plot ...
Im::float := funcenv(Im::float):
Im::float::lazy := Im::lazy:

Im::diff := f -> Im(diff(op(f), args(2..args(0)))):

Im::hull := () -> Im(hull(args())):

Im::Content := stdlib::genOutFunc("Cimaginary", 1):
Im::getprop := proc( x, options )
  local r;
begin
  r := property::_getpropRec( op(x), options );
  case type(r)
    of solvelib::BasicSet do
      if r=C_ then return( R_ ); end_if;
      return( {0} );
    of Dom::Interval do
      return( {0} );
    of Dom::ImageSet do
      return( Im(r) );
    of DOM_SET do
      return( map(r,Im) );
  end_case;
  R_;
end_proc:

// simplify slot
Im::simplify := proc(x)
                  local r;
                begin
                  r := Im(expr(rectform(op(x,1))));
                  if not hastype(r, {"Re", "Im"}) then
                    simplify(r)
                  else
                    r
                  end_if
                end_proc:

// end of file
