/*++
O -- the domain of order terms of polynomial type

O's are used to represent order terms of series for example. They may
be of polynomial type only.

f(x) = O(g(x), x=x_0) means that there is a constant c with
|f(x)/g(x)| < c as x --> x_0. Note that x_0 may not be 'infinity'!

An order term has 2 operands:
- an expression representing the term,
- a set of equations v=p where v is an indeterminate and p a limit point.

The expression is simplified by O::create to get some crude 'normal
form' (not that is really has a normal form).

The operators +, -, *, 1/ and ^ are defined for order terms. Note that
1/ may cause an error if the operator is not O(1).
++*/

O:= newDomain("O"):
O::create_dom:=hold(O):
O::info:= "Domain 'O': Big O order terms of series.":
O::interface:= {}:


/*--
O::new -- create order term

O::new(e, [v1=p1,...])

e  - polynomial expression
vi - variables
pi - limit points

Create order term from expression e. The limit points are all assumed to
be 0, if not given explicitly.
--*/

O::new:=
proc(x)
  local p, flatO;
  name O;
begin
  if args(0) = 0 then
    error("wrong no of args")
  end_if; 

  if x::dom::O <> FAIL then
    return(x::dom::O(args()))
  end_if;

  if domtype(x) = DOM_POLY then 
    x:= expr(x)
  end_if; 

  case args(0)      
    of 1 do
      p:= O::getIndets(x);
      p:= map(p, _equal, 0);
      break;
    otherwise
      p:= {args(2..args(0))};
      if testargs() then
        O::checkargs(p);
      end_if;
  end_case;

    // flat out nested O's 
  if hastype(x, O) then
    flatO:=
    proc()
    begin
      p:= p union extop(args(1),2);
      extop(args(1),1);
    end_proc ;
    x:= misc::maprec(x, {O}=flatO, PreMap);
    if nops(map(p, op, 1)) <> nops(p) then
      error("multiple limit point")
    end_if;
  end_if;

  O::create(x, p);
end_proc:

/*--
O::getIndets -- local method to obtain DOM_IDENT's and indexed identifiers

O::getIndets(x)

x -- order term
--*/

O::getIndets:=
proc(x)
  local p;
begin
  p:= misc::subExpressions(x, {DOM_IDENT, "_index"}, NoOperators) minus
  Type::ConstantIdents;
  // remove operators of index expressions
  p minus map(select(p, y -> type(y)="_index"), op, 1);
end_proc:

/*--
O::checkargs(p) - checks p is a set of equations of the form
variable = value, where each "variable" is an (indexed) identifier and
occurs at most once and "value" is a limit point
--*/

O::checkargs :=
proc(p)
begin
  if p = {} then
    return()
  end;
  if map(p, type) <> {"_equal"} then
    error("equation expected")
  end_if;
  if map(p, type@op, 1) union {DOM_IDENT, "_index"}
    <> {DOM_IDENT, "_index"} then
    error("lhs must be an (indexed) identifier")
  end_if;
  if nops(map(p, op, 1)) <> nops(p) then
    error("multiple limit point")
  end_if;
end_proc:

/*--
O::expr -- convert order term into expression

O::expr(o)

o - order term
--*/

O::expr:= proc() begin 
    hold(O)(expr(extop(args(1),1)), op(
		select(extop(args(1),2), proc() begin op(args(1),2) <> 0 end_proc )))
 end_proc ;

O::print:= x-> hold(O)(generate::sortSums(expr(extop(x,1))),
                       op(select(extop(x,2), xx->op(xx,2) <> 0))):


/*--
O::TeX -- return TeX-string for order term

O::TeX(o)

o - order term
--*/

O::TeX:= proc() begin generate::TeX(O::expr(args(1))) end_proc :

/*--
O::indets -- return list of indets

O::indets(o)

o - order term
--*/

O::indets:= proc() begin map(extop(args(1),2), op, 1) end_proc :

O::freeIndets:= O::indets:

/*--
O::points -- return list of limit points

O::points(o)

o - order term
--*/

O::points:= proc() begin extop(args(1),2) end_proc :

/*--
O::create -- create order term

O::create(e, p)

e - expression representing order term
p - set of equations 'indet=limit_point'
--*/

O::create:=
proc(x, p)
  local v;
begin
  v:= map(p, op, 1);
  if select(map(p, op, 2), not iszero) = {} then
    x:= O::basicSimplify(x, v);
  else
    // shift vars to 0, simplify and shift back
    x:= O::basicSimplify
    (subs(x, [ op(map(p,
                      proc()
                      begin
                        subsop(args(1),2=op(args(1),1)+op(args(1),2))
                      end_proc
                      )) ]), v);
    x:=
    subs(x, [ op(map(p,
                     proc()
                     begin
                       subsop(args(1),2=op(args(1),1)-op(args(1),2))
                     end_proc
                     )) ]);
  end_if;
  if iszero(x) then
    0
  else
    new(O, x, p)
  end_if
end_proc:

O::basicSimplify:=
proc(x, v)
  local p, q, i, j;
begin
  if iszero(x) then
    return(0)
  end_if;
  i:= O::getIndets(x);
  if i minus v <> {} then
    x:= subs(x, map([op(i minus v)], _equal, 1));
    return(O::basicSimplify(x, v));
  end_if;
  if i = {} then return(1) end_if;
  p:= poly(x, [op(v)]);
  if p = FAIL then return(x / O::icontent(x)) end_if;
  if has([coeff(p)], v) then return(x / O::icontent(x)) end_if;
  if ldegree(p) = 0 then return(1) end_if;
  if nops(op(p,2)) = 1 then return(op(p,[2,1])^ldegree(p)) end_if;
    // remove any term which may be divided by another one 
  x:= [ nthmonomial(p,i) $ i=1..nterms(p) ];
  q:= { op(x) };
  for i in x do
    for j in q do
      if i <> j and divide(i, j, hold(Exact)) <> FAIL then
        q:= q minus {i};
        break;
      end_if
    end_for
  end_for;
  x:= _plus(op(map(q, expr)));
  x / O::icontent(x)
end_proc:

O::icontent :=
proc(x)
  local i;
begin
  case type(x)
    of "_mult" do
      if type((i:=op(x,nops(x))))=DOM_INT then i else 1 end_if; break;
    of "_plus" do
      igcd(O::icontent(op(x,i)) $ i=1..nops(x)); break;
    of DOM_INT do
      x; break;
    otherwise
      1;
  end_case
end_proc:

/*--
O::_plus -- add order terms

O::_plus(t1, t2,...)

ti - order terms
--*/

O::_plus:=
proc()
  local a, b, s, o, i, v;
begin
  b:= split([args()], O::thisDomain);
  a:= _plus(op(b[2]));
  if a = undefined then return(undefined) end_if;
  if a = complexInfinity then return(complexInfinity) end_if;
  if a::dom = DOM_POLY then a := op(a, 1); end_if;
  b:= b[1];
  if nops(b) = 1 then
    s:= b;
  else
	// collect O-terms with different vars/points 
    s:= [];
    for o in b do
      for i from 1 to nops(s) do
        v:= extop(o,2) union extop(s[i],2);
        if nops(v) = nops(map(v, op, 1)) then
          s[i]:= O::create(_plus(extop(s[i],1), extop(o,1)), v);
          o:= 0;
          break;
        end_if;
      end_for;
      if o <> 0 then s:= append(s, o) end_if;
    end_for;
  end_if;

  if a = 0 then
    if nops(s) = 1 then
      s[1]
    else
      hold(_plus)(op(s))
    end_if
  elif type(a) = "_plus" then  // flat operands of a
    hold(_plus)(op(a), op(s))
  else
    hold(_plus)(a, op(s))
  end_if
end_proc:


/*--
O::_negate -- negate order term

O::_negate(t)

t - order term
--*/

O::_negate:= proc() begin args(1) end_proc :


// _subtract - necessary to prevent O(x^5)-O(x^5) from being zero

O::_subtract:= (x,y) -> x + (-y):

/*--
O::_mult -- multiply order terms

O::_mult(t1, t2,...)

ti - order terms
--*/

O::_mult:= proc()
    local a, b, s, v, o, i;
begin
    b:= split([args()], O::thisDomain);
    a:= _mult(op(b[2]));
    if a = undefined then return(undefined) end_if;
    if a = complexInfinity then return(complexInfinity) end_if;
    if a::dom = DOM_POLY then a := op(a, 1); end_if;
    b:= b[1];
    if nops(b) = 1 then
      s:= b;
    else
    	// collect O-terms with different vars/points 
    	s:= [];
    	for o in b do
    	    for i from 1 to nops(s) do
    		v:= extop(o,2) union extop(s[i],2);
    		if nops(v) = nops(map(v, op, 1)) then
    		    s[i]:= O::create(_mult(extop(s[i],1), extop(o,1)), v);
    		    o:= 0;
    		    break;
    		end_if;
    	    end_for;
    	    if o <> 0 then s:= append(s, o) end_if;
    	end_for;
    end_if;

    if domtype(a) = domtype(infinity) then 
      return((infinity)::dom::_mult(a, op(s)))
    end_if;
    if nops(s) = 1 then
      s:= s[1];
      O::create(a * extop(s,1), extop(s,2));
    else
    	// try to find matching O-term 
    	v:= indets(a) minus Type::ConstantIdents;
    	for i from 1 to nops(s) do
    	    o:= map(extop(s[i],2), op, 1);
    	    if o union v = o then // o contains v 
    		s[i]:= O::create(a * extop(s[i],1), extop(s[i],2));
    		return(hold(_mult)(op(s)));
    	    end_if;
    	end_for;
    	// be fatalistic 
    	s[1]:= O::create(a * extop(s[1],1), extop(s[1],2));
    	hold(_mult)(op(s))
    end_if
end_proc:


/*--
O::intmult -- multiply order term by integer

O::intmult(t, i)

t - order term
i - integer
--*/

O::intmult:= proc() begin if args(2)=0 then 0 else args(1) end_if end_proc :


/*--
O::_invert -- invert order term

O::_invert(t)

t - order term
--*/

O::_invert:= proc() begin O::create(1 / extop(args(1),1), extop(args(1),2)) end_proc :


/*--
O::_power -- power of order term

O::_power(t, p)

t - order term
p - power
--*/

O::_power:= proc(x,y) local v; begin
    if domtype(x) = O then
        if domtype(y) = O then
	    v:= extop(x,2) union extop(y,2);
	    if nops(v) = nops(map(v, op, 1)) then
		O::create(extop(x,1) ^ extop(y,1), v)
	    else
		hold(_power)(args())
	    end_if
        else
	    O::create(extop(x,1) ^ y, extop(x,2))
        end_if
    else
	O::create(x ^ extop(y,1), extop(y,2))
    end_if
end_proc:


/*--
O::simplify -- simplify order term

O::simplify(o)

o - order term
--*/

O::simplify:=
proc(xx)
  local x, p, i, j, c, di, k, v;
begin
  v:=extop(xx,2);
  x:=extop(xx,1);
    // removes any term in the complex-hull of the other ones 
  if nops(v)>=2 then
    p:= poly(x, map([op(v)], op, 1));
    if p <> FAIL then
      x:= [ nthmonomial(p,i) $ i=1..nterms(p) ];
      v:= { op(x) };
	   /* the following line is needed because linopt::feasible does not
	     like indexed identifiers like c[1], c[2], ... */
      for i from 1 to nops(x)-1 do c[i]:=genident("c") end_for;
      for i in x do
        p:=v minus {i};
        if nops(p)<2 then break end_if;
        di:=degreevec(i); p:=map(p,degreevec);
        p:={di[j]>=_plus(c[k]*op(p,k)[j]$k=1..nops(p))
            $ j=1..nops(di)};
        j:=_plus(c[k]$k=1..nops(p));
        p:=linopt::feasible(p union {j>=1},NonNegative);
        if p then v:= v minus {i} end_if
      end_for;
      x:= _plus(op(map(v, expr)));
    end_if;
  end_if;
  new(O,x,extop(args(1),2))
end_proc:


/*--
O::thisDomain -- test if argument is from domain O

O::thisDomain(x)

x - expression
--*/

O::thisDomain:= proc() begin bool(domtype(args(1)) = O) end_proc :


/*--
O::subs -- substitution

O::subs(x,eq...)
--*/

O::subs:= proc()
  local p;
begin
  p := map(eval(subs(extop(args(1),2),args(2..args(0)))), O::_subs);
  /* if testargs() = FALSE, then O::new does not do argument checking,
     so we must do it here */
  O::checkargs(p);
  O::new(eval(subs(extop(args(1),1), args(2..args(0)))), op(p))
end_proc:

/*--
O::_subs - transforms an equation of the form x-b=a, with x an identifier,
           into the form x=a+b. Used by O::subs to make
           subs(O(x), x=x-2) work correctly. Very crude hack.
--*/

O::_subs :=
proc(eq)
  local x;
begin
  if type(op(eq, 1)) = "_plus" then
    x := select(op(op(eq, 1)), testtype, DOM_IDENT);
    if x <> 0 then
      return(op(x, 1) = op(eq, 2) - op(eq, 1) + op(x, 1))
    end_if
  end_if;
  eq
end_proc:

O::testtype:=
proc(x, T)
begin
if T = Type::Arithmetical then
   TRUE
else
   FAIL
end_if
end_proc:


/*--
O::diff -- differentiation

O::diff(f,x...)
--*/

O::diff:= proc() local f; begin
  f := diff(extop(args(1), 1), args(2..args(0)));
  if iszero(f) then
    // diff(O(1), x) = O(1), diff(O(y), x) = O(1)
    f := 1
  end_if:
  O::create(f, extop(args(1), 2))
end_proc:

/*--
O::has -- tests for the existence of a given subexpression

O::has(f,x)
--*/       
                                     
O::has:=()->has([extop(args(1),1),extop(args(1),2)],args(2..args(0))):

// end of file 
