/*--
simplify::trighyp -- the function attribute for simplifying expressions 
with several calls of trigonometric/hyperbolic functions and exp, 
rewriting them to some set of 'basic' function calls.

We set 
   simplify::sin = simplify::cos = simplify::tan = simplify::cot 
 = simplify::sinh= simplify::cosh= simplify::tanh= simplify::coth 
 = simplify::trighyp.

(Note that there is a separate simplify::exp.)

Strategy:
?0)  rationalize w.r.t. trig/hyp/exp expressions
?    (note that cos(x)*sin(x)*f(sin(x)) with some
?     irrational function f won't combine f(sin(x)) 
?     with any of the other trigs). On the other
?     hand, sqrt(tan(x)) - sqrt(sin(x)/cos(x)) should
?     be simplified to 0 (via simplify::trighyp)
1a) rewrite tan/cot to sin/cos (if 'reasonable')
1b) rewrite tanh/coth to sinh/cosh (if 'reasonable')
2   rewrite sinh/cosh/tanh/coth to sin/cos/tan/cot (if 'reasonable')
3a) rewrite sin/cos/tan/cot to exp (if 'reasonable')
3b) rewrite sinh/cosh/tanh/coth to exp (if 'reasonable')
4)  find arguments that differ by an integer factor and apply addition rules
5a) rewrite powers of cos to sin (if possible)
5b) rewrite powers of cosh to sinh (if possible)
6a) if cos survives 5a), rewrite powers of sin to cos (if possible)
6b) if cosh survives 5b), rewrite powers of sinh to cosh (if possible)
7a) combine sin/cos/tan/cot
7b) combine sinh/cosh/tanh/coth
?8)  undo the rationalization 0)
9)  find and return the intermediate expression with smallest
    Simplify::defaultValuation

Author: W. Oevel
--*/

//======================================
// Note that there are no targets 
// tan/cot/tanh/coth in the usual combine
// function.
// Here is a utility to do combine(f, target) 
// with the missing targets tan/cot/tanh/coth.
//======================================
simplify::combine:= proc(f, target)
local f1, f2;
option remember;
begin
  if target = tan or target = cot then
     f:= subs(f, [hold(tan) = (x -> sin(x)/cos(x)),
                  hold(cot) = (x -> cos(x)/sin(x))],
                  EvalChanges):
     f:= normal(f, Expand = TRUE);
     f1:= combine(numer(f), sincos):
     f2:= combine(denom(f), sincos):
     f:= normal(f1/f2, Expand = TRUE); 
     if target = tan then 
       f:= subs(f, hold(sin) = (x -> cos(x)*tan(x)), EvalChanges);
     elif target = cot then 
       f:= subs(f, hold(sin) = (x -> cos(x)/cot(x)), EvalChanges);
     end_if;
     return(normal(f, Expand = TRUE)):
  elif target = tanh or target = coth then
     f:= subs(f, [hold(tanh) = (x -> sinh(x)/cosh(x)),
                       hold(coth) = (x -> cosh(x)/sinh(x))],
                  EvalChanges):
     f:= normal(f, Expand = TRUE);
     f1:= combine(numer(f), sinhcosh):
     f2:= combine(denom(f), sinhcosh):
     f:= normal(f1/f2, Expand = TRUE); 
     if target = tanh then 
       f:= subs(f, hold(sinh) = (x -> cosh(x)*tanh(x)), EvalChanges);
     elif target = coth then 
       f:= subs(f, hold(sinh) = (x -> cosh(x)/coth(x)), EvalChanges);
     end_if;
     return(normal(f, Expand = TRUE)):
  else
     return(f);
  end_if;
end_proc:


//=================================
// simplify::trighyp
//=================================
/* One observes that there are several calls to simp::trighyp
   with nearly the same arguments, differing only by a constant
   factor. This is a wrapper routine that eliminates constants 
   before passing them to simplify::trighyp_internal which 
   should use option remember.
   We found that there is no measurable benefit from this,
   so we do not use this wrapper.

simplify::trighyp:= proc(a)
local b, dummy;
begin
  if type(a) = "_mult" then
     [a, b, dummy]:= split(a, 
              b -> _lazy_or(has(b, sin),
                            has(b, cos) or
                            has(b, tan) or
                            has(b, cot) or
                            has(b, sinh) or
                            has(b, cosh) or
                            has(b, tanh) or
                            has(b, coth) or
                            has(b, exp) )):
     return(b*simplify::trighyp_intern(a));
  end_if:
  return(simplify::trighyp_intern(a));
end_proc:
*/

simplify::trighyp:= proc(a)
local trigargs, hypargs, expargs,
      an, b, bb, c, x, y, n, XXX,
      siny, cosy, tany, coty, 
      substeqs, donormal, 
      lan, la, lb, lc;
option remember;
begin
  // ==================================
  // fast exit for the cases for which
  // this routine should waste no time:
  // ==================================
  if not testtype(a, Type::Arithmetical) then
     return(a);
  elif not has(a, [sin, cos, tan, cot, sinh, cosh, tanh, coth]) then
     // exclude exp here, because simplify::exp should 
     // handle exp without any further trigs or hyps
     return(a);
  elif contains({"sin", "cos", "tan", "cot",
               "sinh", "cosh", "tanh", "coth",
               "exp"}, type(a)) then
     // sin(some_thing) is handled by sin::simplify,
     // not by this routine!
     return(a);
  elif not contains({"_plus", "_mult", "_power"}, type(a)) then
     // the infrastructure of 'simplify' will automatically
     // apply simplify::trighyp to subtrees of the expression
     return(normal(a, Expand = FALSE));
  elif type(a) = "_power" and 
     not domtype(op(a, 2)) = DOM_INT then
     return(a);
  end_if:
  if type(a) = "_mult" then
     [a, b, c]:= split(a, has, [sin,cos,tan,cot,sinh,cosh,tanh,coth,exp]):
     if not iszero(b - 1) then
        return(b*simplify::trighyp(a));
     end_if;
  end_if:

  an:= normal(a, Expand = TRUE):
  b:= an;

  trigargs:= {}: // container for collecting the arguments of all trigonometric calls
  if has(b, [sin, cos, tan, cot]) then
     misc::maprec(b, {"sin", "cos", "tan", "cot"} = 
                  (x -> (trigargs:= trigargs union {op(x)}: x)));
  end_if:

  hypargs:= {}: // container for collecting the arguments of all hyperbolic calls
  if has(b, [sinh, cosh, tanh, coth]) then
     misc::maprec(b, {"sinh", "cosh", "tanh", "coth"} = 
                  (x -> (hypargs:= hypargs union {op(x)}: x)));
  end_if:

  expargs:= {}: // container for collecting the arguments of all exp calls
  if has(b, exp) then
     misc::maprec(b, {"exp"} = 
                  (x -> (expargs:= expargs union {op(x)}: x)));
  end_if:

  //===============================================
  // reduce the number of different trig functions:
  //===============================================
  if has(b, cot) then // rewrite cot by cos/sin or by tan
     if has(b, sin) or has(b, cos) then // rewrite cot by cos/sin 
        b:= subs(b, hold(cot) = (x -> cos(x)/sin(x)), EvalChanges);
     elif has(b, tan) then // rewrite cot by 1/tan
        b:= subs(b, hold(cot) = (x -> 1/tan(x)), EvalChanges);
     end_if:
  end_if:
  if has(b, tan) then // rewrite tan by sin/cos 
     if has(b, sin) or has(b, cos) then
        b:= subs(b, hold(tan) = (x -> sin(x)/cos(x)), EvalChanges);
     end_if:
  end_if:

  //=====================================================
  // reduce the number of different hyperbolic functions:
  //=====================================================
  if has(b, coth) then // rewrite coth by cosh/sinh or by tanh
     if has(b, sinh) or has(b, cosh) then // rewrite coth by cosh/sinh 
        b:= subs(b, hold(coth) = (x -> cosh(x)/sinh(x)), EvalChanges);
     elif has(b, tanh) then // rewrite coth by 1/tanh
        b:= subs(b, hold(coth) = (x -> 1/tanh(x)), EvalChanges);
     end_if:
  end_if:
  if has(b, tanh) then // rewrite tanh by sinh/cosh 
     if has(b, sinh) or has(b, cosh) then
        b:= subs(b, hold(tanh) = (x -> sinh(x)/cosh(x)), EvalChanges);
     end_if:
  end_if:

  //========================================================
  // rewrite hyperbolic functions by trigonometric functions
  // if there are further trigonometric functions calls with
  // corresponding arguments
  //========================================================
  donormal:= FALSE:
  for x in trigargs do 
   for y in hypargs do 
     if iszero(x) or iszero(y) then next; end_if;
     if domtype(x/(y*I)) = DOM_INT or
        domtype((y*I)/x) = DOM_INT then
        if has(b, sin) or has(b, cos) then 
          siny:= simplify(sin(y*I)); // simplify to get rid of minus signs etc.
          cosy:= simplify(cos(y*I)); // simplify to get rid of minus signs etc.
          // Beware: siny or cosy may have been simplified to 0:
          if iszero(siny) then
             b:= subs(b, [sinh(y) = 0, 
                          cosh(y) =  cosy, 
                          tanh(y) = 0], 
                      EvalChanges);
          elif iszero(cosy) then
             b:= subs(b, [sinh(y) = -I*siny,
                          cosh(y) =  0,
                          coth(y) =  0],
                      EvalChanges);
          else 
             b:= subs(b, [sinh(y) = -I*siny,
                          cosh(y) =    cosy,
                          tanh(y) = -I*siny/cosy,
                          coth(y) =  I*cosy/siny],
                      EvalChanges);
          end_if:
          trigargs:= trigargs union {y*I};
          hypargs:= hypargs minus {y};
          donormal:= TRUE;
        elif has(b, tan) and 
            (has(b, tanh(y)) or
             has(b, coth(y))) then 
          tany:= simplify(tan(y*I)); // simplify to get rid of minus signs etc.
          if iszero(tany) then
             b:= subs(b, [tanh(y) = 0], EvalChanges);
          else 
             b:= subs(b, [tanh(y) = -I*tany,
                          coth(y) =  I/tany],
                      EvalChanges);
          end_if;
          trigargs:= trigargs union {y*I};
          hypargs:= hypargs minus {y};
          donormal:= TRUE;
        elif has(b, cot) and
            (has(b, tanh(y)) or
             has(b, coth(y))) then 
          coty:= simplify(cot(y*I)); // simplify to get rid of minus signs etc.
          if iszero(coty) then
             b:= subs(b, [coth(y) =  I*coty], EvalChanges);
          else
             b:= subs(b, [tanh(y) = -I/coty,
                          coth(y) =  I*coty], 
                      EvalChanges);
          end_if;
          trigargs:= trigargs union {y*I};
          hypargs:= hypargs minus {y};
          donormal:= TRUE;
        end_if:
     end_if;
   end_for;
  end_for;
  if donormal then
    b:= normal(b, Expand = TRUE);
  end_if:

  //=================================================
  // rewrite trigonometric functions by exp, if there 
  // are exp calls with corresponding arguments
  //=================================================
  donormal:= FALSE:
  for x in trigargs do 
     if nops(select(expargs, 
                    y -> domtype(x/(I*y)) = DOM_INT or
                         domtype((I*y)/x) = DOM_INT)) > 0 then
        b:= subs(b, [sin(x) =-I/2*exp(x*I) + I/2*exp(-x*I),
                     cos(x) = exp(x*I)/2 + exp(-x*I)/2,
                     tan(x) =-I*(exp(x*I)-exp(-x*I))/(exp(x*I)+exp(-x*I)),
                     cot(x) = I*(exp(x*I)+exp(-x*I))/(exp(x*I)-exp(-x*I))
                         ],
                 EvalChanges);
        trigargs:= trigargs minus {x};
        expargs:= expargs union {x*I,-x*I};
        donormal:= TRUE;
     end_if;
  end_for;
  if donormal then
    b:= normal(b, Expand = TRUE);
  end_if:

  //==============================================
  // rewrite hyperbolic functions by exp, if there 
  // are exp calls with corresponding arguments
  //==============================================
  donormal:= FALSE:
  for x in hypargs do 
     if nops(select(expargs, 
                    y -> domtype(x/y) = DOM_INT or
                         domtype(y/x) = DOM_INT)) > 0 then
        b:= subs(b, [sinh(x) = exp(x)/2 - exp(-x)/2,
                     cosh(x) = exp(x)/2 + exp(-x)/2,
                     tanh(x) = (exp(x)-exp(-x))/(exp(x)+exp(-x)),
                     coth(x) = (exp(x)+exp(-x))/(exp(x)-exp(-x))
                         ],
                 EvalChanges);
        hypargs:= hypargs minus {x};
        expargs:= expargs union {x,-x};
        donormal:= TRUE;
     end_if;
  end_for;
  if donormal then
    b:= normal(b, Expand = TRUE);
  end_if:


  //========================================================
  // apply addition theorems for the trigonometric functions
  //========================================================
  XXX:= genident("x"):
  donormal:= FALSE:
  for x in trigargs do
   for y in trigargs do
     if x = y then next end_if;
     n:= y/x;
     if domtype(n) = DOM_INT and n > 1 and n <= 100 then
        substeqs:= null():
        if has(b, sin(y)) then
          substeqs:= substeqs, sin(y) = expand(sin(n*XXX));
        end_if: 
        if has(b, cos(y)) then
          substeqs:= substeqs, cos(y) = expand(cos(n*XXX));
        end_if:
        if has(b, tan(y)) then
          substeqs:= substeqs, tan(y) = expand(tan(n*XXX));
        end_if:
        if has(b, cot(y)) then
          substeqs:= substeqs, cot(y) = expand(cot(n*XXX));
        end_if:
        b:= subs(b, [sin(x) = sin(XXX),
                     cos(x) = cos(XXX),
                     tan(x) = tan(XXX),
                     cot(x) = cot(XXX),
                     substeqs  // sin(y) = expand(sin(n*XXX)) etc
                    ]);
        // do *not* evaluate the above when x is replaced by XXX: if it contains 
        // a symbolic int w.r.t. x and only trig(x) is replaced by trig(XXX), 
        // int(f(x)*trig(XXX), x) may suddenly compute a (wrong) explicit result 
        // involving x and XXX! Substitute back before doing an eval:
        b:= subs(b, XXX = x, EvalChanges):
        donormal:= TRUE;
     end_if:
   end_for:
  end_for:
  if donormal then
    b:= normal(b, Expand = TRUE);
  end_if:

  //========================================================
  // apply addition theorems for the hyperbolic functions
  //========================================================
  donormal:= FALSE:
  for x in hypargs do
   for y in hypargs do
     if x = y then next end_if;
     n:= y/x;
     if domtype(n) = DOM_INT and n > 1 and n <= 100 then
        substeqs:= null():
        if has(b, sinh(y)) then
          substeqs:= substeqs, sinh(y) = expand(sinh(n*XXX));
        end_if: 
        if has(b, cosh(y)) then
          substeqs:= substeqs, cosh(y) = expand(cosh(n*XXX));
        end_if:
        if has(b, tanh(y)) then
          substeqs:= substeqs, tanh(y) = expand(tanh(n*XXX));
        end_if:
        if has(b, coth(y)) then
          substeqs:= substeqs, coth(y) = expand(coth(n*XXX));
        end_if:
        b:= subs(b, [sinh(x) = sinh(XXX),
                     cosh(x) = cosh(XXX),
                     tanh(x) = tanh(XXX),
                     coth(x) = coth(XXX),
                     substeqs  // sinh(y) = expand(sinh(n*XXX)) etc
                    ]);
        // do *not* evaluate the above when x is replaced by XXX: if it contains 
        // a symbolic int w.r.t. x and only sinh(x) etc. is replaced by sinh(XXX), 
        // int(f(x)*sinh(XXX), x) may suddenly compute a (wrong) explicit result 
        // involving x and XXX! Substitute back before doing an eval:
        b:= subs(b, XXX = x, EvalChanges):
        donormal:= TRUE;
     end_if:
   end_for:
  end_for:
  if donormal then
    b:= normal(b, Expand = TRUE);
  end_if:


  bb:= b: // remember this form of b for a potential later rewrite sin -> cos
  //========================================================
  // Take care of sin(x)^2 + cos(x)^2 = 1 by rewriting
  // cos(x)^n -> (1-sin(x)^2)^(n/2) for even n and
  // cos(x)^n -> (1-sin(x)^2)^((n-1)/2)*cos(x) for odd n
  //========================================================
  if has(b, sin) and has(b, cos) then // rewrite cos -> sin
     b:=misc::maprec(b,{"_power"} = 
          proc(x) begin 
            if type(op(x,1))="cos" then
               if testtype(op(x,2),Type::Even) then
                 return((1-sin(op(x,[1,1]))^2)^(op(x,2)/2))
               elif testtype(op(x,2),Type::Odd) then
                 if op(x, 2) >= 1 then 
                   return((1-sin(op(x,[1,1]))^2)^((op(x,2)-1)/2)*cos(op(x, [1, 1])));
                 else
                   return((1-sin(op(x,[1,1]))^2)^((op(x,2)+1)/2)/cos(op(x, [1, 1])));
                 end_if;
               end_if;
            end_if;
            return(x)
         end_proc);
     b:= normal(b, Expand = TRUE):
  end_if;
  //========================================================
  // Take care of cosh(x)^2 - sinh(x)^2 = 1 by rewriting
  // cosh(x)^n -> (1+sin(x)^2)^(n/2) for even n and
  // cosh(x)^n -> (1+sin(x)^2)^((n-1)/2)*cosh(x) for odd n
  //========================================================
  if has(b, sinh) and has(b, cosh) then // rewrite cosh -> sinh
     b:=misc::maprec(b,{"_power"} = 
         proc(x) begin 
            if type(op(x,1))="cosh" then
               if testtype(op(x,2),Type::Even) then
                 return((1+sinh(op(x,[1,1]))^2)^(op(x,2)/2))
               elif testtype(op(x,2),Type::Odd) then
                 if op(x, 2) >= 1 then 
                   return((1+sinh(op(x,[1,1]))^2)^((op(x,2)-1)/2)*cosh(op(x, [1, 1])));
                 else
                   return((1+sinh(op(x,[1,1]))^2)^((op(x,2)+1)/2)/cosh(op(x, [1, 1])));
                 end_if;
               end_if;
            end_if;
            return(x)
         end_proc);
     b:= normal(b, Expand = TRUE):
  end_if;

  //========================================================
  // Before, sin(x)^2*cos(x)^3 was rewritten into 
  // sin(x)^2*(1 - sin(x)^2)*cos(x).
  // If some cos term survived, try rewriting sin -> cos.
  // However, do not do this on the last b but on the expression
  // bb above to avoid a desastrous  
  //    cos(x)^(2*n) 
  //      -> expand((1 - sin(x)^2)^n)
  //       -> expand(subs(%, sin(x)^2 = 1 - cos(x)^2))
  //========================================================
  if has(b, cos) and has(b, sin) then // rewrite sin -> cos
      b:=misc::maprec(bb,{"_power"} = 
          proc(x) begin 
            if type(op(x,1))="sin" then
               if testtype(op(x,2),Type::Even) then
                 return((1-cos(op(x,[1,1]))^2)^(op(x,2)/2))
               elif testtype(op(x,2),Type::Odd) then
                 if op(x, 2) >= 1 then 
                   return((1-cos(op(x,[1,1]))^2)^((op(x,2)-1)/2)*sin(op(x, [1, 1])));
                 else
                   return((1-cos(op(x,[1,1]))^2)^((op(x,2)+1)/2)/sin(op(x, [1, 1])));
                 end_if;
               end_if;
            end_if;
            return(x)
          end_proc);
      b:= normal(b, Expand = TRUE):
      bb:= b; // bb will be called again for rewriting sinh -> cosh
  end_if;

  //========================================================
  // As above, now for the hyperbolic functions
  //========================================================
  if has(b, cosh) and has(b, sinh) then // rewrite sinh -> cosh
     b:=misc::maprec(bb,{"_power"} = 
        proc(x) begin 
          if type(op(x,1))="sinh" then
             if testtype(op(x,2),Type::Even) then
                return((cosh(op(x,[1,1]))^2 - 1)^(op(x,2)/2))
             elif testtype(op(x,2),Type::Odd) then
                if op(x, 2) >= 1 then 
                   return((cosh(op(x,[1,1]))^2-1)^((op(x,2)-1)/2)*sinh(op(x, [1, 1])));
                else
                   return((cosh(op(x,[1,1]))^2-1)^((op(x,2)+1)/2)/sinh(op(x, [1, 1])));
                end_if;
             end_if;
          end_if;
          return(x)
        end_proc);
      b:= normal(b, Expand = TRUE):
  end_if;

  //=================================================
  // Combine powers of trigonometric and hyperbolic 
  // functions to 'simpler' calls with 'higher' 
  // arguments
  //=================================================
  if has(b, sin) or 
     has(b, cos) then
     c:= combine(b, hold(sincos));
  elif has(b, cot) and not has(b, tan) then
     c:= simplify::combine(b, cot)
  elif has(b, tan) then
     c:= simplify::combine(b, tan);
  else
     c:= b
  end_if:

  if has(c, sinh) or 
     has(c, cosh) then
     c:= combine(c, hold(sinhcosh));
  elif has(c, coth) and not has(c, tanh) then
     c:= simplify::combine(b, coth)
  elif has(c, tanh) then
     c:= simplify::combine(c, tanh);
  //else
  //   c:= c
  end_if:

  if has(c, exp) then
     c:= combine(c, exp);
  end_if:

  //=================================================
  // Finally, return the simplest of the various
  // intermediate results
  //=================================================
  lan:= Simplify::defaultValuation(an);
  la:= Simplify::defaultValuation(a);
  lb:= Simplify::defaultValuation(b);
  lc:= Simplify::defaultValuation(c);

  //=============================================
  // debugging facility for the design phase
  // if args(0) = 2 and args(2) = "Test" then
  //   print([an, lan], [a, la], [b, lb], [c, lc]):
  // end_if:
  //=============================================

  if lc = min(lan, la, lb, lc) then
     return(c)
  elif lb = min(lan, la, lb, lc) then
     return(b)
  elif lan = min(lan, la, lb, lc) then
     return(an)
  else
     return(a)
  end_if:
end_proc:
