/*--
	_mult/simplify -- the function attribut "simplify" for _mult

      For products, we use an expanding normal form: (2*a)^n ->^2^n * a^n
      But: we do combine exponents belonging to the same base: x^a * x^b -> x^(a+b)

   
      However, powers of integers and rationals are combined where possible: 2^n * 3^n -> 6^n 


--*/

_mult::simplify:=
proc(e, options = simplify::defaultOptions: DOM_TABLE)
  name _mult::simplify;
  local t, s, opt,
  combineIntPowers: DOM_PROC,
  combineExponents: DOM_PROC,
  combineConstantWithRoots: DOM_PROC,
  combineSin: DOM_PROC;
begin

  // local simplification methods
  
  combineIntPowers:=
  proc(e)
    local t, T, i, expo, bas, minusOnes, equ;
  begin  
    t:= split(e, fctor -> type(fctor) = "_power" and
              type(op(fctor, 1)) = DOM_INT);
    // there should be no "unknown"
    assert(t[3] = 1);
    
    if type(t[1]) = "_mult" then 
      // combine m^z * n^z -> (m*n)^z 
      // (-m)^z -> (-1)^z * m^z 
      // however, do not make m^(2*z) into (m^2)^z
      T:= table();
      minusOnes := 1;
      for i from 1 to nops(t[1]) do 
      	if type(op(t, [1, i])) <> "_power" then 
          // it turns out that this can happen due to evaluation
          minusOnes:= minusOnes * op(t, [1, i]);
          next 
        end_if;
        [bas, expo]:= [op(op(t, [1, i]))];
        if bas < 0 then
          // make (-n)^x into (-1)^x * n^x
          minusOnes:= minusOnes * (-1)^expo;
          bas:= -bas
        end_if;  
        if contains(T, expo) then 
          T[expo]:= T[expo] * bas
        else
          T[expo]:= bas
        end_if
      end_for;  
          
      t[1]:= _mult(op(equ, 2)^op(equ, 1) $equ in T) * minusOnes  
    end_if;  
    
    if type(t[1]) = "_mult" then
      t[1]:= combine::combineExponents(t[1], options)
    end_if;
    t:= t[1]*t[2]*t[3];
    // combine powers with non-rational exponent and the constant
    // (such as 4*2^n -> 2^(n+2))
    if type(t) <> "_mult" then
      return(t)
    end_if;
    
    t:= split(t, proc(fctor) 
                 begin 
	                 type(fctor) = "_power" and
  	               type(op(fctor, 1)) = DOM_INT and 
    	             type(op(fctor, 2)) <> DOM_RAT
      	           or type(fctor) = DOM_INT
                 end_proc
                 );
    if type(t[1]) = "_mult" then
      t[1]:= combine::combineExponents(t[1], options)
    end_if;  
    t[1]*t[2]*t[3]
    
  end_proc;


  combineExponents:=
  proc(e)
    local t: DOM_TABLE, a, sgn: DOM_INT, reduceMod2, addToTable: DOM_PROC;
  begin
 // rewrite x^a*x^b --> x^(a+b)
 
  // addToTable(a) - adds the factor a to the table of bases
  // entries are of the form base = exponent
  addToTable:= 
  proc(a)
    local bas, expo, u;
  begin
     if type(a) = "_power" then
        bas:= op(a, 1);
        expo:= op(a, 2)
      elif type(a) = DOM_RAT then
        addToTable(op(a, 1));
        addToTable(hold(_power)(op(a, 2), -1));
        return()
      else
        bas:= a;
        expo:= 1;
      end_if;
      // special case
      if bas = I then
        bas:= -1;
        expo:= expo/2
      elif bas = -I then
        bas:= -1;
        expo:= -expo/2
      end_if;
      
      if type(bas) = DOM_INT and ((u:= numlib::ispower(bas))) <> FALSE then 
        bas:= op(u, 1);
        expo:= expo*op(u,2)
      end_if;  
      
      if contains(t, bas) then
        t[bas]:= t[bas] + expo
      elif not iszero(bas) and contains(t, 1/bas) and type(expo) = DOM_INT then
        t[1/bas]:= t[1/bas] - expo
      else
        t[bas]:= expo
      end_if;
   end_proc;
 

    sgn:= 1;
    
    t:= table();
    for a in e do
      addToTable(a)
    end_for;  
     

    if contains(t, -1) then
  // symmetric reduction modulo 2
      reduceMod2:=
      proc(b)
        local p, q;
      begin
        case type(b)
          of DOM_INT do
            return(b mod 2)
          of DOM_RAT do
            return(frac(1/2 + b/2)*2 - 1)
          of DOM_COMPLEX do
            p:= op(b, 1);
            q:= op(b, 2);
            // reduce p modulo 2
            return(reduceMod2(op(b, 1)) + op(b, 2)*I)
          of "_plus" do
            return(map(b, reduceMod2))
        end_case;
        if is(b, Type::Integer)=TRUE and stdlib::hasmsign(b) then
          -b
        else
          b
        end_if
      end_proc;

      t[-1] := reduceMod2(t[-1]);
      if contains({DOM_INT, DOM_RAT, DOM_FLOAT}, domtype(t[-1])) and t[-1] < 0 then
        t[-1]:= t[-1] + 1;
        sgn:= -sgn
      end_if
    end_if;
  
    sgn*_mult(op(a, 1)^op(a, 2) $a in t)
  end_proc;



  combineConstantWithRoots:=
  proc(e: "_mult")
    local constant, num, den, expo, base, d,
    other: DOM_LIST,
    i: DOM_INT;
  begin

    constant:= op(e, nops(e));
    other:= [op(e, 1..nops(e) - 1)];
    
    case domtype(constant)
      of DOM_INT do
        den:= 1;
        num:= constant;
        break
      of DOM_RAT do
        [num, den]:= [op(constant)];
        break
      of DOM_COMPLEX do
        if map({op(constant)}, type) minus {DOM_INT, DOM_RAT} = {} then
          // extract common factors of real and imaginary part
          // we compute gcd(op(e,1), op(e, 2)) "by hand", avoiding a gcd call
          num:= igcd(numer(op(constant, 1)), numer(op(constant, 2)));
          if op(constant, 1) = 0 then
            den:= denom(op(constant, 2))
          else        
            den:= ilcm(denom(op(constant, 1)), denom(op(constant, 2)))
          end_if;
          other:= append(other, constant/ (num/den));
          constant:= num/den;
          break
        else
          return(e)
        end_if
      otherwise
        return(e)
    end_case;


  
    for i from 1 to nops(other) do
      if type(other[i]) = "_power" and
        type((expo:= op(other[i], 2))) = DOM_RAT then

        case domtype((base:= op(other[i], 1)))
          of DOM_INT do

           /*
            // let den = d*a, base = d*b; then
            // (num/den) * base^(expo) = (num/a) * b^expo * d^(expo - 1)
            // = constant * d * (base/d)^expo * d^(expo - 1) 
                       d:= igcd(base, den);
            if d<>1 then
              constant:= constant * d;
              den:= den/d;
              other[i]:= subsop(other[i], 1 = base/d);
              other:= append(other, d^(expo-1));
              /* since num and den cannot have a common divisor,
                 we may stop here */
              break
            end_if;
           */
           
            // let num = d*a, base = d*b; then
            // (num/den) * base^(expo) = (a/den) * b^expo * d^(expo + 1)
            // = constant / d * (base/d)^expo * d^(expo + 1)
            d:= igcd(base, num);
            if d<>1 then
              constant:= constant / d;
              num:= num/d;
              other[i]:= subsop(other[i], 1 = base/d);
              other:= append(other, d^(expo + 1))
              end_if;
            break
        end_case;

        
      end_if;
        
    end_for;

    return(_mult(op(other)) * constant )
    

  end_proc;

  // The following identity holds for all n = 1, 2, .. :
  //
  //   _mult(sin(k*PI/n) $ k=1..floor(n/2)) = sqrt(n/2^(n-1)))
  //
  combineSin:=
  proc(e: "_mult")
    local s, Args, n, k;
  begin
    s:= split(e, e -> type(e) = "sin");
    if nops(s[1]) < 3 then
       // The formula is simplified for n = 1,2,..,6 because
       // sin returns explicit values for sin(k*PI/n).
       // The simplest interesting case is n = 7 where the
       // formula involves a product of 3 terms.
       return(e);
    end_if;
    // extract all arguments of the sin calls and
    // store them in the set 'Args':
    Args:= {}:
    misc::maprec(s[1], {"sin"} = 
         proc(s)
         local e;
         begin
           e:= op(s)/PI;
           if domtype(e) = DOM_RAT then
              Args:= Args union {e};
           end_if;
           s;
         end_proc);
     // now: e = c*_mult(sin(a*PI)) with a in Args
     if nops(Args) < 1 then
        return(e);
     end_if;
     n:= op(gcd(op(Args)), 2);
     if domtype(n) <> DOM_INT then
        return(e);
     end_if:
     Args:= map(Args, _mult, n);
     Args:= Args intersect {$ 1 .. floor(n/2)};
     // If there are many terms of the type sin(k*PI/n),
     // replace them using the identity
     if nops(Args) > floor(n/2) - nops(Args) + 1 then
       e:= e*sqrt(n/2^(n-1))/_mult(sin(k*PI/n) $ k = 1.. floor(n/2));
     end_if;
     return(e);
  end_proc;
  
  
  // main program of _mult::simplify

  if MAXEFFORT < 500 or MAXEFFORT < property::complexity(e) then
    return(e)
  end_if;
  
  if not testtype(e, Type::Arithmetical) then
    return(e)
  end_if;
  
  if simplify::expandComplexity(e) < 100000 then  
     opt:= rationalize::defaultOptions;
     opt[ReplaceTypes]:= {"int"};

     [t, s]:= rationalize::replaceTypes(e, opt);  

     t:= normal(t, Expand = TRUE);
     t:= subs(t, s);
  
     if length(t) < length(e) then
       e:= t
     end_if;
  end_if; 

  if type(e) <> "_mult" then
    return(e)
  end_if;

  /* we use an expanding normal form for expressions like 2^n*x^n; thus a call of the form 
  
      e:= _mult::combine(e /*, options */);
      if type(e) <> "_mult" then
        return(e)
      end_if;
 
     is unwanted here
  */  

  t:= simplify::fractionalPowers(e);
  if length(t) < length(e) then
    e:= t
  end_if;

  if type(e) <> "_mult" then
    return(e)
  end_if;

  
  e:= combineIntPowers(e);
  if type(e) <> "_mult" then
    return(e)
  end_if;
  

  e:= combineExponents(e);

  if type(e) <> "_mult" then
    return(e)
  end_if;

  e:= combineConstantWithRoots(e);

  if type(e) <> "_mult" then
    return(e)
  end_if;

  e:= combineSin(e);

  return(e)
  
end_proc:

// end of file 
