//
//   

/****************************************************************

simplify  - the MuPAD simplifier

simplify(e <, f> <, options>)

e - any object
f - identifier or function
options - IgnoreAnalyticConstraints (no other option implemented yet)

If a second argument f is given, just simplify::f is called.


simplify(e) returns e for atomic objects, and maps simplify to e for
non-atomic objects that are not expressions.


For expressions e, simplify calls the following simplification methods

- simplify::f with argument e for every type f appearing in the
  expression (context - sensitive simplifications)
- f::simplify with argument e if e is of type f
- a function that maps simplify to the operands of e

repeatedly, until no further simplifications happen.


simplify uses a double remember mechanism. Firstly, each simplification
method has option remember, where all recursive calls for subexpressions share
the same remember table. Secondly, calls to the same method with the same input
are prevented if they occur within the same recursive call to the main program
of simplify.

****************************************************************/


alias(MAXFACTOR = 2.0):

alias(complexity = Simplify::defaultValuation):

simplify :=
proc(e, f)
  local
  g: DOM_STRING,
  options: DOM_TABLE,
  optiontypes: DOM_TABLE,
  rememberTable: DOM_TABLE,
  checkSlot1: DOM_PROC,
  checkSlot2: DOM_PROC,
  localSimplifications: DOM_PROC,
  simplifyExpr: DOM_PROC,
  units; // check for units in the expression

begin

// local methods

  checkSlot1:=
  proc(e)
    option remember;
    local p, _t, olde;
  begin
    // we do not call slots simplify::f if f occurs inside a non-arithmetical expression
    if testtype(e, Type::Arithmetical) <> TRUE then
      return(e)
    end_if;

    olde:= e;
    for _t in sort([op(map(indets(e, All), expr2text))]) do
      if (p:= slot(simplify, _t)) <> FAIL then
        if options[hold(ShowSteps)] then
           
           fprint(Unquoted, 0, "Memory: ", (bytes()[1] div 10000)/100.0, " MB   Time: ", time(), "\nsimplify::",_t , " ", e)
        end_if;
        e:= p(e, options);
        if e <> olde then
          return(e)
        end_if
      end_if;
    end_for;
    e
  end_proc;

  checkSlot2:=
  proc(e)
    option remember;
    local s, _t;
    begin
      _t := eval(op(e,0));
      if domtype(_t) = DOM_FUNC_ENV and
        (s:= slot(_t,"simplify")) <> FAIL
        then
        if options[hold(ShowSteps)] then
           fprint(Unquoted, 0, "Memory: ", (bytes()[1] div 10000)/100.0, " MB   Time: ", time(), "\n", expr2text(_t)."::simplify ", e)
        end_if;
        s(e, options)
      else
        e
      end_if
  end_proc;

  localSimplifications:=
  proc(e)
    option remember;
  begin
    stdlib::mapEvalChanges(e, simplifyExpr)
  end_proc;

  // main method of simplify
  // note that its rememberTable is local, while the tables
  // created by option remember above are the same for every
  // instance


  units:= {}: // storage for the units in the expression

  simplifyExpr:=
  proc(e)
    option remember;
    local
    _e,
    n: DOM_INT,
    resultsSoFar: DOM_SET,
    f: DOM_PROC;


  begin
    if domtype(e) <> DOM_EXPR then
      if expr2text(domtype(e)) = "unit" then
        units:= units union {e};
      end_if;
      if e::dom::simplify <> FAIL then
        return(e::dom::simplify(e, options))
      end_if;
      return(e)
    end_if;
    
    resultsSoFar:= {e};


      // try some simplification methods, as long as new results emerge
    repeat

      n:= nops(resultsSoFar);

      for f in [checkSlot1, checkSlot2, localSimplifications] do


        if not contains(rememberTable[f], e) then

          rememberTable[f]:= rememberTable[f] union {e};
          _e:= e;

          e:= f(e);
          
          // applying f to e leads to debug nodes in _e
          // e.g. for the call simplify(subs(diff(y(x), x) - y(x), y = exp))
          if nops(_e) > 0 then
            _e := subs(_e, op(_e,1)=op(_e,1));
          end_if;

          if testtype( e,Type::Numeric ) then
            rememberTable[f]:= rememberTable[f] minus {_e};
            return( e )
          end_if;

          if contains(resultsSoFar, e) and _e <> e then
            // we have to resolve a conflict between two simplification
            // methods f and g with f(_e) = e and g(e) = _e .
            // use complexity to do this
            userinfo(5, "Contradicting simplification methods");
            userinfo(5, "Using simpler result");
            if complexity(_e) < complexity(e) then
              e:= _e
            else
              rememberTable[f]:= rememberTable[f] minus {_e}
            end_if;
          else
            if e <> _e then
              if MAXFACTOR * complexity(_e) + 100.0 <=
                complexity(e) then
                e:= _e
              else
                // useful result 
                rememberTable[f]:= rememberTable[f] minus {_e};
                resultsSoFar:= resultsSoFar union {e};
                break
              end_if;
            end_if;
          end_if;

        end_if;

      end_for;

    until n = nops(resultsSoFar) end_repeat;

    if nops(units) > 1 then
       e:= unit::simplify(e);
    end_if;

    e

  end_proc;


  options:= simplify::defaultOptions;

  optiontypes:= table(IgnoreAnalyticConstraints = DOM_BOOL, hold(ShowSteps) = DOM_BOOL);
  

  //////////////////////////////////////////////
  //  m a i n   p r o g r a m  o f  s i m p l i f y
  //////////////////////////////////////////////


  if args(0) = 0 then
    error("simplify called without arguments")
  end_if;


  if args(0) >= 2 then
    if type(f) = DOM_TABLE then
      options:= f
    else 
      g:= slot(simplify, expr2text(f));
      if g <> FAIL then
        options:= prog::getOptions(3, [args()], options, TRUE, optiontypes)[1];
        return( g(e, options))
      else
        options:= prog::getOptions(2, [args()], options, TRUE, optiontypes)[1];
      end_if;
    end_if;
  end_if;

   // is overloaded ?
  if e::dom::simplify<>FAIL then
    return(e::dom::simplify(e, options))
  end_if;
  
  case domtype(e)
    of DOM_INT     do
    of DOM_RAT     do
    of DOM_FLOAT   do
    of DOM_COMPLEX do
    of DOM_HFARRAY do
      return( e )
    of DOM_POLY    do
      return( mapcoeffs(e, simplify, args(2..args(0))))
    of DOM_LIST    do
    of DOM_ARRAY   do
    of DOM_SET     do
    of DOM_TABLE   do
      return( map(e, simplify, args(2..args(0))))
    of DOM_EXPR    do
      rememberTable:= table();
      for f in [checkSlot1, checkSlot2, localSimplifications] do
        rememberTable[f] := {}
      end_for;
      return(simplifyExpr(e))
  end_case;
  e
end_proc:

DOM_NIL::simplify:=proc() begin NIL end_proc :

simplify:=funcenv(simplify):

simplify::defaultOptions:= table(IgnoreAnalyticConstraints = FALSE, hold(ShowSteps) = FALSE):


//---- units -----
simplify::unit:= X -> unit::simplify(X):


//---- trigonometric and  -------
//---- hyperbolic functions -----
simplify:= slot(simplify,"trighyp",
  loadproc(slot(simplify,"trighyp"),pathname("STDLIB","SIMPLIFY"),
        "SIMPLtrig")
):
simplify::sin := simplify::trighyp:
simplify::cos := simplify::trighyp:
simplify::tan := simplify::trighyp:
simplify::cot := simplify::trighyp:
simplify::sinh:= simplify::trighyp:
simplify::cosh:= simplify::trighyp:
simplify::tanh:= simplify::trighyp:
simplify::coth:= simplify::trighyp:

//---- sqrt -----
stdlib::deferredAlias( simplify::sqrt, radsimp ):

//---- exp -----
simplify:=slot(simplify,"exp",
    loadproc(slot(simplify,"exp"),pathname("STDLIB","SIMPLIFY"),
        "SIMPLexp")
):

//---- logarithms -----
simplify:=slot(simplify,"ln",
    loadproc(slot(simplify,"ln"),pathname("STDLIB","SIMPLIFY"),
        "Sln")
):

simplify:=slot(simplify,"log",
    loadproc(slot(simplify,"log"),pathname("STDLIB","SIMPLIFY"),
        "Slog")
):

//----- various special functions ----
simplify:=slot(simplify,"gamma",
    loadproc(slot(simplify,"gamma"),pathname("STDLIB","SIMPLIFY"),
        "SIMPLgamma")
):

simplify:= slot(simplify,"dirac",
    loadproc(slot(simplify,"dirac"),pathname("STDLIB","SIMPLIFY"),
        "SIMPLdirac")
):

simplify:=slot(simplify,"besselJ",
    loadproc(slot(simplify,"besselJ"),pathname("STDLIB","SIMPLIFY"),
        "SIMPLbessel")
):

simplify:=slot(simplify,"besselI",
    loadproc(slot(simplify,"besselI"),pathname("STDLIB","SIMPLIFY"),
        "SIMPLbessel")
):

simplify:=slot(simplify,"besselY",
    loadproc(slot(simplify,"besselY"),pathname("STDLIB","SIMPLIFY"),
        "SIMPLbessel")
):

simplify:=slot(simplify,"besselK",
    loadproc(slot(simplify,"besselK"),pathname("STDLIB","SIMPLIFY"),
        "SIMPLbessel")
):

simplify::conjugate:= loadproc(simplify::conjugate, pathname("STDLIB","SIMPLIFY"), "SIMPLconjugate"):

//---------------------------------

//---------------------------------

//---------------------------------
_power:= slot(_power,"simplify",
    loadproc(slot(_power,"simplify"),pathname("STDLIB","SIMPLIFY"),
        "_power")
):

_mult:= slot(_mult,"simplify",
    loadproc(slot(_mult,"simplify"),pathname("STDLIB","SIMPLIFY"),
        "_mult")
):

simplify::logic :=
loadproc(simplify::logic, pathname("STDLIB","SIMPLIFY", "LOGIC"),"simplify"):


_plus:= slot(_plus,"simplify",
    loadproc(slot(_plus,"simplify"),pathname("STDLIB","SIMPLIFY"),
        "_plus")
):

_fconcat:= slot(_fconcat, "simplify",
    loadproc(slot(_fconcat, "simplify"), pathname("STDLIB","SIMPLIFY"),
        "_fconcat")
):


_minus::simplify:=  loadproc(_minus::simplify, pathname("STDLIB","SIMPLIFY"), "_minus"):



/* STDLIB/SIMPLIFY/int.mu, STDLIB/SIMPLIFY/maxmu, STDLIB/SIMPLIFY/min.mu  and STDLIB/SIMPLIFY/sum.mu
 * are now defined in their function environment */

simplify::simplifySets := loadproc( slot(simplify,"simplifySets" ), pathname("STDLIB", "SIMPLIFY"), "simplifySets"):
simplify::simplifyCondition := loadproc( slot(simplify,"simplifyCondition" ), pathname("STDLIB", "SIMPLIFY"), "simplifyCondition"):
simplify::simplifyLogic := loadproc( slot(simplify,"simplifyLogic" ), pathname("STDLIB", "SIMPLIFY"), "simplifyLogic"):

simplify::expandComplexity:= loadproc(simplify::expandComplexity, pathname("STDLIB", "SIMPLIFY"), "expandComplexity"):
simplify::fractionalPowers:= loadproc(simplify::fractionalPowers, pathname("STDLIB", "SIMPLIFY"), "fractionalPowers"):

simplify::condition:= ()->simplify::simplifyCondition(args()):

simplify::normal := X->normal(X):
simplify::print:= "simplify":
