/* ========================================================
 This is a utility function that accepts arbitrary arithmetical
 expressions that are (multivariate) polynomial expressions
 in one or more dirac calls such as

    f = c0(x) + c1(x)*dirac(c2*x + c3)*(dirac(c4*y - c5) + c6)

 It returns a list [coeffs, reconstruct], where coeffs is a nested
 list of coefficients such that f = poly(coeffs, [diraccalls]).
 The dirac calls are built into 'reconstruct' via option escape.
 The returned procedure 'reconstruct' rebuilds the simplified 
 expression f via 
   simplified_f := reconstruct(simplify(coeffs)). 

 This routine is used by simplify (in simplify::dirac) and
 by Simplify.
========================================================= */

stdlib::simplify_dirac:= proc(f)
local diracCalls, Arg, x, n, inds, i, j, 
      X, x0, c, cc, DD, p, result, multilinear;
option escape;
begin
  if not hastype(f, "dirac") then
      return([f, id]);
  end_if;
  diracCalls:= [op(misc::subExpressions(f, "dirac"))];
  
  for i from 1 to nops(diracCalls) do
    if nops(diracCalls[i]) = 1 then
       [Arg, n]:= [op(diracCalls[i]), 0];
    else
       [Arg, n]:= [op(diracCalls[i])];
    end_if:
    if domtype(n) <> DOM_INT or n < 0 then
       next;
    end_if;

    // Analyze the call dirac(Arg, n). If there is any
    // unknown entering Arg linearly, solve for this
    // unknown (i.e., write Arg = c*(x - x0)) and
    // simplify the term with dirac(Arg, n) -- of the form
    // c(x, x0)* dirac(x - x0, n).
    // All calls use the sysorder of the unknowns
    // to make sure that the same unknown x is expressed
    // by the remaining unknowns x0 etc.
    // Search for an unknown x that enters Arg linearly:
    inds:= DOM_SET::sort(numeric::indets(Arg)):
    x:= NIL; // 'the variable' (as opposed to symbolic parameters)
    x0:= NIL: 
    c:= NIL;
    multilinear:= FALSE;
    for X in inds do
      p:= poly(Arg, [X]);
      if p <> FAIL and degree(p) <= 1 then
        if x <> NIL and X <> x then
          // another candidate for the variable was found before
          multilinear:= TRUE;
        end_if;
        cc:= coeff(p, 1);
        if // domtype(float(cc)) = DOM_FLOAT and 
           not iszero(cc) then
           // X is a candidate for 'the Dirac variable' x.
           // We can extract the numerical coefficient c = cc 
           // without producing a symbolic abs in
           // dirac(c*(x - x0)) = 1/abs(c)*dirac(x - x0)
           x:= X;
           c:= cc:
           x0:= -coeff(p, 0)/c;
        end_if:
      end_if;
    end_for:
    if x0 = NIL or c = NIL then
      // we cannot do anything. Go to the next dirac call.
      next;
    end_if:
    if multilinear and 
       domtype(float(c)) <> DOM_FLOAT then
       next;
    end_if;
    // assert that everything is fine. If something's wrong
    // (i.e., Arg <> c*(x - x0)), then ignore this dirac expression
    if normal(Arg - c*(x - x0)) <> 0 then
      next;
    end_if:
    // Now, rewrite f as a poly in dirac(Arg) and substitute
    // x = x0 in the coefficient of dirac(Arg, n):
    p:= subs(f, dirac(Arg, n) = `#dirac`);
    p:= poly(p, [`#dirac`]);
    // The expression should be a 1st degree poly in dirac (if
    // not, something was passed that is not an acceptable model
    // of anything 'out there'). Still, we need to deal with this
    // case, too:
    if p = FAIL then
       next;
    end_if;
    // assert(degree(p) <= 1);
    if degree(p) > 1 then
       next;
    end_if;
 
    // Stuff like dirac(x - y, n)/(x - y) is not really defined.
    // Shall we return an error/a warning/the original expression?
    // The formula for simplification is:
    //   f(x)*dirac(x - x0, n) =
    //    _plus((-1)^i*binomial(n,i)*(D@@(i))(f)(x0)*dirac(x-x0, n-i) $ i = 0..n)
 
    // Utility to compute DD(i, f) = (D@@i)(f)(x)
    DD:= proc(i, f)
    begin
      f:= diff(f, x $ i);
      if has(f, diff) then
         f:= rewrite(f, D);
      end_if:
      if has(f, diff) then
         error("could not rewrite diff to D");
      end_if;
      return(f);
    end_proc:
 
    if traperror((
       result:= _plus(coeff(p, 0),
                      (-1)^j*binomial(n,j)/abs(c)/c^n
                      *evalAt(DD(j, coeff(p, 1)), x = x0)
                      *dirac(x-x0, n-j) $ j = 0..n);
       )) <> 0 then
       next;
    else
       // it may happen that dirac(Arg, n) has been replaced by #dirac under
       // an integral sign, i.e., within a coefficient
       if has(result, #dirac) then
         next;
       elif (multilinear and n <> 0) then
         // we need to know what 'the variable' is to
         // produce an explicit result. All we can do
         // is: extract the coefficient c:
         f:= coeff(p, 0) + coeff(p, 1)/abs(c)/c^n*dirac(x - x0, n)
       else
         f:= result;
       end_if:
    end_if:
  end_for:

  // return the coefficients of the diracs as a list of list
  // (generated via poly2list) for simplification via simplify
  // or Simplify. Also return a procedure that will be called
  // called with the simplified coefficient list to generate
  // the dirac poly with the simplified coeffs.
  diracCalls:= [op(misc::subExpressions(f, "dirac"))];
  if nops(diracCalls) = 0 then
    return([f, id]):
  end_if;
  // Use traperror: if diracCalls contains constant expressions 
  // such as dirac(0), dirac(0, 1) etc., poly will error 
  // ('Illegal indeterminate').
  if traperror((p:= poly(f, diracCalls))) <> 0 then
     p:= FAIL;
  end_if;
  if has(p, FAIL) then
    return([f, id]):
  end_if:
  return([poly2list(p), 
          proc(List) 
           local y;
          begin 
            y:= expr(poly(List, diracCalls));
            if has(y, FAIL) then
               return(f)
            end_if;
            return(y);
          end_proc]);
end_proc:
