//     

/*++
    factor -- return the factorization of multivariate polynomial in Z or
              factorization of univariate polynomial in Zp

              output is given in form
              [u, f1, e1, ..., fn, en], (p=u*f1^e1*...*fn^en, u is the constant
              content), fi has the same form of input p.

    factor(p<, options>)

    p - an expression or a polynomial

    options: MaxDegree = n (default: 1)
             factor in an algebraic extension given by adjoining those
             coefficients of the input to Q whose degree is at most n
             Adjoin    = list
             factor in the algebraic extension generated by the elements
             of list

++*/

factor:=
proc(p /*, options */)
  local _p_, f, Factors, ff, i, pp, q, t, xx, z, Float, factored,
  expandToRootOf: DOM_PROC,
  factorRC: DOM_PROC,
  round2zero: DOM_PROC,
  options: DOM_TABLE, optiont: DOM_TABLE;
begin

    // local method factorRC written by Oliver
  factorRC :=
  proc(p)
    local a, b, c, contains_, i,
    l, lc, r, r1, r2, u, x, z;
    save DIGITS;
  begin

    // Local procedure contains_

    contains_ :=
    proc(l, x)
      local i, n;
    begin
      i := 1; n := nops(l);
      while i <= n do
        if specfunc::abs(l[i]-x) < 10^(-DIGITS) then
          return(i)
        end;
        i := i + 1
      end;
      return(0)
    end;

    // local procedure round2zero
    round2zero:=
    proc(x, eps = 10^(-DIGITS))
    begin
      if type(x) = DOM_COMPLEX then
        round2zero(op(x,1), eps) +
        I * round2zero(op(x,2), eps)
      else
        if abs(x) < abs(eps*1e-2) then
          0
        else
          x
        end
      end
    end:

    /////////////////////////////
    // Main procedure of factorRC
    ////////////////////////////

    if testtype(( a := poly(float(p)) ), Type::PolyOf(Type::Complex, 1)) <> TRUE then
      return( p )
    end;

    x := numeric::indets(p);
    if nops(x) <> 1 then
      return( p )
    end;
    x := op(x);

    if not testtype(op(a,[2,1]), DOM_IDENT ) then
      p := subs( p, op(a,[2,1])=genident() );
      x := op(a,[2,1]);;
    end_if;

    DIGITS := 2 * DIGITS;
    lc := lcoeff(p); p := p / lc;
    l := numeric::polyroots(p, Factor);
    if options[Domain] = C_ then
      l:= Dom::Multiset(op(l));
      return( [lc,  ((x-z[1]), z[2]) $ z in [extop(l)] ])
    end;
    [r,c,u] := split(l, testtype, DOM_FLOAT);
    r1 := map([extop(map(Dom::Multiset(op(r)), z-> x-z))], op);
    r2 := []: u := []:
    while c <> [] do
      r := c[1]; delete c[1];
      // Note: The system function contains doesn't work.
      //       So we define our own function, namely contains_
      i := contains_(c, conjugate(r));
      if i <> 0 then
        // z:=a+b*I, a in R_ and b in R_
        // (x-z)*(x-conjugate(z)) -> x^2 + (-2*a)*x + (a^2 + b^2)
        // So, -2*a and a^2 + b^2 are real coefficients.
        a := op(r,1); b := op(r,2);
        r2 := r2.[(x^2 -2*a*x + a^2+b^2)];
        delete c[i]
      else
        u := u.[x - r]
      end
    end:
    r2:= map([extop(Dom::Multiset(op(r2)))], op);
    u:= map(u, mapcoeffs, round2zero);
    u:= map([extop(Dom::Multiset(op(u)))], op);
    [lc].r1.r2.u
  end_proc;


  /**************************************
     main program of factor
  **************************************/


  if args(0) = 0 then
    error("factor called without arguments")
  end_if;

  options:= table(MaxDegree = 2,
                  Adjoin = {},
                  Full = FALSE,
                  Domain = Expr);

  if args(0) > 1 then

    if args(2) = R_ or args(2) = C_ then
      options[Domain] := args(2)
    elif type(args(2)) = DOM_TABLE then
      options:= args(2)
    else
      optiont:= table(MaxDegree = Type::PosInt,
                      Adjoin = Type::AnyType,
                      Domain = Type::AnyType,
                      Full   = DOM_BOOL  
                        );

      options:= prog::getOptions(2, [args()], options, TRUE, optiont)[1];
    end_if;

    case options[Domain]
      of R_ do
      of C_ do
        return(Factored::create(factorRC(args(1)), "irreducible"))
      of Expr do
        break
      otherwise
        error("Cannot factor over that domain")
    end_case;

    
    case domtype(options[Adjoin])
      of DOM_LIST do
        options[Adjoin]:= {op(options[Adjoin])};
        break
      of DOM_SET do
        break
      otherwise
        options[Adjoin]:= {options[Adjoin]}
    end_case
  end_if;

  // test the type of the input expression, and
  // change the suitable one to polynomial form
  case domtype(p)
    of DOM_EXPR do
      if stdlib::hasfloat(p) then
        _p_:= numeric::rationalize( p, Restore );
        f:= faclib::factor_expr( _p_, options);
        (f[2*i]:= float( f[2*i])) $ i=1..nops(f) div 2
      else
        f:=faclib::factor_expr( p, options)
      end_if;
      return(Factored::create( f,"irreducible" ))
    of DOM_POLY do
      // the result type of the factorization:
      factored:= l -> Factored::create( l,"irreducible" );

      if degree(p) = 0 then
        return(factored( [tcoeff(p)] ))
      end_if;

      // remove indets which have 0 degree
      xx:= op(p,2);
      f:= select(xx, x -> degree(p,x)>0 );
      if f <> xx then
        p:= poly( p,f );
        _p_:= Factored::convert_to( factor(p, options),DOM_LIST );
        return(factored(
            [_p_[1], (poly(_p_[2*i],xx), _p_[2*i+1]) $ i=1..(nops(_p_) div 2) ]
        ))
      end_if;
      // replace indets that are not in DOM_IDENT
      f:= select(xx,v -> domtype(v) <> DOM_IDENT );
      if f <> [] then
        f:=[(f[i]=genident())$i=1..nops(f)];
        p:=poly(subs(op(p,1..3),f));
        _p_:=Factored::convert_to( factor(p, options),DOM_LIST );
        f:=[(op(f[i],2)=op(f[i],1))$i=1..nops(f)];
        return(factored( subs(_p_,f) ))
      end_if;
      if nterms(p)=1 then
        return(factored( [lcoeff(p),(poly(xx[i],op(p,2..3)),degree(p,xx[i]))
                          $ i=1..nops(xx)] ))
      end_if;
      break;
    of DOM_INT do
      return(Factored::create( stdlib::ifactor(p),"irreducible",Dom::Integer ))
    of DOM_RAT do
      f:= stdlib::ifactor(denom(p));
      return(Factored::create(
        stdlib::ifactor(numer(p)).[(op(f,2*i), -op(f,2*i+1))
                                   $i=1..nops(f) div 2],
        "irreducible", Dom::Integer
      ))
    of DOM_COMPLEX do
      // the result type of the factorization:
      factored:= l -> Factored::create( l,"irreducible" );

      if type(op(p,1))=DOM_FLOAT or type(op(p,2))=DOM_FLOAT then
        return(factored( [1,p,1] ))
      else
        return(factored( [p] ))
      end_if
    of DOM_IDENT do
    of DOM_FLOAT do
      return(Factored::create( [1,p,1],"irreducible" ))
    otherwise
      // overloaded?
      if (f:= p::dom::factor) <> FAIL then
        // Note: Only for test purposes, because the method should already
        //       an object of type Factored!
        f:= f(p);
        if domtype(f) = DOM_LIST then
          return(Factored::create( f,"irreducible" ))
        else
          return(f)
        end_if
      else
        error("Illegal argument: ".expr2text(p))
      end_if
    end_case;

    // only case DOM_POLY was not left using return, so
    // p must be of type DOM_POLY now!

    assert(type(p) = DOM_POLY);


    if (q:=op(p,3)) <> Expr then
      if op(q,0) <> IntMod then
        f:= faclib::domfact(p);
        if domtype(f) = DOM_LIST then
          return(Factored::create( f,"irreducible" ))
        else
          return( f )
        end_if
      end_if;

      q:= op(p,[3,1]);
      // factorization in Zq for ...
      if isprime(q) then
        // .. multivariate polynomials over the field Zq:
        if nops((xx:=op(p,2)))=1 then
          // univariate polynomial given:
          ff:=[lcoeff(p)];
          p:=multcoeffs(p,1/op(ff));
          if degree((t:=nthterm(p,nterms(p))))<>0 then
            ff:=append(ff,poly(op(xx),xx,IntMod(q)),degree(t));
            p:=divide(p,t,Exact);
          end_if;

          if degree(p)=1 then
            return(factored( append(ff,p,1) ))
          else
            return(factored( ff.faclib::factor_mod(p,op(xx),q) ))
          end_if
        else
          // multivariate polynomial given:
          p:=faclib::multifactor(subsop(p,3=Dom::IntegerMod(q)));
          p:=subsop(p,1=expr(p[1]));
          return(factored( subs(p,Dom::IntegerMod(q)=IntMod(q)) ))
        end_if;
      elif nops(op(p,2)) > 1 then
        error("Multivariate factorization modulo composite numbers not yet implemented")
      else
        // ... univariate polynomials over the ring Zq:
        return(factored( [1].faclib::factor_mod_composite(p,op(p,[2,1]),q) ))
      end_if

    else

      // polynomial over Expr
      assert(op(p, 3) = hold(Expr));

      // the result type of the factorization:
      factored:= l -> Factored::create( l,"irreducible" );

      if not testtype(p, Type::PolyOf(Type::Rational))
        or options[Adjoin] <> {} then
        if hastype(p, DOM_FLOAT, {DOM_COMPLEX, DOM_POLY}) then
          _p_:= mapcoeffs( p, numeric::rationalize,  Restore );
          Float:= l -> map(l,
               proc(x)
                 name factor_poly2float;
               begin
                 if domtype(x)=DOM_POLY then
                   mapcoeffs(x,float)
                 else
                   x
                 end_if
               end_proc)
        else
          _p_:=p;
          Float:=id
        end_if;

        
        f:= faclib::factor_expr( expr(_p_), options);
        Factors:=[f[1], (poly(f[2*i], op(p,2)), f[2*i+1]) $ i=1..nops(f) div 2];
        // remove constant factors
        i:=2;
        while i<= nops(Factors) do
          if degree(Factors[i])=0 then
            Factors[1]:=Factors[1]*expr(Factors[i])^Factors[i+1];
            delete Factors[i+1];
            delete Factors[i];
          else
            i:=i+2
          end_if
        end_while;
        return(factored( Float(Factors)) )
      end_if;

      // polynomial has rational coefficients. 
      // factorization in Z

      Factors:=[icontent(p)*sign(lcoeff(p))];
      p:= multcoeffs(p,1/op(Factors));
      if op((t:=faclib::monomial(p)),1)<>[] then
        Factors:=Factors.op(t,1);
        p:= divide(p, poly(op(t,2), op(p,2..3)), Exact);
        // PZ : added the following line because divide does not
        //      reduce the number of indets, which causes a loop
        //      for example with factor(a^2*k-2*a*k+k)
        p:=poly(expr(p));
      end_if;
      pp:=faclib::pfactor(p,0);
      // check sign
      if expr(poly(_mult(_power(expr(op(pp,2*i-1)),op(pp,2*i)) $ i=1..nops(pp)/2))) <> expr(p)
      then
        Factors[1]:= -Factors[1];
      end_if;


    end_if;

    Factors:=Factors.pp;

    if options[Full] and nops(xx) = 1 then
      z:= genident("z");
      expandToRootOf:=
      proc(f: DOM_POLY, n: DOM_INT)
        local r;
      begin
        if degree(f) <= 1 then
          return(f, n)
        end_if;
        r:= solve(op(f, 1), op(f, [2, 1]), MaxDegree = options[MaxDegree]);
        if type(r) <> DOM_SET and type(r) <> RootOf then
           r:= RootOf(subs(op(f,1), op(f, [2, 1]) = z), z)
        end_if;
        (poly(op(xx, 1) - r[i], xx), n) $i=1..degree(f)
      end_proc;
      
      Factors:= [Factors[1],
                 expandToRootOf(Factors[2*i], Factors[2*i+1])
                 $i=1..nops(Factors) div 2
                 ]
    end_if;

  return(factored(
                  subsop( Factors, (((2*i)=subsop(op(Factors,2*i),2=xx))
                          $ i=1..(nops(Factors)-1)/2) )
                  ))
end_proc:
