/* --------------------------------------------------------------------------

stdlib::hasmsign(x) -- heuristic check, whether x  "contains a minus sign"

Call:        stdlib::hasmsign(x)
Parameter:   x -- an object of type Type::Arithmetical
ReturnValue: TRUE or FALSE

Details:

  o This function is used in the simplify attributes of
    functions with reflection formulas f(x) = +/- f(-x)
    (such as sin, sinh, cos etc.) to decide, whether f(x)
    or +/- f(-x) is the preferred output of simplify(f(x)).

    The criterion should be implemented as follows:

    if stdlib::hasmsign(x) = TRUE
    then // -x is (heuristically) simpler than x,
         // so prefer +/- f(-x) over f(x) '
         +/- f(-x)
    else f(x)
    end_if;

  o One always has stdlib::hasmsign(-x) = - stdlib::hasmsign(x) for x <> -x, (i.e., x <> 0).
    This avoids infinite recursions in the simplify application above.
    If x = -x, then stdlib::hasmsign(x) = FALSE.

  o If stdlib::hasmsign(x) = TRUE, then it is guaranteed that x is an
    arithmetical expression containing at least one minus sign.
    Note that -x still may or may not have minus signs.

  o Nothing can be deduced from the result stdlib::hasmsign(x) = FALSE.
    In this case x may or may not have minus signs.

  o This is not an interface function and should not be documented

Examples:

>> stdlib::hasmsign(b-a);           --> TRUE
>> stdlib::hasmsign(a-b);           --> FALSE
>> stdlib::hasmsign( (b+a)*(b-a) ); --> FALSE
>> stdlib::hasmsign(-(b+a)*(b-a) ); --> TRUE
>> stdlib::hasmsign( (b+a)*(a-b) ); --> FALSE
>> stdlib::hasmsign(-(b+a)*(a-b) ); --> TRUE
>> stdlib::hasmsign( 3 + I);        --> FALSE
>> stdlib::hasmsign( 3 - I);        --> FALSE
>> stdlib::hasmsign(-3 + I);        --> TRUE
>> stdlib::hasmsign(-3 - I);        --> TRUE
>> stdlib::hasmsign(  3*x + I );    --> FALSE
>> stdlib::hasmsign(-(3*x + I));    --> TRUE
>> stdlib::hasmsign(  3*x - I);     --> FALSE
>> stdlib::hasmsign(-(3*x - I));    --> TRUE
>> stdlib::hasmsign(-3*x + I);      --> TRUE
>> stdlib::hasmsign(-(-3*x + I));   --> FALSE
>> stdlib::hasmsign(-3*x - I);      --> TRUE
>> stdlib::hasmsign(-(-3*x - I));   --> FALSE

----------------------------------------------------------*/

stdlib::hasmsign:=
proc(x)
local xx, Lp, Lm, Lpm, term, cp, cm;
begin

  if x::dom::hasmsign <> FAIL then
    return(x::dom::hasmsign(x))
  end_if;

  if domtype(x) = DOM_EXPR then
    if generate::isNeg(x) then
      return(TRUE);
    elif generate::isNeg(-x) then
      return(FALSE);
    end_if;
  end_if;
  // if generate::isNeg doesn't help, carry on...

  case expr2text(type(x))
   of "DOM_POLY" do
        return(stdlib::hasmsign(lcoeff(x)))
   of "DOM_INT" do
   of "DOM_RAT" do
   of "DOM_FLOAT" do
        if x<0 then return(TRUE) else return(FALSE) end_if:
   of "DOM_COMPLEX" do
        if Re(x)>0 then return(FALSE) end_if:
        if Re(x)<0 then return(TRUE) end_if:
        // now Re(x) = 0
        if Im(x)>=0 then return(FALSE) end_if:
        if Im(x)< 0 then return(TRUE) end_if:
        break
   of "\"_mult\"" do
        if type(-x) = "_mult" then
          // check only the sign of the numerical constants
          // Beware: the following
          //   return(stdlib::hasmsign(select(x, testtype, Type::Constant)))
          // would yield an infinite recursion for x = 2*sqrt(2)!
          xx:= select(x, testtype, Type::Constant);
          if xx <> x then
             return(stdlib::hasmsign(select(x, testtype, Type::Constant)))
          end_if;
          // all factors of x are of Type::Constant
          for xx in x do
             if stdlib::hasmsign(xx) then
               return(TRUE);
             end_if;
          end_for:
          return(FALSE);
        else
          return(not stdlib::hasmsign(-x))
        end_if;
      break;
    of "\"_plus\"" do
      // If stdlib::hasmsign is called with one argument, then
      // it is made sure that stdlib::hasmsign(x) and stdlib::hasmsign(-x)
      // use exactly the same criterion for reporting
      // signs. If this is not required, then pass any
      // dummy parameter as second argument to switch off
      // this symmetry:

      if args(0)=2
      then Lp:= [op( x)];
           for term in Lp do
               if stdlib::hasmsign(term, 0) = TRUE then return(TRUE) end_if;
           end_for:
      else Lp:= [op( x)];
           Lm:= [op(-x)];
           // make sure that stdlib::hasmsign(x) and stdlib::hasmsign(-x) use
           // exactly the same criterion for determining
           // their result. Use the unique sysorder of the
           // following sorted list (it is the same for
           // stdlib::hasmsign(x) and stdlib::hasmsign(-x)):
           // numerical values are not considered,
           // otherwise a-1 would yield TRUE
           Lpm:= sort(select([op(x), op(-x)], _not@testtype, Type::Constant));

           for term in Lpm do

               // Report a minus sign in term. There
               // is no need for "skew symmetry" between
               // stdlib::hasmsign(term) and stdlib::hasmsign(-term), so
               // do not call stdlib::hasmsign(term) but stdlib::hasmsign(term, 0).
               // This is faster, because only the signs
               // in Lp are investigated:

               if stdlib::hasmsign(term, 0) = TRUE then

                  cp:= bool(contains(Lp, term) <> 0):
                  cm:= bool(contains(Lm, term) <> 0):

                  // Make sure that stdlib::hasmsign(-x) yields FALSE,
                  // if stdlib::hasmsign(x) yields TRUE. So, return TRUE
                  // only, if x has the minus sign but -x has not:

                  if cp and not cm then return(TRUE); end_if;

                  // We have to make this symmetric, i.e.,
                  // stdlib::hasmsign(-x) must return FALSE exactly when
                  // stdlib::hasmsign( x) returns TRUE

                  if cm and not cp then return(FALSE); end_if;

               end_if;
           end_for:
       end_if;
      break;
    of "Series::Puiseux" do
   of "Series::gseries" do
       if (x := lmonomial(x)) = FAIL then // x = O(..)
         return(FALSE);
       else
         return(stdlib::hasmsign(x));
       end_if;
   end_case;

   // No criterion based on minus signs was found.
   // Just make sure that stdlib::hasmsign(-x) = - stdlib::hasmsign(x)
   // for -x <> x (i.e., x <> 0)

   if testtype(x,Type::Arithmetical) then
     if sysorder(-x, x)
        then if x = -x
             then return(FALSE);
             else return(TRUE);
             end_if;
        else return(FALSE);
     end_if;
   else
     return(FALSE);
   end_if;
end_proc:
//-------------- end of file ----------------
