/*++
        linsolve  --  solve a system of linear equations

        linsolve( sys, vars <, ShowAssumptions> ) 
        linsolve( sys, vars, Domain=R )

        sys : set or list of equations
        vars: set or list of variables
        R   : coefficient domain

        'linsolve' solves the set sys of linear equations with 
        respect to the unknowns in unk. The output is either FAIL if 
        there is no solution, or a list 'SolvedEqs'. SolvedEqs is a  
        list of equations of the form x[1]=..,x[2]=..,..
        representing the general solution of sys. Those unknowns not
        subject to any constraint (free parameters) will not turn up
        on the left hand sides in SolvedEqs. E.g., if vars=[x[1],x[2]]
        and SolvedEqs=[x[2]=..], then x[1] is a free parameter spanning
        the kernel of sys. SolvedEqs=[] corresponds to a trivial system
        sys, where all unknowns are arbitrary.

        With assign(SolvedEqs) the solution may be assigned to the unknowns.

  (*) Without option ShowAssumptions:

     (*) Warning: if the equations contain symbolic parameters, then only
         those solutions are returned which hold for arbitrary values of
         these parameters.

         Symbolic denominators occuring in the final result or in
         intermediate results are implicitly assumed to be non-zero.
         The fact that they may vanish for certain values of the
         symbolic parameters is ignored.

  (*) With option ShowAssumptions:

     (*) PivotAssumptions is a list of inequalities involving the
         symbolic parameters of the coefficient matrix. The
         corresponding expressions appeared as symbolic pivot elements
         in the internal Gauss elimination. They are assumed to be
         nonzero and appear as denominators in the final result or in
         itermediate results.

         If no symbolic pivot elements were used, then PivotAssumptions
         is the empty list. This will always be the case when
         there are no symbolic parameters in the coefficient
         matrix.

         The inequalities arising in PivotAssumptions are
         determined by the algorithm during run time. The user
         has no control over these assumptions. In particular,
         changing the ordering of the equations in Eqs or the
         ordering of the variables in Vars may result in a
         different solution returned by linsolve.

     (*) Constraints is a list of equations involving the symbolic parameters
         in the equations.

         Eqs is solvable if and only if the
         equations given by Constraints are satisfied.

         Such constraints arise only when the coefficient matrix
         does not have full rank.

         If no such contraints arise, then
         the returned value of Constraints is the empty list.

     (*) Under the assumptions PivotAssumptions and assuming solvability
         of Eqs (i.e., assuming the equations given by Constraints),
         SolvedEqs represents the general solution of Eqs.

        Note: If a third argument is given then the system is solved
              over the given CoeffRing. If CoeffRing <> Dom::ExpressionField() 
              then constraints can not be computed. Hence, Contraints is the 
              empty list.

        Examples:

        linsolve({x+y=1,2*x+2*y=3},{x,y})  ->  FAIL
        linsolve({x+y=1,2*x+y=3},{x,y})    ->  [x = 2, y = -1]
        linsolve({x+y+z=1,2*x+y=3},{x,y})  ->  [x = z + 2, y = - 2 z - 1]
        linsolve({0},{x,y})		   ->  []
        linsolve({x=x+1},{x})		   ->  FAIL
        linsolve({y=0},{x,y});		   ->  [y = 0]
++*/

linsolve:= proc(sys,vars,R)
  local sol, i, tmp_R, tmp_vars, AB;
begin

  if testargs() then
    if args(0) < 1 or args(0) > 3 then
      error("wrong no of args")
    end_if;
    if not contains( {DOM_SET,DOM_LIST},domtype(sys) ) then
      error("expecting set or list of equations")
    end_if;
    if args(0) > 1 then
      tmp_vars:= vars;

      if contains( {DOM_SET,DOM_LIST},domtype(tmp_vars) ) then
        if nops(tmp_vars) = 0 then
          error("no variables given")
        elif has( tmp_vars,Type::ConstantIdents ) then
          error("invalid variables found")
        end_if;
        if args(0) = 3 then tmp_R:= R else tmp_R:= NIL end_if
      elif args(0) = 3 then
        error("2nd argument must be a set or list of variables")
      else
        tmp_R:= tmp_vars;
      end_if;
      if tmp_R <> NIL and tmp_R <> hold(ShowAssumptions) then
        if type(tmp_R) = "_equal" then
          if op(tmp_R,1) = hold(Domain) then
            tmp_R:= op(tmp_R,2);
            if tmp_R::dom <> DOM_DOMAIN or tmp_R::hasProp( Cat::Field ) <> TRUE then
              error("expecting 'Domain=R', where R is a domain of category 'Cat::Field'")
            end_if
          else
            error("invalid argument")
          end_if
        else
          error("invalid argument")
        end_if
      end_if
    end_if
  end_if;

  case args(0)
  of 1 do
    R:= NIL; vars:= null(); 
    break
  of 2 do
    if not contains( {DOM_SET,DOM_LIST},domtype(vars) ) then
      R:= vars;
      vars:= null()
    else
      R:= NIL
    end_if
  end_case;

  // convert each equation l = r to the form l-r=0, because
  // indets(eqn,PolyExpr) does not work as we need here!
  sys:= map( sys,x->( if type(x) = "_equal" then op(x,1)-op(x,2) else x end ) );
  if vars = null() then
    vars := _union( indets(op(sys,i),hold(PolyExpr)) $ i=1..nops(sys) );
    vars := map( vars,x->(if has(x,Type::ConstantIdents) then null() else x end) );
  end_if;

  if type(R) = "_equal" then 
    R:= op(R,2);
    if R = Dom::ExpressionField() then 
      // use numeric::linsolve!
      R:= NIL 
    end_if
  end_if;

  //--------------------------------------------------
  // Walter, 23.2.01:
  // generate a predictable ordering of the variables.
  // Note that the internal ordering of sets is not
  // necessarily the same as produced by sort.
  // Use DOM_SET::sort: now we can document what ordering
  // will be used. It coincides with the ordering of
  // sets as printed on the screen.
  //--------------------------------------------------
  if domtype(vars) = DOM_SET then
     vars := DOM_SET::sort(vars);
  end_if;
  // Now, vars ist a list!

  if R = NIL or R = hold(ShowAssumptions) then
    if testargs() then
      if solvelib::solve_islinear(sys,vars) = FALSE then
        error("equations must be linear")
      end_if
    end_if;

    userinfo(1,"call numeric::linsolve with option 'Symbolic'");
    if R = hold(ShowAssumptions) then
      return(numeric::linsolve( sys,vars,hold(Symbolic),hold(ShowAssumptions)))
    else
      return(numeric::linsolve( sys,vars,hold(Symbolic) ))
    end_if
  elif R = Dom::Float then
    if testargs() then
      if solvelib::solve_islinear(sys,vars) = FALSE then
          error("equations must be linear")
      end_if
    end_if;

    userinfo(1,"call numeric::linsolve");
    return( numeric::linsolve(sys,vars) )
  else
    AB:= linalg::expr2Matrix( sys,vars,R );
    userinfo(1,"call linalg::matlinsolve");
    if (sol:= linalg::matlinsolve( AB, vars)) =
      FAIL then
      // the general solution cannot be computed over the component domain,
      // try to compute at least one solution:
      warning("solution(s) may be lost");
      sol:= linalg::matlinsolve( AB, hold(Special) );
    end_if;

    if sol = [] then 
      // no solution
      return( FAIL )
    else
      sol:= [op(vars,i)=sol[i] $ i=1..nops(vars)];
      // remove solutions of the form 'x = x':
      return( select(sol,x -> expr(op(x,1)) <> expr(op(x,2))) )
    end_if
  end_if
end_proc:

