/* arithmetical operations for lists */

/*
See http://inside.mathworks.com/wiki/G544893_map_arithmetical_base_operation_in_lists_RFA
*/

/*
DOM_LIST::_plus()
The sum of two lists is [a1, ..., an] + [b1, ..., bn] = [a1 + b1, ..., an + bn]. It is an error if the lists do not have the same length. 
The sum of a list and any non-list is [a1, ..., an] + b = [a1 + b, ..., an + b] and b + [a1, ..., an] = [b + a1, .., b + an]. 
For any list _plus([a1, ..., an]) = [_plus(a1), ..., _plus(an)]. 
The sum of three or more operands is defined by adding them from left to right, e.g., a1 + a2 + a3 = (a1 + a2) + a3. 
*/
DOM_LIST::_plus:=
proc(a, b)
begin
  if args(0) = 1 then
    return(map(a, _plus))
  end_if;
  // the case args(0) > 2 should have been handled by the kernel
  assert(args(0) = 2);
  if domtype(a) = DOM_SET or domtype(b) = DOM_SET then
    return(DOM_SET::_plus(a, b))
  end_if;
  if domtype(a) = DOM_LIST then 
    if domtype(b) = DOM_LIST then
       if nops(a) <> nops(b) then
         error("Lists do not have the same length")
       else
         return(zip(a, b, _plus))
       end_if
    else
      return(map(a, _plus, b))
    end_if
  else
    assert(domtype(b) = DOM_LIST); // otherwise this method would not have been called
    return(map(b, x -> a+x))  // works for noncommutative _plus also
  end_if   
end_proc:

// DOM_LIST::_mult
// the same as _plus. Should we avoid code duplication by using a generator for list operations?
DOM_LIST::_mult:=
proc(a, b)
begin
  if args(0) = 1 then
    return(map(a, _mult))
  end_if;
   // the case args(0) > 2 should have been handled by the kernel
  assert(args(0) = 2);
  if domtype(a) = DOM_SET or domtype(b) = DOM_SET then
    return(DOM_SET::_mult(a, b))
  end_if;
  if domtype(a) = DOM_LIST then 
    if domtype(b) = DOM_LIST then
       if nops(a) <> nops(b) then
         error("Lists do not have the same length")
       else
         return(zip(a, b, _mult))
       end_if
    else
      return(map(a, _mult, b))
    end_if
  else
    assert(domtype(b) = DOM_LIST); // otherwise this method would not have been called
    return(map(b, x -> a*x))  // works for noncommutative _mult also
  end_if   
end_proc:


// DOM_LIST::_power
// the same as _plus and _mult, but must be called with exactly two arguments
DOM_LIST::_power:= 
proc(a, b)
begin
  if args(0) <> 2 then
    error("Wrong number of arguments")
  end_if;
  if domtype(a) = DOM_SET or domtype(b) = DOM_SET then
    return(DOM_SET::_power(a, b))
  end_if;

  if domtype(a) = DOM_LIST then 
    if domtype(b) = DOM_LIST then
       if nops(a) <> nops(b) then
         error("Lists do not have the same length")
       else
         return(zip(a, b, _power))
       end_if
    else
      return(map(a, _power, b))
    end_if
  else
    assert(domtype(b) = DOM_LIST); // otherwise this method would not have been called
    return(map(b, x -> a^x))  
  end_if   
end_proc:


