//      

/*++
loadproc -- load an object from file on demand

loadproc(p, l, f [,c [,d,...]])

p - an expression
l - a path name
f - a file name without suffix
c - a function (optional)
d - data (optional)

loadproc on demand reads in the file f containing the definition of
the object p. On demand means: the file is read when the result
of the loadproc-call is evaluated for the first time. The argument
p is not evaluated.

After the result of the loadproc-call is evaluated and the file is
read p is evaluated and returned instead of the result of the
loadproc-call.

The typical use is loading a procedure p on demand as in
    p:= loadproc(p, "LIB/", "p");
Here the definition of p is read when p is evaluated for the first
time.

The file name f must not contain a suffix. First the binary file
	l.f.".mb"
is searched for. If this binary file doesn't exist the t file
	l.f.".mu"
is searched for.

Note that the varaible LIBPATH is used to locate the file.

If the additional options c and d,... are given, the expression 'c(d,...)'
is evaluated after p is read from file, but before p is executed. The
callback function c may be used to change p in a certain way after it is
read from file.
++*/

loadproc := proc(p)
    option noDebug, hold;
begin
    if testargs() then
	if args(0) < 3 then
	    error("wrong no of arguments")
	end_if;

	case domtype(p)
	of DOM_IDENT do
	of DOM_VAR do
	of DOM_EXPR do
	    break;
	otherwise
	    error("wrong procedure name")
	end_case;
    end_if;

    new(stdlib::LoadProc, p, op(context([args(2..args(0))])))
end_proc:

// LoadProc -- domain to load library functions on demand 

stdlib::LoadProc:= newDomain("stdlib::LoadProc"):
stdlib::LoadProc::create_dom:=hold(stdlib::LoadProc):
stdlib::LoadProc::evaluateIndex:=TRUE:

stdlib::LoadProc::isValid :=
proc(lpn)
  option noDebug;
  name loadproc;
  local op1, op0, err;
begin
  err := FALSE;
  op1 := op(lpn,1);
  if domtype(op1) = DOM_IDENT then
    if eval(subsop(hold(val)(1), 1=op(lpn,1))) = lpn then
      err := TRUE;
    end_if;
  elif type(op1) = "slot" then
    op0 := op(op1, 1);
    if domtype(op0) = DOM_IDENT then
      op0 := eval(op0);
    end_if;
    if domtype(op0) = DOM_DOMAIN and
      eval(subsop(hold(indexval)(op0, 1), 2=op(op1,2))) = lpn then
      err := TRUE;      
    elif domtype(op0) = DOM_FUNC_ENV and domtype(op(op0, 3)) = DOM_TABLE and
      eval(subsop(hold(indexval)(op(op0, 3), 1), 2=op(op1,2))) = lpn then
      err := TRUE;      
    end_if;
  end_if;
  if err = TRUE then
      error("'".expr2text(op(lpn,1))."' not defined in file '".op(lpn,2).op(lpn,3).".mu'");
  end_if;
end_proc:


stdlib::LoadProc::new := proc()
    option noDebug;
    name loadproc;
begin
    new(stdlib::LoadProc, args())
end_proc:

// indent counter for verboseRead
stdlib::LoadProc::indent := 0:

stdlib::LoadProc::read_file:=
proc(path, fnam)
  option escape, noDebug;
  name loadproc;
  local p, f, pref_val, prevmem, prevtime, prevrtime;
  save PRETTYPRINT;
begin
  pref_val:= 0;
  // be shure not to load Pref::verboseRead if not used yet
  if domtype(val(Pref)) = DOM_DOMAIN and domtype(Pref::_verboseRead) = DOM_INT then
    pref_val := Pref::_verboseRead;
  end_if;
  if pref_val <> 0 then
    PRETTYPRINT:= FALSE;
    prevmem := bytes()[1];
    prevtime := time();
    prevrtime := rtime();
  end_if;
  for p in LIBPATH do
    // read file, try mb-file first, then mu-file
    f:= fopen(p.path.fnam.".mb");
    if f <> FAIL then
      case pref_val
        of 0 do
          break
        of 1 do
          if strmatch(path,"LIBFILES") then
            print(Unquoted,"loading package '".fnam."' from ".p);
          end_if;
          break
        of 2 do
          if strmatch(path,"LIBFILES") then
            print(Unquoted,"loading package '".fnam."' [".p."]");
          else
            print(Unquoted,"reading file ".p.path.fnam.".mb");
          end_if;
        of 3 do
          if strmatch(path,"LIBFILES") then
            fprint(Unquoted, 0, "  " $stdlib::LoadProc::indent,
                   "loading package '".fnam."' [".p."]");
          else
            fprint(Unquoted, 0, "  " $stdlib::LoadProc::indent,
                   "reading file ".p.path.fnam.".mb");
          end_if;
          // increase indent counter
          sysassign(stdlib::LoadProc::indent, stdlib::LoadProc::indent+1)
      end_case;

      stdlib::syseval(fread(f, Plain, Quiet));
      fclose(f);
      break;
    end_if;

    f:= fopen(p.path.fnam.".mu");
    if f = FAIL then next end_if;
    case pref_val
      of 0 do
        break
      of 1 do
        if strmatch(path,"LIBFILES") then
          print(Unquoted,"loading package '".fnam."' [".p."]");
        end_if;
        break;
      of 2 do
        if strmatch(path,"LIBFILES") then
          print(Unquoted,"loading package '".fnam."' [".p."]");
        else
          print(Unquoted,"reading file ".p.path.fnam.".mu");
        end_if;
        break;
      of 3 do
        if strmatch(path,"LIBFILES") then
          fprint(Unquoted, 0, "  " $stdlib::LoadProc::indent,
                 "loading package '".fnam."' [".p."]");
        else
          fprint(Unquoted, 0, "  " $stdlib::LoadProc::indent,
                 "reading file ".p.path.fnam.".mu");
        end_if;
        // increase indent counter
        sysassign(stdlib::LoadProc::indent, stdlib::LoadProc::indent+1)
    end_case;
    
    stdlib::syseval(fread(f, Plain, Quiet));
    fclose(f);
    break
  end_for;
                                
  case pref_val
    of 3 do
      // decrease indent counter
      sysassign(stdlib::LoadProc::indent, stdlib::LoadProc::indent-1);
      if strmatch(path,"LIBFILES") then
        fprint(Unquoted, 0, "  " $stdlib::LoadProc::indent,
               "finished loading package '".fnam."' [".p."]");
      else
        fprint(Unquoted, 0, "  " $stdlib::LoadProc::indent,
               "finished reading file ".p.path.fnam.".mu");
      end_if;
      fprint(Unquoted, 0, "  " $stdlib::LoadProc::indent,
	     "memory usage increased by ".(bytes()[1]-prevmem)." bytes, time diff = ".
	     (time()-prevtime).", rtime diff = ".(rtime()-prevrtime));
  end_case;

    if f = FAIL then
        error("can't read file '".path.fnam.".mu'")
    end_if;
end_proc:

stdlib::LoadProc::testtype :=
proc(elem, ttype) : DOM_BOOL 
  option noDebug;
  name loadproc::testtype;
begin
  if ttype = dom then
    bool(type(elem) = ttype /* stdlib::LoadProc */ )
  else
    ttype::dom::testtype(elem, ttype)
  end_if
end_proc:

stdlib::LoadProc::evaluate := proc(lpn)
    option noDebug;
    name loadproc;
begin
    stdlib::LoadProc::read_file(op(lpn,2), op(lpn,3));
    dom::isValid(lpn);
    if nops(lpn) > 3 then
    // call float attribute 
	op(lpn,4)(op(lpn, 5..nops(lpn)))
    end_if;
    context(op(lpn,1))
end_proc:

stdlib::LoadProc::func_call := proc(lpn)
    option noDebug;
    name loadproc;
begin
    stdlib::LoadProc::read_file(op(lpn,2), op(lpn,3));
    dom::isValid(lpn);
    if nops(lpn) > 3 then
    // call float attribute 
	op(lpn,4)(op(lpn, 5..nops(lpn)))
    end_if;
    context(op(lpn,1)(args(2..args(0))))
end_proc:

stdlib::LoadProc::set_func_call := proc(lpn)
    option noDebug;
    name loadproc;
    local a,v;
begin
    stdlib::LoadProc::read_file(op(lpn,2), op(lpn,3));
    dom::isValid(lpn);
    if nops(lpn) > 3 then
    // call float attribute 
	op(lpn,4)(op(lpn, 5..nops(lpn)))
    end_if;
    context(op(lpn,1))
end_proc:

stdlib::LoadProc::posteval := proc(lpn)
    option noDebug;
    name loadproc;
begin
    stdlib::LoadProc::read_file(op(lpn,2), op(lpn,3));
    dom::isValid(lpn);
    if nops(lpn) > 3 then
    // call float attribute 
	op(lpn,4)(op(lpn, 5..nops(lpn)))
    end_if;
    context(op(lpn,1))
end_proc:

// end of file 
