/*
Autor: O. Kluge and W. Oevel
Date of first implementation: 25.1.2005
shipped since Version 4.0

mtaylor - compute a multivariate Taylor series expansion

mtaylor(f, [x = x0, y = y0, ...] computes the first terms 
of the multivariate Taylor series of f with respect to the 
variables x, y, ... around the points x0, y0, ...

Call(s):
     mtaylor(f, [x < = x0>, y < = y0>, ...] <, weights> )
     mtaylor(f, [x < = x0>, y < = y0>, ...], order <, mode>, <, weights>)
     mtaylor(f, [x < = x0>, y < = y0>, ...], mode = order <, weights>)

Parameters:
     f - an arithmetical expression representing a 
         multivariate function in x, y, ...  
     x, y, ...   - identifiers or indexed identifiers
     x0, y0, ... - the (optional) expansion point w.r.t. x, y, ...
                   The expansion point should be an arithmetical
		   expression; if not specified, the default 
	           expansion point 0 is used.
     order   - (optional) non-negative integer; the number of terms 
               to be computed. The default order is given by the
               environment variable ORDER (default value 6).
     mode    - RelativeOrder (default) or AbsoluteOrder
     weights - (optional) list of positive integers

Overloadable by: f

Return Value: a symbolic expression.

Side Effects: The function is sensitive to the environment 
              variable ORDER, which determines the default 
              truncation order.

Related Functions: taylor, series

See Examples

Details:

* The mtaylor function computes a truncated multivariate Taylor 
  series expansion of the input expression f, with respect to the 
  variables x, y, ..., to order n, using the variable weights w. 
* The variables x, y, ... can be a list or set of identifiers or 
  equations. 
* If the third argument n is present then it specifies the 
  "truncation order" of the series. The concept of "truncation order" 
  used is "total degree" in the variables. If n is not present, the 
  truncation order used is the value of the environment variable ORDER, 
  which is 6 by default. 
* If the fourth argument w is present it specifies the variable weights 
  to be used (by default all 1). A weight of 2 will halve the order in 
  the corresponding variable to which the series is computed.
* Note: mtaylor restricts its domain to "pure" Taylor series, those 
  series with non-negative powers in the variables. 
  Exception: if an expansion point such as x0 = +-infinity is specified,
             a Taylor expansion w.r.t. 1/x around x = 0 is computed.

Example 1:
>> mtaylor(exp(x^2+y), [x,y], 4);
                         2        2    2          3
                1 + y + x  + 1/2 y  + x  y + 1/6 y

Example 2:
>> mtaylor(sin(x^2+y^2), [x,y], 8);

   - 1/6*x^6 - 1/2*x^4*y^2 - 1/2*x^2*y^4 + x^2 - 1/6*y^6 + y^2

Example 3:
>> mtaylor(sin(x^2+y^2), [x,y], 10, [2,1]);

- 1/2*x^4*y^2 - 1/2*x^2*y^4 + x^2 + 1/120*y^10 - 1/6*y^6 + y^2

Example 4:
>> mtaylor(sin(x^2+y^2), [x=1,y], 3);

                                                           2           2
   sin(1) + 2 cos(1) (x - 1) + (-2 sin(1) + cos(1)) (x - 1)  + cos(1) y


Example 5:
>> mtaylor(f(x,y), [x,y], 3);

                                                                    2
f(0, 0) + D[1](f)(0, 0) x + D[2](f)(0, 0) y + 1/2 D[1, 1](f)(0, 0) x  

                                                    2
     + x D[1, 2](f)(0, 0) y + 1/2 D[2, 2](f)(0, 0) y
*/

mtaylor := proc (f, vars) 
  local vars0, vars_orig, weights, furtherargs, 
        k, m, eps, ff, g, i; 
begin
  // if args(0) < 2 then
  //   vars:= [op(numeric::indets(f))];
  // end_if;
  if args(0) < 2 then
     error("expecting at least two arguments"):
  end_if;
  if f::dom::mtaylor <> FAIL then
     return(f::dom::taylor(args()));
  end_if;
  //----------------------------
  // get the expansion variables
  //----------------------------
  if testtype(vars, DOM_SET) then 
     vars := [op(vars)] 
  elif not testtype(vars, DOM_LIST) then 
     vars := [vars] 
  end_if; 
  if not testtype(
       vars, 
       Type::ListOf(
         Type::Union(
           DOM_IDENT, 
           "_index",
           Type::Equation(DOM_IDENT, Type::Arithmetical),
           Type::Equation("_index", Type::Arithmetical)
         ), 1
       )
     ) then
     error("invalid 2nd argument (expansion point)")
  end_if; 
  m := nops(vars); 
  if m <> nops({op(vars)}) then 
     error("duplicate variables (2nd argument)") 
  end_if; 

  //-----------------------------------
  // split the weights from the options
  //-----------------------------------
  weights := [1 $ m];   // default
  furtherargs:= null(); // default

  for i from 3 to args(0) do
     if domtype(args(i)) = DOM_LIST then
        weights:= args(i);
        if not testtype(weights, Type::ListOf(Type::PosInt, m, m)) then 
           error("the weights must be a list of positive integers")
        end_if;
     else
        if not (
                 domtype(args(i)) = DOM_INT or   // order
                 has(args(i), AbsoluteOrder) or
                 has(args(i), RelativeOrder) or
                 has(args(i), Left) or
                 has(args(i), Right) or
                 has(args(i), Real) or
                 has(args(i), NoWarning) or
                 has(args(i), Mapcoeffs)
               ) then
           error("unexpected argument: ".expr2text(args(i)));
        end_if;
        furtherargs:= furtherargs, args(i);
     end_if;
  end_for;

  //----------------------------
  // action:
  //----------------------------
  vars0 := map(vars, x -> if type(x) = "_equal" then op(x, 2) else 0 end_if); 
  vars_orig:= vars;
  vars  := map(vars, x -> if type(x) = "_equal" then op(x, 1) else x end_if); 

  // allow expansions around +/- infinity:
  for i from 1 to m do
    if hastype(vars0[i], stdlib::Infinity) or 
       has(vars0[i], complexInfinity) then
       weights[i]:= -weights[i];
       vars0[i]:= 0; 
    end_if;
  end_for:

  // do the expansion as a univariate expansion w.r.t. eps, then put eps=1:
  eps := genident();
  ff := subs(f, [vars[k] = vars0[k] + vars[k]*eps^weights[k] $ k = 1 ..m], EvalChanges); 

  //---------------------------------------------------------------------------
  // Beware:  in mtaylor(f(x/y), [x, y]) the expansion variable eps disappears,
  // leading to the ``expansion'' f(x/y). This is nonsense. Of course, this
  // is a bad request by the user, only mtaylor(f(x/y), [x = 0, y = infinity])
  // make sense. However, we still need to deal with this in a reasonable way:           
  //---------------------------------------------------------------------------
  if traperror((
     g:= subs(ff, hold(_mult) = 
     proc()
     local Args, result;
     begin
       Args:= [args()];
       result:= _mult(op(Args));
       if _or(has(result, vars[k])  $ k = 1..m) and
          not has(result, eps) then
           error("cancellation of expansion variable");
       end_if;
       result;
     end_proc, Unsimplified);
     eval(g))) > 0 then
     return(procname(args()))
  end_if;

  //--------------------------
  // Proceed: do the expansion
  //--------------------------
  if traperror((g:= taylor(ff, eps = 0, furtherargs, NoWarning))) > 0 then
     error("cannot compute a Taylor expansion of ".expr2text(f));
  end_if;

  if type(g) = "taylor" then
     vars:= vars_orig; // reset args() to the user's input
     return(procname(args()));
   end_if;

  ff:= subs(expr(g), eps = 1, EvalChanges):

  // The expansion is done. Start post-processing.
  // We wish to use poly(ff, vars) to collect the variables.
  // However, the variables with infinite expansion points
  // (determined by negative weights) would make poly produce FAIL.
  // Replace these variables by their reciprocals temporarily.
  for k from 1 to m do
    if weights[k] < 0 then
       ff:= subs(ff, vars[k] = 1/vars[k]);
    end_if;
  end_for;

  // use poly to collect the exrpession w.r.t. the variables:
  if traperror((ff:= expr(poly(ff, vars));)) > 0  or
     ff = FAIL or
     has([coeff(ff)], [vars[k] $ k =1..m]) then
       error("cannot compute a Taylor expansion of ".expr2text(f));
  end_if;

  // Undo replacement of variables by their reciprocals:
  for k from 1 to m do
    if weights[k] < 0 then
       ff:= subs(ff, vars[k] = 1/vars[k]);
    end_if;
  end_for;

  ff:= subs(expr(ff), [vars[k] = vars[k]-vars0[k] $ k = 1 .. m]);

  return(eval(ff));
end_proc:
