/* ==================================================

stdlib::normalNoExpand -- alternative normal form of a rational expression

Call: stdlib::normalNoExpand(x)
      stdlib::normalNoExpand(List);
      stdlib::normalNoExpand(Set);

Parameters:
      x    - an arithmetical expression
      List - a list of arithmetical expressions
      Set  - a set of arithmetical expressions

The normal form of x produced by stdlib::normalNoExpand is numer(x)/denom(x),
where the numerator and the denominator are products of powers
of expanded polynomial expressions. The factors of numerator 
and denominator are coprime (as polynomials over Q_). 

Note: Numerator and denominator are not coprime as
polynomials over Q_[I], i.e.: 
  stdlib::normalNoExpand((x^2 + 1)/(x + I)) = (x^2 + 1)/(x + I).

Note: Since the factors of numerator and denominator do
      not correspond to a full factorization, stdlib::normalNoExpand
      generates of 'normal form' in the following sense:

      If f1 and f2 are multivariate rational expressions,
      then we do **not** have 

         f1 = f2   <==>   normal(f1) = normal(f2).

      However, the following does hold:

         f1 = f2   <==>   stdlib::normalNoExpand(f1 - f2) = 0.

Examples:

>> stdlib::normalNoExpand( (x^2 + 4*x + 4) / (x^2 + 5*x + 6))

-->    (x + 2) / (x + 3)

>> stdlib::normalNoExpand( expand((x^2 + 1)*(x + 2))*(x + sin(x)) /
            expand((x + 2)^2 * (x + I))/(x + 1)^2)

--->  (x^2 + 1)*(x + sin(x))
       /(x + 1)^2 / (x^2 + x*(2 + I) + 2*I)


==================================================== */
stdlib::normalNoExpand:= proc(x)
local floatmode, substeqs, substeqs2, rnormal, options, options2, 
      myrationalize, mypower, substitutions, minpolys, doiterate,
      counter, oldresults, substexprs, y, t, exps, undo, _split, 
      exactdivide, GCD; 
//option remember;  // use prog::remember to make sure that the
                    // remember table gets cleaned up regularly
begin

  // ===========================================================
  // rnormal is the recursive normalization routine. It expects
  // its argument x to be rationalized (i.e. to be composed of
  // identifiers and integers/rationals and I).
  // Note that I is never passed to gcd ! (Before a gcd is 
  // computed, I is replaced by a symbolic identifier.)
  // ===========================================================

  rnormal:= proc(x, options)
  local den, den1, den2, d1, d2, commondenom, 
        num, num1, num2, n1, n2, commonnumer,
        c, c1, c2, i, j, vars, y, GCD, 
        g, minexpo, minexpo2, z,
        myexpand, mycontent, extractContent, 
        mygcd;
  option remember; // use option remember instead of prog::remember,
                   // because the local variable and its remember 
                   // table is thrown away after return of normalNoExpand
  begin
    // possible options: NoGcd, "Squarefree", "Prefactor"
    case type(x)
    of DOM_INT do
    of DOM_RAT do
    of DOM_FLOAT do
    of DOM_COMPLEX do
    of DOM_INTERVAL do 
       return([x, 1, 1]);
    of DOM_IDENT do
    of "_index" do
       return([1, x, 1])
    of "_power" do
       [c, num, den]:= rnormal(op(x, 1), options);
       if op(x, 2) >= 0 then
          return([c^op(x, 2), num^op(x, 2), den^op(x, 2)]);
       else
          return([c^op(x, 2), den^(-op(x, 2)), num^(-op(x, 2))]);
       end_if;
    end_case;
  
   // ======================================================
   // after improvements in gcd over Q_[I] (now avoids costly
   // computations over Dom::AlgebraicExtension), we use the
   // usual gcd:
   // ======================================================

    mygcd:= gcd:

    // =============================================
    // utility for fast extraction of the content of
    // expressions. It calls the standard procedure
    // 'content` but is smarter about products and
    // powers.
    // =============================================
    mycontent:= proc(p)
      option remember;
    begin
      case type(p)
      of "_power" do
         return(mycontent(op(p,1))^op(p, 2));
      of "_mult" do
         return(map(p, mycontent));
      otherwise
         return(content(p));
      end_case;
    end_proc:
  
    //======================================================
    // [content, y] = extractContent(x) extracts the content
    // from a rational expression x that is the quotient of 
    // products of polynomial expressions. The resulting
    // expression y is again the quotient of products of
    // polynomial expressions in which each factor has the
    // integer content 1. One has x = content*y.
    //======================================================
    extractContent:= proc(x)
      local i, c;
    begin
      case type(x) 
      of DOM_INT do
      of DOM_RAT do
         return([x, 1]);
      of DOM_FLOAT do
      of DOM_COMPLEX do
      of DOM_IDENT do
      of "_index" do
         return([1, x]);
      of "_mult" do
         c:= 1:
         x:= map(x, proc(x) local cc; begin
                      [cc, x]:= extractContent(x);
                      c:= c*cc;
                      x;
                    end_proc);
         return([c, x]);;
      of "_power" do
         i:= mycontent(op(x, 1));
         if iszero(i) then
            // if x is the power of a poly that is not expanded,
            // op(x, 1) may be the zero poly in disguise. Note
            // that the content of the zero poly is zero.
            return([i, 0]);
         end_if;
         c:= i^op(x, 2);
         x:= (op(x, 1)/i)^op(x, 2);
         return([c, x]);;
      of "_plus" do
         c:= mycontent(x);
         if iszero(c) then
            return([c, 0]);
         end_if;
         // since x is of type "_plus" and c is an
         // integer, we can be sure that the kernel
         // turns x/content into an expanded sum
         return([c, x/c]);;
      otherwise
         error("forgotten case");
      end_case;
    end_proc;
  
    // ================================================
    // expand a polynomial expression via poly. This is
    // (much) faster than expand(.., ArithmeticOnly).
    // ================================================
    myexpand:= proc(x)
    local y;
    begin
      y:= poly(x);
      if y <> FAIL then
        expr(y)
      else
         x
      end_if:
    end_proc:
  
    // =============================================================
    // =============================================================
    //                    main of rnormal
    // =============================================================
    // =============================================================
  
    // =====================
    // normalization of sums
    // =====================
    // use 'case type(x) of "_plus"' instead of 'if type(x) = "_plus"',
    // because we want to be able to use 'break' to proceed to the
    // "_mult" code further down below:
    case type(x) of "_plus" do

     // =====================================================
     // --------- begin of pre_factor_in_normal ------------- 
     // =====================================================
     // Check if there can be common factors in the
     // sum. Simple stuff as x^2 + x^3 with the common
     // factor x^2 is not very interesting (i.e., do not
     // waste time by looking into expanded polynomials).
     // Look only for stuff such as 
     //     (x+1)*x - (x+1)*y = (x+1)*(x-y)
     // Thus, we need to check if all terms in a sum 
     // contain at least 1 factor that is a sum.

     // Memory tends to explode with options["Prefactor"]} = TRUE.

     if options["Prefactor"] and 
         map((y:= {op(x)}), hastype, "_plus") = {TRUE} then
      // All terms in the sum x contain a sum. Thus,
      // there is some chance that there is a common
      // non-trivial factor.

      // ===================================
      // extract common factors from the sum
      // ===================================
      y:= map(y, x -> if type(x) = "_mult" then {op(x)} else {x} end_if);
      // now, y = y11*y12*.. + y21*y22*.. + ... is represented by
      // y = { {y11,y12,..}, {y21,y22,..}, ..} 
      // Search for the minimal powers occuring in the terms:

      y:= sort([op(y)], (a, b) -> (nops(a) < nops(b))):
      minexpo:= table(RD_INF);
      map(y[1], x -> if type(x) = "_power" then 
                        minexpo[op(x, 1)]:= min(minexpo[op(x, 1)], op(x, 2)); 
                     else 
                        minexpo[x]:= min(minexpo[x], 1); 
                     end_if);
      for i from 2 to nops(y) do 
          // 1/x^3 + 1/x^2 = 1/x^3 *(1 + x)   -- extract min expo 
          // 1/x^3 + x^2   = 1/x^3 *(1 + x^5) -- extract min expo
          // x^3 + 1/x^2   = 1/x^2 *(x^5 + 1) -- extract min expo
          // x^3 + x^2     =  x^2  *(x + 1)   -- extract min expo
          minexpo2:= table(RD_INF);
          map(y[i], x -> if type(x) = "_power" then 
                            minexpo2[op(x, 1)]:= min(minexpo2[op(x, 1)], op(x, 2)); 
                         else 
                            minexpo2[x]:= min(minexpo2[x], 1); 
                         end_if);
          for z in map([op(minexpo)], op, 1) do
            if not contains(minexpo2, z) then
               delete minexpo[z];
            else
               minexpo[z]:= min(minexpo[z], minexpo2[z]);
            end_if;
          end_for:
          if nops(minexpo) = 0 then
             break;
          end_if;
      end_for;
      if nops(minexpo) > 0 then
         g:= _mult(op(x, 1)^op(x, 2) $ x in minexpo);
         if indets(g) <> {} and hastype(g, "_plus") then 
           // divide the terms in x by g in the form as found above
           y:= map(x, _divide, g);
           if type(y) = "_plus" then
              // There is no point in trying again to 
              // extract factors from the sum y.
              options["Prefactor"]:= FALSE;
           end_if;
           y:= g*y;
           if y <> x then 
              return(rnormal(y , options));
           end_if:
         end_if;
      end_if;
     end_if; // of 'if options["Prefactor"]'
     //===================================================
     //--------- end of pre_factor_in_normal ------------- 
     //===================================================
      
      // ===================================
      // Check if the input is a polynomial.
      // If 'yes', return immediately.
      // ===================================
      vars:= indets(x):
      if nops(vars) <> 0 and
         (y:= poly(x, [op(vars)])) <> FAIL then
         [c, x]:= extractContent(expr(y));

         if options["Squarefree"] and
            (vars:= indets(x)) <> {} then
           y:= poly(x, [op(vars)]);
           x:= expr(polylib::sqrfree(y));
         end_if:

         return([c, x, 1]);
      end_if;

      // ============================================
      // Reduce to a sum with 2 terms. If there are
      // more than 2 terms in the sum, then do
      // 'bisectioning', i.e., 
      //    normal(x1 + x2 + x3 + x4) 
      //  = normal(normal(x1 + x2) + normal(x3 + x4))
      // ============================================
      if nops(x) > 2 then 
        [c1, num1, den1]:= rnormal(_plus(op(x, i) $ i = 1 .. trunc(nops(x)/2)), options);
        [c2, num2, den2]:= rnormal(_plus(op(x, i) $ i = trunc(nops(x)/2)+1 .. nops(x)), options);
      else 
        [c1, num1, den1]:= rnormal(op(x, 1), options);
        [c2, num2, den2]:= rnormal(op(x, 2), options);
      end_if:

      if iszero(c2) then 
         return([c1, num1, den1]);
      end_if;
      if iszero(c1) then 
         return([c2, num2, den2]);
      end_if;
  
      // ==========================================================
      // From here, we are dealing with a sum x1 + x2 of two terms
      // x1 = c1*num1/den1, x2 = c2*num2/den2. We need to make the
      // numerator and the denominator of x1 + x2 coprime.
      // ==========================================================
  
      // Special case:
      if den1 = den2 then

         x:= myexpand(c1*num1 + c2*num2);

         if options["Squarefree"] and
            (vars:= indets(x)) <> {} then
           y:= poly(x, [op(vars)]);
           x:= expr(polylib::sqrfree(y));
         end_if:

         [c1, num1]:= extractContent(x);
         if not options[NoGcd] and
            indets(num1) intersect indets(den1) <> {} then
            GCD:= mygcd(num1, den1);
            if GCD <> 1 and GCD <> -1 then
               num1:= exactdivide(num1, GCD);
               den1:= exactdivide(den1, GCD);
            end_if;
         end_if;
         return([c1, num1, den1]);
      end_if;
  
      // =======================================================
      // Step A: in c1*num1/den1 + c2*num2/den2, find all common 
      // factors in den1, den2 and extract it as 'commondenom'.
      // =======================================================
  
      [d1, d2]:= _split(den1/den2); //den1/den2 = d1/d2
      commondenom:= exactdivide(den1, d1);
  
      // Now, den1 = d1*commondenom, den2 = d2*commondenom
  
      assert(myexpand(den1 - d1*commondenom) = 0);
      assert(myexpand(den2 - d2*commondenom) = 0);
  
      // ==========================================================
      // Step B: in c1*num1/den1 + c2*num2/den2, find all common
      //         factors in num1, num2 and extract it as 'commonnumer'.
      // ==========================================================
      [n2, n1]:= _split(num2/num1);        //num2/num1 = n2/n1, num1 = n1*commonnumer, num2 = n2*commonnumer
      commonnumer:= exactdivide(num2, n2); // num2 = n2*commonnumer, num1 = num2*n1/n2 = n1*commonnumer

  /*
      if not options[NoGcd] then
         if indets(n1) intersect indets(n2) <> {} then 
            GCD:= mygcd(n1, n2): 
         else
            GCD:= 1;
         end_if;
         if GCD <> 1 and GCD <> -1 then 
            n1:= exactdivide(n1, GCD);
            n2:= exactdivide(n2, GCD);
            commonnumer:= commonnumer*GCD;
         end_if;
      end_if:
  */
  /*
      // If no common numerator is to be extracted, just replace the
      // lines above by the following 2 lines:
      [n2, n1]:= [num2, num1];
      commonnumer:= 1;
  */

      assert(_split(commonnumer)[2] = 1):
      // x = c1*num1/den1 + c2*num2/den2 
      //   = (c1*n1/d1 + c2*n2/d2)*commonnumer/commondenom
      //   = (c1*n1*d2 + c2*d1*n2)*commonnumer/(d1*d2*commondenom)     
  
      if not options[NoGcd] then
         if indets(d1) intersect indets(d2) <> {} then 
            GCD:= mygcd(d1, d2): 
         else
            GCD:= 1;
         end_if;
         if GCD <> 1 and GCD <> -1 then 
            d1:= exactdivide(d1, GCD);
            d2:= exactdivide(d2, GCD);
            commondenom:= commondenom*GCD;
         end_if;
      end_if:
  
      assert(myexpand(den1 - d1*commondenom) = 0);
      assert(myexpand(den2 - d2*commondenom) = 0);
  
      // =================================================
      // compute the numerator x of the result
      //   commonnumer/commondenom * (c1*n1/d1 + c2*n2/d2)
      // =================================================
      x:= myexpand(c1*n1*d2 + c2*d1*n2);

      if options["Squarefree"] and
         (vars:= indets(x)) <> {} then
        y:= poly(x, [op(vars)]);
        x:= expr(polylib::sqrfree(y));
      end_if:

      // =================================================
      // cancel the gcd of x = c1*n1*d2 + c2*d1*n2 and
      // commondenom:
      // =================================================

      [c, x]:= extractContent(x);
      if iszero(x) then
         return([c, 0, 1]);
      end_if:
  
      // let the kernel do cancellations of common
      // factors in the new numerator x and commondenom:
      [x, commondenom]:= _split(x/commondenom);

      if not options[NoGcd] then
         if indets(x) intersect indets(commondenom) <> {} then
            GCD:= mygcd(x, commondenom);
         else
            GCD:= 1;
         end_if;
         if GCD <> 1 and GCD <> -1 then 
            x:= exactdivide(x, GCD);
            commondenom:= exactdivide(commondenom, GCD);
         end_if;
         assert(indets(mygcd(x, commondenom)) = {});
      end_if:

      return([c, x*commonnumer, d1*d2*commondenom]);
    end_case; // case type(x) of "_plus"
  
    // =========================
    // normalization of products
    // =========================
  
    if type(x) = "_mult" then
  
     // First, normalize each factor of the product.
     // Keep the factors apart by putting them into
     // a list: normalization may have turned each
     // factor into a rational expression. We want
     // to make use of the fact, that numerators and
     // denominators of each factor are coprime:
  
     x:= map([op(x)], rnormal, options);
  
     // Now x = [[c1, n1, d1], [c2, n2,d2], ...] representing 
     //    x = (c1*n1/d1)*(c2*n2/d2)*....
     // Here n1 and d1 are coprime, the same holds for n2 and d2 etc. 
     // However, 
     // n1 is not necessarily coprime to d2, d3, ... etc. and
     // d1 is not necessarily coprime to n2, n3, ... etc.
     // We need to cancel the gcds:
  
  /* 
     // The following lines of code would make denominators look
     // real by multiplying both numerator and denominator with
     // the 'complex conjugate' of the denominator. This may look
     // nice, but it is terribly inefficient because the complexity
     // of numerator and denominator usually increases (apart from 
     // very few cases such as (x^2 + 1)/(x + I) --> x - I).
     if has(x, I) then
        x:= map(x, proc(L) 
                   local dbar;
                   begin
                     if not has(L[3], I) then
                        return(L);
                     end_if;
                     dbar:= subs(L[3], I = -I, EvalChanges);
                     L[2]:= L[2]*dbar;
                     L[3]:= myexpand(L[3]*dbar);
                     return(L):
                   end_proc);
     end_if;
  */
  
     // The content of the product
     c:= _mult(x[i][1] $ i = 1..nops(x)); 
  
     // remove the contents from the factors of x:
     x:= map(x, subsop, 1 = null());
  
     // =======================================
     // Now, cancel all GCDs from all factors
     // of the numerators and the denominators:
     // =======================================
     if not options[NoGcd] then
       for i from 1 to nops(x) do
         for j from 1 to nops(x) do
           if i = j then 
              // the i-th numerator and the j-th denomionator are coprime
              next; 
           end_if;
           [n1, d1]:= x[i];
           [n2, d2]:= x[j];
           if n1 <> 1 and n1 <> -1 and
              d2 <> 1 and d2 <> -1 then 
              GCD:= mygcd(n1, d2);
              if GCD <> 1 and GCD <> -1 then
                 n1:= exactdivide(n1, GCD);
                 d2:= exactdivide(d2, GCD);
                 x[i][1]:= n1: 
                 x[j][2]:= d2: 
              end_if;
           end_if;
         end_for:
       end_for:
     end_if;

     if not options[NoGcd] then
       x:= [_mult(op(x[i][1] $ i = 1..nops(x))),
            _mult(op(x[i][2] $ i = 1..nops(x)))];
     else
       // multiply all numerators and denominators
       // and let the kernel do cancellations
       x:= _split(_mult(op(x[i][1] $ i = 1..nops(x)))/
                  _mult(op(x[i][2] $ i = 1..nops(x))));
     end_if;
  
     return([c, x[1], x[2]]):
    end_if;

    if options[Rationalize] = None or
       // heuristic test if rationalization was switched 
       // off in some way:
       [myrationalize(#X^(1/2))][1] = #X^(1/2) then
       myrationalize:= normal::rationalize;
    else
       error("unexpected expression in 'normal'"):
    end_if;
  
  end_proc:  // end of recursive utility rnormal


    // =====================================================
    // [numer, denom]:= _split(x) computes numerator and
    // denominator of a rational expression x that is the
    // quotient of products of powers of polynomial expressions. 
    // One has x = numer/denom.
    // =====================================================
    _split:= proc(x)
       local L;
       begin
         if type(x) = "_mult" then
           L:= split(x, proc(x) begin
                         if type(x) = "_power" and
                            op(x, 2) < 0 then
                              FALSE
                         else TRUE
                         end_if
                        end_proc);
           return([L[1], 1/L[2]]);
         elif type(x) = "_power" then
           if op(x, 2) >= 0 then
              return([x, 1])
         else
              return([1, 1/x])
           end_if;
         else
           return([x, 1]);
         end_if;
       end_proc;
  

    // ====================================================
    // utility to compute divide(p, q, Quo) for polynomial
    // expressions p, q (with q dividing p). In contrast to
    // the system's 'divide', this utility tries to avoid the 
    // expansion implied by converting the arguments to DOM_POLY
    // objects:
    // ====================================================
    exactdivide:= proc(p, q)
    local i, d, OK, m, n, GCD;
    option remember;
    begin
      if q = 1 then 
         return(p);
      elif q = -1 then 
         return(-p);
      elif p = q then
         return(1);
      elif p = -q then
         return(-1);
      end_if;

      if indets(q) = {} then
         return(p/q);
      end_if;

      if (type(p) = "_mult" or type(p) = "_power") then
         d:= p/q;     // let the kernel do the division
         OK:= TRUE;
         // check that there are no factors
         // with negative exponents:
         misc::maprec(d, {"_power"} = proc(x)
                          begin
                            if indets(op(x, 1)) <> {} and
                               domtype(op(x, 2)) = DOM_INT and
                               op(x, 2) < 0 then
                                 OK:= FALSE;
                                 misc::breakmap();
                            end_if;
                            x;
                          end_proc);
         if OK then
            return(d);
         end_if;
         // common factors in p, q were cancelled.
         [p, q]:= _split(d);
      end_if;

      if type(q) = "_power" then
         [q, n]:= [op(q)];
         if type(p) = "_power" then
            [p, m]:= [op(p)];
         else
            m:= 1;
         end_if:
         // Compute p^m / q^n
         if m = n then
            return( exactdivide(p, q)^n );
         elif m > n then 
            GCD:= gcd(p, q);
            p:= exactdivide(p, GCD);
            q:= exactdivide(q, GCD);
            // p^m*GCD^m / (q^n*GCD^n) and p, q are coprime
            return(p^m*exactdivide(GCD^(m-n), q^n))
         else // 1 <= m < n
            // the division p^m/q^n with n > m can only
            // be exact if p = a^r*b^n with q dividing b,
            // i.e.,  p^m / q^n = (p/q^(n/m))^m 
            if m > 1 and (n mod m = 0) then
               return(exactdivide(p, q^(n div m))^m);
            end_if;
            // Compute p^m/q^n
            if ceil(n/m) < n then // i.e., m > 1
              d:= q^ceil(n/m);
              GCD:= gcd(p, d);
              p:= exactdivide(p, GCD);
              d:= exactdivide(d, GCD);
              return(exactdivide(p, d)^m * q^(ceil(n/m)*m - n));
            end_if;
            if n < 2*m then // i.e., n-m < n/2
              GCD:= gcd(p, q);
              p:= exactdivide(p, GCD);
              q:= exactdivide(q, GCD);
              // (p^m*GCD^m) / (q^n*GCD^n) = p^m/GCD^(n-m)/q^n
              // The result is an exact poly and p, q are coprime.
              // This can only be if q = 1:
              if indets(q) = {} then // p^m / GCD^(n-m)
                return(exactdivide(p^m, GCD^(n-m))/q^n);
              end_if:
              return(exactdivide(exactdivide(p^m, GCD^(n-m)), q^n) );
            else
              // For exactdivide( (x^1001*y + x^1234), x^1000),
              // the code above with m = 1, n = 1000 and GCD = q
              // would recursively reduce n by 1 and end up in
              // an 'recursive definition' error. Proceed to the
              // 'divide by polys' code below:
              // undo the previous splitting into base and exponent
              // and proceed.
              p:= p^m;
              q:= q^n;
            end_if;
         end_if;
      end_if;

      if type(p) = "_power" then  // type(q) <> "_power" 
         [p, m]:= [op(p)];
         GCD:= gcd(p, q);
         if GCD <> 1 and GCD <> -1 then
            p:= exactdivide(p, GCD);
            q:= exactdivide(q, GCD);
            // p^m*GCD^m / (q*GCD)
            return(p^m*exactdivide(GCD^(m-1), q));
         else
            p:= p^m;
         end_if;
      end_if;
      // There were some divisions left. E.g. for
      //   p = expand((x+1)*(x+2)), q = (x+1).
      // In this case, we do a polynomial expansion using divide.
      // However, we do not want to loose factors that are not 
      // involved in the exact division! E.g., the division of
      // p=expand((x+1)*(x+2))*(x+3) by q=x+1 should still contain
      // x+3 as a separate factor.
      if type(p) = "_mult" then
         for i from 1 to nops(p) do
             d:= divide(op(p, i), q, Exact);
             if d <> FAIL then
                return(subsop(p, i = d));
             end_if;
         end_for;
      end_if;
      return(divide(p, q, Quo));
    end_proc:
  

  // =========================================
  // =========================================
  // =========================================
  //    main of normalNoExpand
  // =========================================
  // =========================================
  // =========================================

  if args(0) = 2 and type(args(2)) = DOM_TABLE then
    options:= table(args(2), 
                    "Squarefree" = FALSE, // additional option for rnormal
                    "Prefactor" = FALSE // additional option for rnormal
                   );
    myrationalize:= options[Rationalize];
  else
    // allow input normalNoExpand(f, option_table, List)
    options:= prog::getOptions(2,                     // start scanning for options with the 2nd argument
                               [args()],              // all arguments of the call
                               table(                 // table of default values
                                 normal::defaultOptions,
                                 "Squarefree" = FALSE, // additional option for rnormal
                                 "Prefactor" = FALSE // additional option for rnormal
                               ),
                               TRUE,                  // react to unexpected arguments with an error?
                               table(
                                 normal::optiontypes, // table of expected option types
                                 "Squarefree" = DOM_BOOL,
                                 "Prefactor" = DOM_BOOL
                               )
                              )[1];
    // option ToCancel implies NoGcd
    if options[ToCancel] <> FAIL then
      options[NoGcd]:= TRUE
    end_if:
    myrationalize:= options[Rationalize];
  end_if;
  if type(myrationalize) = DOM_IDENT then
     // some procedures have default names
     case myrationalize
      of None do
      //myrationalize:= normal::rationalizeNone; 

      // In normal, we can get away without rationalization, 
      // since gcd does it, anyway. In normalNoExpand, we **must** 
      // do the rationalization.
        myrationalize:= normal::rationalize;

        break
      otherwise
        error("Unknown method in option Rationalize")
      end_case
  end_if;


  // map into the most important containers
  if domtype(x) = DOM_SET or
     domtype(x) = DOM_LIST or
     domtype(x) = DOM_ARRAY or
     domtype(x) = DOM_TABLE then
     return(map(x, stdlib::normalNoExpand, options));
  elif (domtype(x))::hasProp(Cat::Matrix) = TRUE then
     return((domtype(x))::mapNonZeroes(x, stdlib::normalNoExpand, options));
  end_if:

  if stdlib::hasfloat(x) then
    floatmode:= TRUE;
    x:= numeric::rationalize(x);
  else
    floatmode:= FALSE:
  end_if;

  // Preprocessing: rationalize the expression

  // take care of stuff like y^(n+1)/y^n that rationalize
  // would turn into Y1/Y2, {Y1 = y^(n+1), Y2 = y^n} if n
  // is a symbol. We need to expand powers before 
  // rationalization:
  undo:= FALSE;
  x:= misc::maprec(x, {"_power"} = proc(x) begin
          if iszero(op(x, 1)) then
            // do not expand 0^(n-2) --> 0^k/0^2 (Error)
            return(x);
          elif type(op(x, 2)) = "_plus" then
            // Do not expand integer powers of sums, apply
            // expand(..,ArithmeticOnly) only to the other
            // parts of the product: 
            // (a+b)^(2+y1+y2) -->  (a+b)^2*expand((a+b)^(y1+y2))
            [t, y]:= split(op(x, 2), testtype, DOM_INT)[1..2]; 
            return(op(x, 1)^t * expand(op(x, 1)^y, ArithmeticOnly));
          elif domtype(op(x, 2)) = DOM_COMPLEX then
            undo:= TRUE; // the substitution I = `#X_I` needs to be undone later
            return(op(x, 1)^Re(op(x, 2)) * op(x, 1)^(Im(op(x, 2))*`#X_I`));
          elif domtype(op(x, 2)) = DOM_EXPR then 
            // take care of y^((n+1)/n) --> y*y^(1/n)
            return(expand(x, ArithmeticOnly));
          end_if;
          x
          end_proc);
  substitutions := [myrationalize(x)]; // use the same rationalize options as normal!
  if undo then
    substitutions := subs(substitutions, `#X_I` = I);
  end_if;
  case nops(substitutions) 
  of 2 do
    [x, substeqs]:= substitutions;
    minpolys:= {}:
    break;
  of 3 do
    [x, substeqs, minpolys]:= substitutions;
    break;
  otherwise
    substeqs:= {};
    minpolys:= {};
  end_case;

  //=====================================================
  // Do the normalization!
  x:= rnormal(x, options); // x = [content, numer, denom]
  //=====================================================

  // Postprocessing:

  if options[Recursive] then
     // Apply the normalization to the operands of the
     // irrational subexpressions found by myrationalize.
     // E.g., in substeqs = {X1 = sin(x/(x+y)+y/(x+y)-1)}, 
     // we need to apply normalNoExpand to the operands of the sin 
     // expression: {X1 = 0)}
     options2:= options;
     options2[List]:= FALSE;
     substeqs2:= map(substeqs, eq -> (op(eq, 1) = 
                   if domtype(op(eq, 2)) = DOM_EXPR then
                      map(op(eq, 2), stdlib::normalNoExpand, options2);
                   else
                      op(eq, 2)
                   end_if));
     if substeqs2 <> substeqs then
        substeqs:= eval(substeqs2)
     else
        substeqs:= substeqs2;
     end_if;
  end_if;

  // =======================================================
  // undo the substitutions introduced by myrationalize
  // =======================================================
  // For x = (X+1/X)/(X-1/X) = (X^2+1)/(X^2-1) with X = sqrt(a/b) 
  // we would get x = (a/b+1)/(a/b-1) after the re-substitution of 
  // the irrational subexpressions. This is not a normalized result! 
  // We have to do the normalization again to produce (a + b)/(a - b)! 
  // Try options[Iterations] of such steps:

  oldresults:= {}:

  mypower:= proc(x, n : DOM_INT)
  begin 
    if type(x) = "_power" and
       domtype(n*op(x, 2)) = DOM_INT then
       doiterate:= TRUE;
    end_if;
    return(x^n);
  end_proc:

  for counter from 0 to options[Iterations] do
    if substeqs = {} then 
      break;
    end_if; 

    substexprs:= map(substeqs, op, 2): // the irrational sub-expressions
    doiterate:= bool(nops(minpolys) > 0); // default of the flag to be set by 'mypower'

    if nops((t:= select(substeqs, x -> (op(x, 2) = I)))) > 0 then 
       x:= subs(x, op(t[1], 1) = I, EvalChanges);
       if has(x, I) then
          GCD:= gcd(x[2], x[3], "Rationalized");
          if GCD <> 1 then 
            x[2]:= exactdivide(x[2], GCD);
            x[3]:= exactdivide(x[3], GCD);
          end_if;
       end_if;
    end_if;

    if contains(map({op(substexprs)}, 
                        x -> _lazy_and(type(x) = "_power",
                                       domtype(op(x, 2)) = DOM_RAT
                             )),
                    TRUE) then
       // There is a substitution of the form X = (some expr)^(p/q) which 
       // can disappear when the q-th power is computed. We need to check
       // via 'mypower' if this really happens (and set the flag 'doiterate')
       x:= eval(subs(subs(x, hold(_power) = mypower), substeqs)); 
    else
       x:= subs(x, substeqs, EvalChanges):
    end_if;

    if length(x) > 10^5 then
       break;
    end_if;
    if counter = options[Iterations] or
       (not doiterate) or
       contains(oldresults, x) then
       break;
    end_if;
    oldresults:= oldresults union {x};
    substitutions := [myrationalize(x)];
    case nops(substitutions)
    of 2 do
      [y, substeqs]:= substitutions;
      minpolys:= {}:
      break;
    of 3 do
      [y, substeqs, minpolys]:= substitutions;
      break;
    otherwise
      y:= x;
      substeqs:= {};
      minpolys:= {};
    end_case;

    // If some of the right hand sides in substeqs contain some of the
    // newly generated variables, this loop would not terminate. So 
    // make sure that the variables on the left hand sides of the 
    // substeqs do not turn up on the right hand sides:
  
    substeqs:= map(substeqs, eq -> (op(eq, 1) = subs(op(eq, 2), op(substeqs))));

    // Do the iterated normalization:
    y:= rnormal(y[1]*(y[2]/y[3]), options);
    if nops(minpolys) > 0 then
       y:= property::normalGroebner(y, [op(minpolys)]);
       GCD:= gcd(y[2], y[3]);
       if GCD <> 1 and GCD <> -1 then
          y[2]:= exactdivide(y[2], GCD);
          y[3]:= exactdivide(y[3], GCD);
       end_if;
    end_if;
    x:= y;
  end_for: // for counter from 1 to options[iterations]

  // ==============================================
  if floatmode then
    x:= map(x, float):
  end_if:

  // ==============================================
  y:= table():
  if options[ToCancel] <> FAIL then
     for t in options[ToCancel] do
       while traperror((y[2]:=divide(x[2], t, Exact))) = 0 and 
             y[2] <> FAIL and
             traperror((y[3]:=divide(x[3], t, Exact))) = 0 and 
             y[3] <> FAIL do
             // t was succesfully cancelled both from numerator and denominator
             x[2]:= y[2];
             x[3]:= y[3];
       end_while
     end_for;
  end_if;

  if options[List] then
     // ==============================================
     // move exponentials from the denominator to the numerator
     if type(x[3]) = "exp" then
        [x[2], x[3]] := [x[2]/x[3], 1];
     elif type(x[3]) = "_mult" then
      [exps, x[3]]:= split(x[3], testtype, "exp")[1..2];
      x[2]:= x[2]/exps;
     end_if;

     if domtype(x[1]) = DOM_RAT then 
        return([op(x[1],1)*x[2], op(x[1],2)*x[3]]);
     else
        return([x[1]*x[2], x[3]]);
     end_if;
  end_if;

  return(x[1]*(x[2]/x[3]));
end_proc:

stdlib::normalNoExpand:= prog::remember(stdlib::normalNoExpand, DIGITS):
