/* ------------------------------------------------------------------ 
stdlib::normalizesign(x) -- heuristic normal form of arithmetical expressions

Call:        stdlib::normalizesign(x) 
Parameter:   x -- an object of type Type::Arithmetical
ReturnValue: [s, y] with either s = 1 or s = -1

Details: 

  o Mathematically, x = s*y.

  o This functions serves for normalizing x and -x to
    the same "normal form" y.

    Indeed, it is guaranteed (is it really?), 
    that x and -x yield the same normal form, i.e,
    if   stdlib::normalizesign(x)  = [ s, y], 
    then stdlib::normalizesign(-x) = [-s, y].

  o The "normal form" y is 'heuristically simpler'
    than x w.r.t. to the occurences of minus signs

  o Drastic simplifications can happen! E.g.,

    stdlib::normalizesign( (a-b)^3/(b-a)^2 )  --> [1, a - b]

  o This is not a proper normal form in the sense
    that stdlib::normalizesign(y) is again y!

    >> z := -a - (x-y)/(y-x)*(u-v)
                                 (u - v) (x - y)
                           - a - ---------------
                                      y - x
    >> stdlib::normalizesign(z)
                              [1, u - a - v]
    >> stdlib::normalizesign(%[2])
                              [-1, a - u + v]


Examples:

2.0.0 > stdlib::normalizesign( x)
                            [1, x]
2.0.0 > stdlib::normalizesign(-x)
                            [-1, x]
2.0.0 > stdlib::normalizesign(1+x)
                          [1, x + 1]
2.0.0 > stdlib::normalizesign(-x-1)
                          [-1, x + 1]
2.0.0 > stdlib::normalizesign(x-1)
                          [-1, x - 1]
2.0.0 > stdlib::normalizesign(1-x)
                          [1, x - 1]
2.0.0 > stdlib::normalizesign(a-b)
                          [1, a - b]
2.0.0 > stdlib::normalizesign(b-a)
                          [-1, a - b]
2.0.0 > stdlib::normalizesign((a-b)^2)
                                    2
                         [1, (a - b) ]
2.0.0 > stdlib::normalizesign((b-a)^2)
                                    2
                         [1, (a - b) ]
2.0.0 > stdlib::normalizesign((a-b)^3)
                                    3
                         [1, (a - b) ]
2.0.0 > stdlib::normalizesign((b-a)^3)
                                    3
                        [-1, (a - b) ]
2.0.0 > x:= -(a-b)^3 + (b-a)^2 - sqrt(PI)*I:
2.0.0 > stdlib::normalizesign( x);
                          3          2       1/2
              [-1, (a - b)  - (a - b)  + I PI   ]
2.0.0 > stdlib::normalizesign(-x);
                          3          2       1/2
              [ 1, (a - b)  - (a - b)  + I PI   ]

----------------------------------------------------------*/

stdlib::normalizesign:= proc(x)
local tmp, s, ss, result, term, finished;
begin
   if testargs() then
      if not testtype(x, Type::Arithmetical) then
         error("argument is not of type 'Type::Arithmetical'");
      end_if;
   end_if;

   repeat
    finished := TRUE;

    case type(x)

    of "_power" do 
      // recursively map stdlib::normalizesign to the base
      // to make sure that also all operands of the
      // base are stdlib::normalized.
      // Split op(x, 1) = tmp[1]*tmp[2]:
      tmp:= stdlib::normalizesign(op(x,1));

      result := (tmp[1] * tmp[2])^op(x, 2);
      if type(result) <> "_power" then
        // some simplification has occurred, repeat computations
        x := result;
        finished := FALSE;
        next;
      end_if;

      // check whether the sign tmp[1] can be omitted:
      if is(op(x,2), Type::Even)=TRUE then
         return([ 1, tmp[2]^op(x,2)])
      end_if:
      if is(op(x,2), Type::Odd)=TRUE then
         return([tmp[1], tmp[2]^op(x,2)])
      end_if:
      // power is unknown, cannot simplify.
      // Use tmp[1]*tmp[2] instead of op(x,1)
      // to garantee that the base is normalized:
      return([1, result]);

    of "_plus" do
      // first, normalize all terms in the sum:
      s:= 0:
      result:= 0:
      for term in x do
          tmp:= stdlib::normalizesign(term):
          // count number of minussigns:
          if tmp[1] = -1 then s:= s + 1; end_if;
          result:= result + tmp[1]*tmp[2];
      end_for;

      if type(result) <> "_plus" or nops(result) < nops(x) then
         // some cancellation has happened, repeat computations
         x := result;
         finished := FALSE;
         next;
      end_if;

      // now decide, whether the normal
      // form is result or -result.
      // If there are more plus signs than
      // minus signs, then result is the 
      // "simpler" than -result:
      ss:= 1;
      if s > nops(result)/2 then ss:= -1 end_if;

      // If there are exactly as many + and - signs
      // then use sysorder to make sure that
      // result and -result yield the same normal form.
      // Note that if type(result) = "_plus", then
      // also type(-result) = "_plus" and the terms 
      // in -result are -terms in result:
      if s = nops(result)/2 then
         if stdlib::hasmsign(result) then ss:= -1 end_if;
      end_if:
      return([ss, ss*result]);

    of "_mult" do 
      // normalize all terms and collect the signs:
      s:= 1;
      result:= 1:
      for term in x do
          tmp:= stdlib::normalizesign(term):
          s:= s * tmp[1]:
          result:= result * tmp[2];
      end_for;

      if type(s * result) <> "_mult" or nops(s * result) < nops(x) then
         // some cancellation has happened, repeat computations
         x := s * result;
         finished := FALSE;
         next;
      end_if;

      return([s, result]);
 
   // of ???? do ???
   // We could treat other expressions such as
   // functions calls etc. as well.
   // However, instead of mapping stdlib::normalizesign
   // into such sub expressions we decide to treat
   // them as leaves of the expression tree and
   // do not normalize them.
    end_case;

   until finished end_repeat;


   // The normal form of the leaves of the expression
   // tree should have no unnecessary minus signs.
   // Use stdlib::hasmsign to test for minus signs:
   if stdlib::hasmsign(x) 
   then return([-1, -x])
   else return([ 1,  x])
   end_if;

end_proc:
//-------------- end of file ----------------
