/* -----------------------------------------------------------
pade - pade approximation

Call: pade(f(x), x, <[m, n]>)
      pade(f(x), x = x0, <[m, n]>)

Parameters:  f(x) -- expression in x or Series::Puiseux
             x    -- DOM_IDENT
             x0   -- Type::Arithmetical, default x0 = 0 
             m, n -- integers >= 0, default m = n = 3 

Details: 
    pade(f(x), x = x0, [m, n])

      =  (x-x0)^p * (a0+a1*(x-x0)+..+a.m*(x-x0)^m)
                   /( 1+b1*(x-x0)+..+b.n*(x-x0)^n)

   such that series(P, x=x0) = series(f(x), x=x0)
   to maximal order.

Example:
>> pade(exp(x), x, [3, 2])
                               2    3
                     36 x + 9 x  + x  + 60
                     ---------------------
                          2
                       3 x  - 24 x + 60

------------------------------------------------------------*/

pade:= proc(ff, xx, mn)
local x, x0, f, ldeg, m, n, p, q,
      a, b, i, j, solb, unk, eq;
begin
  if args(0)<2 then
    error("expecting at least 2 arguments");
  end_if;

  if args(0)>3 then
    error("expecting at most 3 arguments");
  end_if;

  //---- check the first argument -------------
  if (not testtype(ff, Type::Arithmetical)) and
     (domtype(ff) <> Series::Puiseux) then
     error("expecting an arithmetical expression or a ".
           "series of domain type Series::Puiseux");
  end_if;

  //---- check the second argument -------------
  if type(xx) = "_equal" then
       x:= op(xx, 1);
      x0:= op(xx, 2);
      if not testtype(x0, Type::Arithmetical)
         or has(x0, infinity) or has(x0, complexInfinity) then
         error("illegal expansion point");
      end_if;
  else x:= xx;
      x0:= 0;
  end_if:

  if domtype(x) <> DOM_IDENT then
     error("no expansion variable");
  end_if:

  //---- check the third argument -------------
  if args(0)<3 then
        [m, n]:= [3, 3]; // default
  else if domtype(mn) <> DOM_LIST or nops(mn)<>2 then
          error("expecting a list [m, n] as 3rd argument");
       end_if:
       [m, n]:= mn; 
       if domtype(m) <> DOM_INT or m < 0 then
          error("orders must be nonnegative integers"):
       end_if:
       if domtype(n) <> DOM_INT or n < 0 then
          error("orders must be nonnegative integers"):
       end_if:
  end_if;

  //----------------------------
  // Compute series expansion:
  //----------------------------
  mn:= n+m;

  if domtype(ff) = Series::Puiseux 
  then f:= ff;
       ldeg:= f::dom::ldegree(f);
  else 
       // computing leading order and sufficiently
       // many terms of the series expansion. 
       // First try:
       f:= series(ff, xx, mn + 2);

       if domtype(f) <> Series::Puiseux then
          userinfo(5, "could not compute series expansion");
          return(FAIL)
       end_if;

       ldeg:= f::dom::ldegree(f);

       // check that we have sufficiently many terms in the
       // series expansion.
       if ldeg <> FAIL 
          and f::dom::order(f) <> FAIL
          and f::dom::order(f) < ldeg + mn
          then
          f:= series(ff, xx, mn + 2 + ldeg+mn-f::dom::order(f));
          if domtype(f) <> Series::Puiseux then
             userinfo(5, "could not compute series expansion");
             return(FAIL)
          end_if
       end_if:

       if ldeg = FAIL then
          // ldeg = FAIL may happen for series(x^8, x, 5) -> O(x^8)
          // Try again using the kernel's ldegree:
          ldeg:= ldegree(subs(ff, x = x+x0), x):
          if ldeg = FAIL then
             userinfo(5, "could not compute leading order");
             return(FAIL);
          end_if;
          if ldeg + mn + 2 <= 1 then
             userinfo(5, "could not compute leading order");
             return(FAIL);
          end_if;
          f:= series(ff, xx, ldeg + mn + 2);
          if domtype(f) <> Series::Puiseux then
             userinfo(5, "could not compute series expansion");
             return(FAIL)
          end_if;
          ldeg:= f::dom::ldegree(f);
          if ldeg = FAIL then
             userinfo(5, "could not compute leading order");
             return(FAIL);
          end_if;
       end_if:
  end_if;

  if domtype(ff) <> Series::Puiseux and
     f::dom::order(f) < ldeg + mn then
     userinfo(5, "could not determine series expansion");
     return(FAIL)
  end_if:

  if x <> f::dom::indet(f) then 
     userinfo(5, "expansion variables do not match");
     return(FAIL);
  end_if:

  if x0 <> f::dom::point(f) then
     userinfo(5, "expansion points do not match");
     return(FAIL);
  end_if;

  // ---------------------------------------------------
  // End of argument check and preparation. Now the work
  // starts:
  // The following code could be easily modified to 
  // generate Pade-approximants in terms of x^(1/bo),
  // where bo = extop(f, 2) is the branch order.
  // However, we keep things simple and compute
  // Pade approximants only for proper Laurent 
  // expansions, i.e., bo = extop(f, 2) = 1:
  // Moreover, we do not allow flag = 1 or
  // coefficients depending on x.

  if not testtype(f, Type::Series(Laurent)) then
     userinfo(5, "this is not a Laurent series");
     return(FAIL);
  end_if;
  
  // extract the coefficients of the series and store them in
  // a list ff = [coeff(f, ldeg), .., coeff(f, ldeg + m + n)]
  // First, the list of all known coefficients:
  ff:= [f::dom::coeff(f)];

  // if f is a series with an order less than the
  // requested Pade order n+m, then coeff returns
  // FAIL beyond the truncation term. Compute the
  // Pade approximand of the truncated series, i.e.,
  // pad f with zeroes.
  // No need to worry, if nops(ff) > mn + 1. Only
  // the first mn+1 entries will be called lateron.
  if nops(ff) < mn+1 then
     ff:= [op(ff), 0 $ mn + 1 - nops(ff)];
  end_if;

  // use b[i] as indeterminates in Gaussian elimination
  b[0]:=1;
  for i from 1 to n do
    b[i]:= genident("b")
  end_for;

  // Consider (with x == (x-x0))
  //    x^ldeg*(f0+f1*x+...+f.(m+n)*x^(m+n))*(1+b1*x+...+bn*x^n)
  //  = x^ldeg*(a0+a1*x+...+am*x^m) 
  //
  // The orders 0..m of (f0+f1*x+..)*(1+b1*x+..) 
  // just define a[0]..a[m]:
  for i from 0 to m do
    a[i]:=_plus(ff[j+1]*b[i-j] $ j= max(0,i-n)..i)
  end_for;

  // The orders m+1..m+n of (f0+f1*x+..)*(1+b1*x+..) 
  // provide a set of n linear equations for b[1]..b[n]

  // Warning: this system may be degenerate. The following
  // code does Gaussian elimination and stops automatically,
  // when the maximal approximation order is reached.

  unk:={b[j] $ j=1..n};
  solb:=null();
  for i from m+1 to m+n do
    eq:=subs(_plus(ff[j+1]*b[i-j] $ j=max(0,i-n)..i), solb);
    if eq = 0 then next end_if;
    // solve for one of the unknowns in eq:
    p:=indets(eq,PolyExpr) intersect unk;
    if p = {} then
       break; // maximal approximation order reached
    end_if;
    p:= op(p, 1);
    unk:= unk minus {p};
    solb:= solb, p = -coeff(eq,[p],0)/coeff(eq, [p], 1);
  end_for:

  // do not(!) subsitute simultaneously via subs(.., [solb]),
  // but do a systematic back substitution with a sequential
  // subs(..,solb)!
  p:= subs(_plus(   a[i]*(x-x0)^i $ i=0..m), solb, EvalChanges);
  q:= subs(_plus(1, b[i]*(x-x0)^i $ i=1..n), solb, EvalChanges);
  if iszero(q) then
     return(FAIL) 
  end_if:
  normal((x-x0)^ldeg*p/q);
end_proc:
