//   

/* product(f(k), k) returns g(k) such that g(k+1)/g(k)=f(k)
Examples:
product(k^2,k=1..4);
                                    576

product(1/(k+2),k=RootOf(x^2-1));
                                    1/3

product(k, k in RootOf(x^2 - 4*x + 4));
                                    4

product(k^2,k);
                                         2
                                 gamma(k)
product(a[k],k=1..4);
                            a[1] a[2] a[3] a[4]
product(n+k,k=1..4);
                      (n + 1) (n + 2) (n + 3) (n + 4)
product(n+k,k);
                               gamma(k + n)
product(n+k,k=1..m);
                             gamma(m + n + 1)
                             ----------------
                               gamma(n + 1)
product(2*h(k+1)/h(k),k);

                                        k
                                  h(k) 2
*/
product :=
proc(f, k)
  local S,
  i;
begin
  if args(0) = 0 then
    error("wrong number of arguments")
  end_if;

  if f::dom::product <> FAIL then
    return(f::dom::product(args()))
  elif testtype(f, Type::Arithmetical) = FALSE then
    error("Illegal argument")
  end_if;

  case type(k)
    of DOM_IDENT do
      return(stdlib::prod_indef(f,k))
    of "_index" do
      i:= genident();
      return(subs(stdlib::prod_indef(subs(f, k = i), i), i = k))
    of "_equal" do
      if type(op(k, 1)) = "_index" then
        i:= genident();
        f:= subs(f, op(k, 1)=i);
        k:= subsop(k, 1 = i)
      elif type(op(k, 1)) <> DOM_IDENT then
        error("Left hand side of second argument must be identifier or ".
              "indexed identifier")
      end_if;
      if type(op(k, 2)) = "_range" then
        return(subs(stdlib::prod_def(f, op(k,1), op(op(k,2))),
               `#product` = hold(product), Unsimplified))
      elif type(op(k, 2)) = RootOf then
        return(stdlib::prod_rootof(f, k, op(op(k,2))))
      else
        error("Right hand side of second argument must be range or RootOf")
      end_if;
    of "_in" do
      if type((S:= op(k, 2))) = DOM_SET then
        return(_mult(subs(f, op(k, 1) = op(S, i)) $i=1..nops(S)))
      elif type(op(k,2)) = RootOf then
        return(stdlib::prod_rootof(f, k, op(op(k,2))))
      else
        return(procname(args()))
      end_if
    otherwise
      error("invalid argument")
  end_case
end_proc:

/*

Product over roots of a polynomial (product(f, k = RootOf(p,x))),
where f is a rational expression in k, p a polynomial in x

The resultant R of polynomials f and g satisfies
 R = a_n^m prod_{x_i} g(x_i)
where a_n is the leading coefficient of f, x_i are the roots of f and
g is of degree m.

*/
stdlib::prod_rootof :=
proc(f, k, p, x: Type::Union(DOM_IDENT, "_index"))
local a,b,rp;
begin
  // k is either 'n = RootOf(p, x)' or 'n in RootOf(p, x)'
  if not testtype(f, Type::RatExpr(op(k, 1))) or
     not testtype(p, Type::PolyExpr(x)) then
    return(hold(product)(f, k))
  end_if;

  if degree(p,x) = 0 then
    if is(p <> 0) = TRUE then
      return(1)
    else
      return(hold(product)(f, k))
    end_if;
  end_if;

  rp := subs(p, x= op(k, 1));
  a := stdlib::prod_rootofpoly(numer(f), op(k, 1), rp);
  b := stdlib::prod_rootofpoly(denom(f), op(k, 1), rp);
  return(a/b)
end_proc:

stdlib::prod_rootofpoly :=
proc(f, k, rp)
begin
  if type(f) = "_power" and op(f,1) = k then
    return(((-1)^degree(rp)*coeff(rp,k,0))^op(f,2))
  end_if;

  f := divide(f, rp, [k], Rem);
  return(polylib::resultant(rp,f,k)/(lcoeff(rp, [k])^degree(f,k)))
end_proc:


// indefinite product, k is an identifier
stdlib::prod_indef :=
proc(f: Type::Arithmetical, k: DOM_IDENT)
  local a, b, c, dg, S,
  indef_rec: DOM_PROC,
  productOverSet: DOM_PROC;

begin
  assumeAlso(k in Z_);

  // local method indef_rec
  // input: f, an arithmetical expression
  // returns [a, b] such that product(f, k) = a * b^k
  // this is necessary because MuPAD is unable to collect
  // terms with power k later on :-(

  indef_rec:=
  proc(f)
  begin
    if not has(f, k) then
      return([1, f])
    end_if;

    case type(f)
      of "_power" do
        if not has(op(f, 2), k) then
          // product(f^a, k) = product(f, k)^a
          return(map(indef_rec(op(f, 1)), _power, op(f, 2)))
        elif not has(op(f, 1), k) then
          // Let g:= product(C^f(k), k)
          // then g(k+1)/g(k) = C^f(k)
          // we may write this as
          // ln(g(k+1)) - ln(g(k)) = ln(C)*f(k)    (*)
          // now let h= sum(f(k), k)
          // then f(k) = h(k+1) - h(k)               (**)
          // From (*) and (**), it follows that
          // g= C^h  is just the function we are looking for
          return([op(f, 1)^sum(op(f, 2), k), 1])
        else
          return([hold(product)(f, k), 1])
        end_if;
        // NOT REACHED
        assert(FALSE);
      of "exp" do
        // this has to be handled like the _power case
        return([exp(sum(op(f), k)), 1])
      of "_mult" do
        a := numer(f);
        b := denom(f);
        if not has((c:= a/subs(b, k = k+1)), k) then
          return([b, c])
        elif not has((c := subs(a, k = k+1)/b), k) then
          return(piecewise([a = 0, [0, 0]], [a <> 0, [1 / a, c]]))
        else
          f:= map([op(f)], indef_rec);
          return([_mult(op(map(f,
                               proc(l)
                               begin
                                 if type(l) = piecewise then
                                   piecewise::extmap(l, op, 1)
                                 else
                                   op(l, 1)
                                 end_if
                               end_proc
                               ))),
                  _mult(op(map(f,
                               proc(l)
                               begin
                                 if type(l) = piecewise then
                                   piecewise::extmap(l, op, 2)
                                 else
                                   op(l, 2)
                                 end_if
                               end_proc
                                 )))])
        // toDo: this may be a product of many gammas.
        // therefore, one should apply rules for simplifying
        // gamma(1+y*I) gamma(1-I*y) here, cf. Abramowitz-Stegun
        end_if;
      // NOT REACHED
        assert(FALSE);
    end_case;

    if testtype(f, Type::RatExpr(k)) then
      if (dg:= degree(f, [k])) = 1 then
        a:= coeff(f, [k], 1);
        b:= coeff(f, [k], 0);
      // Computing \prod_k a*k + b
      // we have to distinguish:
      // a=0. Then the result is b^k
      // Otherwise we write the product as \prod_k a (k + b/a)
      // If the argument is not 0 or a negative integer, the
      // result g can be expressed in terms of gamma
      // For the moment, we do not pay respect to that restriction
      //
        return(piecewise([a=0, [1, b]],
                         [a<>0 and not (k+b/a <= 0 and k+b/a in Z_),
                          [gamma(k + b/a), a]],
                          [a<>0 and k+b/a < 0 and k+b/a in Z_,
                           [1/gamma(-k -b/a + 1), -a]]))

      elif contains({"_mult", "_power"},
                    type((a:= Factored::convert_to(factor(f), DOM_EXPR)))) then
        return(indef_rec(a))
      elif dg <> FAIL and dg <= 4 then
      // proceed by solving
        S:= solve(f, k, MaxDegree = 4, IgnoreProperties);

      // local method productOverSet:
      // computes product(f, n) where f(x) =0 iff x \in S

        productOverSet:=
        proc(S)
          local mylcoeff: DOM_PROC, j, Op;
        begin
        // local method mylcoeff
        // leading coefficient with case analysis

          mylcoeff:=
          proc(f)
            local coefflist, i, j;
          begin
            coefflist:= map(poly2list(poly(f, [k])), _index, 1);
            eval(hold(piecewise)([coefflist[i] <>0 and
                                  _and(coefflist[j]= 0 $j=1..i-1),
                                  coefflist[i]] $i=1..nops(coefflist),
                                 [_and(coefflist[j] = 0
                                       $j=1..nops(coefflist)), 0]))
          end_proc;

          Op:=
          proc(x, n)
          begin
            if domtype(x) <> piecewise then
              op(x, n)
            else
              piecewise::extmap(x, op, n)
            end_if
          end_proc:

          case type(S)
            of piecewise do
              return(piecewise::extmap(S, productOverSet))
            of DOM_SET do
              [indef_rec(k-j, k) $j in S];
              return([_mult(op(map(%, Op, 1))), _mult(op(map(%, Op, 2))) *
                      mylcoeff(f)])
            of solvelib::BasicSet do
            // the polynomial has infinitely many roots and must therefore
            // be zero
              return([0, 0])
          end_case;
        // RootOf etc.
          FAIL
        end_proc;

        productOverSet(S);
        if not has(%, FAIL) then
          return(%)
        end_if
      end_if
    end_if;

  // give up:
    [hold(product)(f, k), 1];
  end_proc;


indef_rec(f);
piecewise::extmap(%, _index, 1) * piecewise::extmap(%, _index, 2) ^ k

end_proc:




stdlib::prod_def :=
proc(f: Type::Arithmetical, k: DOM_IDENT, a, b)
  local p, i, g, getProductFromIndefiniteProduct;
begin

  // do not allow numbers in the range that are not integers

  if {domtype(b), domtype(a)} intersect
    {DOM_FLOAT, DOM_RAT, DOM_COMPLEX} <> {} then
    error("Illegal range")
  end_if;

  if testargs() then
    if is(a<=b) = FALSE then
      error("Lower border must not exceed upper border")
    end_if
  end_if;


  // We cannot do this in the usual way since the recursion 0= g(n+1)/g(n) has
  // no solution
  if iszero(f) then
    return(0)
  end_if;

  if not has(f, k) then
    return(f^(b-a+1))
  end_if;


  // fuer grosse Werte von b-a nicht zweckmaessig
  // Regelung analog zu sum

  if type(b-a) = DOM_INT and b-a < 1000 then // finite product
    p:= 1;
   // if b-a >= 0 then
    for i from 0 to b-a do
      p:= p*subs(f, k=a+i, EvalChanges)
    end_for;
   // else
   //   for i from 1 to a-b-1 do
   //     p:= p/eval(subs(f, k=b+i))
   //   end_for
   // end_if;
    return(p)
  end_if;


  // use the
  // information that the product goes through integers between a and b only
  assumeAlso(k in Z_ intersect Dom::Interval([a], [b]));
  f:= simplify(f);

  // use the information that left border <= right border
 
  assumeAlso(a in Z_):
  assumeAlso(b in Z_):
  assumeAlso(a<=b);
 

  g:=stdlib::prod_indef(f, k);
  g:= rewrite(g, fact);

  getProductFromIndefiniteProduct:=
  proc(g)
    local L, S;
  begin
//    if type(g) = piecewise then
//      piecewise::extmap(g, getProductFromIndefiniteProduct)
//    else


      if hastype(g, "product") then
        // try limit heuristic
        if b = infinity then
          L:= limit(f, k= infinity);
          if is(L>-1 and L<1)=TRUE then
            return(0)
          end_if;
          if is(L>1 and f>0)=TRUE then
            // all factors are positive, and almost all > 1+epsilon
            return(infinity)
          end_if;
        end_if;
        if is(f<>0) <> TRUE then
          // can we find a factor that is zero?
          S:= solve(f, k, DontRewriteBySystem, IgnoreProperties);
          // handle only simple cases
          if domtype(S) = DOM_SET then
            S:= S intersect Dom::Interval([a, b]) intersect Z_;
            if solvelib::isEmpty(S) = FALSE then
              return(0)
            end_if;
          end_if;
        end_if;
        `#product`(f, k=a..b)
      else
        // our indefinite product works only if no factor becomes zero
        S:= solve(f, k);

        proc(S)
          local cond;
          begin
            if domtype(S) = DOM_SET then
              S:= S intersect Dom::Interval([a, b]) intersect Z_;
              cond:= solvelib::isEmpty(S);
              if type(cond) <> "isEmpty" then
                return(piecewise([cond, eval(sum::myeval(g, k, b+1)/
                                          sum::myeval(g, k, a))],
                          [not cond, 0]
                                 )
                       )
              end_if
            end_if;
          `#product`(f, k=a..b)
        end_proc;

        if type(S) = piecewise then
          piecewise::extmap(S, %)
        else
          %(S)
        end_if

      end_if;

//    end_if
  end_proc;

  return(getProductFromIndefiniteProduct(g))

end_proc:

product:=funcenv(product,
proc(a)
  local eq, productSign;
begin
  if PRETTYPRINT = TRUE then
    productSign := "-----",
                   " | | ",
                   " | | ":
    if type(op(a, 2)) = "_equal" then
      // definite product
      eq := op(a, [2,2]);
      if domtype(eq) = RootOf then
         _outputSequenceMult(stdlib::Exposed(stdlib::align(productSign,
                                                    expr2text(op(a, [2,1])=eq))),
                                                    stdlib::Exposed(" "), op(a, 1));
      else
        _outputSequenceMult(stdlib::Exposed(stdlib::align(expr2text(op(eq, 2)),
                                                    productSign,
                                                    expr2text(op(a, [2,1])=op(eq, 1)))),
                                                    stdlib::Exposed(" "), op(a, 1));
      end_if;
    else
      // indefinite product
      _outputSequenceMult(stdlib::Exposed(stdlib::align(productSign,
                                      expr2text(op(a, 2)))),
                      stdlib::Exposed(" "), op(a, 1));
     end_if;
  else
    FAIL
  end:
end_proc):
product:=slot(product, "type", "product"):
product:=slot(product, "print", "product"):

product::float := loadproc(product::float, pathname("NUMERIC"), "floatproduct"):

product::diff:=
proc(e)
  local _x, r;
begin
  if args(0) = 1 then
     return(e);
  end_if:
  if map({args(2..args(0))}, testtype, Type::Indeterminate) = {TRUE} then
    _x := op(e, 2);
    if not has(numeric::indets(_x), args(2..args(0))) then
      r:= e*sum(diff(op(e, 1), args(2))/op(e, 1), _x):
      if args(0) = 2 then
        return(r);
      else
        return(diff(r, args(3) .. args(0)));
      end_if:
    end_if:
  end_if;
  hold(diff)(e, args(2..args(0)))
end_proc:

// a utility for patterns
product::product_fn :=
proc(f, range)
local k;
begin
  k := solvelib::getIdent(Z_, indets(f) union indets(range));
  product(f(k), k=range);
end:

product::freeIndets:=
proc(P: "product")
begin
  if type(op(P, 2)) = "_equal" then
    (freeIndets(op(P, 1), args(2..args(0))) union 
     freeIndets(op(P, [2, 2]), args(2..args(0)))
    ) minus
    freeIndets(op(P, [2, 1]))
  else
    freeIndets(op(P, 1), args(2..args(0))) union 
    freeIndets(op(P, 2), args(2..args(0)))
  end_if
end_proc:

product::evalAt:=
proc(P: "product", subst : Type::SetOf("_equal"))
  local notx;
begin
  if type(op(P, 2)) = "_equal" then
    // definite product
    notx := select(subst, s -> op(s, 1) <> op(P, [2, 1]));
    product(op(P, 1) | notx, op(P, [2, 1]) = op(P, [2, 2]) | subst, op(P, 3..nops(P)))
  else
    if contains(map(subst, op, 1), op(P, 2)) then
      error("Cannot evaluate indefinite product at a particular value of the product variable")
    end_if;
    eval(hold(product)(op(P, 1) | subst, op(P, 2), op(P, 3..nops(P))))
  end_if
end_proc:



// when calling Simplify(product(f, ..)), only recurse on f
product::operandsToSimplify:= [1]:


product::Content :=
proc(Out, data)
local bvar;
begin
  if nops(data) <> 2 then
    return(Out::stdFunc(data));
  end_if;
  if type(op(data, 2)) = "_equal" then
    if type(op(data, [2,2])) = "_range" then
      Out::Capply(Out::Cproduct,
                  Out::Cbvar(Out(op(data, [2,1]))),
                  Out::Clowlimit(Out(op(data, [2,2,1]))),
                  Out::Cuplimit( Out(op(data, [2,2,2]))),
                  Out(op(data,1)));
    else
      bvar := Out::Cbvar(Out(op(data, [2,1])));
      Out::Capply(Out::Cproduct, bvar,
                  Out::Ccondition(
                                  Out::Capply(Out::Cin,
                                              bvar,
                                              Out(op(data, [2,2])))),
                  Out(op(data,1)));
    end_if:
  else
    Out::Capply(Out::Cproduct,
                Out::Cbvar(Out(op(data, 2))),
                Out(op(data,1)));
  end_if;
end_proc:
