//     

/*++
    taylor -- expand expression into taylor series

    taylor(f, x < =a> <, order> <, mode> <,NoWarning>)

    f     - expression to expand
    x     - identifier or indexed identifier
    a     - arithmetical expression
    order - nonnegative integer or
            RelativeOrder = Integer or
            AbsoluteOrder = Integer
    mode  - RelativeError (default) or AbsoluteError
            (only has an effect when the order is
            specified by an integer, not by an equation
            such as RelativeError = integer.

    taylor(f, x=a <, order>) expands f according to the variable 
    x at point p into a taylor series of order o.

    taylor(f, x <, order>) expands f at x=0.

    'order' may be missing, in this case the global variable ORDER
    determines the order of expansion. The default value of ORDER 
    is 6.

    With RelativeOrder = order, the result is
        coeff[m]*(x - a)^m + .. + coeff[m + order]*(x - a)^(m + order - 1) + O((x-a)^(m + order))
    with coeff[m] <> 0 and some (a priori unknown) leading exponent m.
    With AbsoluteOrder = order, the result is
        coeff[m]*(x - a)^m + .. + coeff[order]*(x - a)^(order - 1) + O(x-a)^order)
    with coeff[m] <> 0 and some (a priori unknown) leading exponent m.

    An error is reported, if a Taylor series expansion does not exist.
++*/

taylor:= proc(f, x)
   local ff, x_orig, xx, x0, nowarning, ord, order, i, val,
         relative, n, g, count, furtherargs, mapCoeffs;
begin
   if args(0) <> 0 and f::dom::taylor <> FAIL then
      return( f::dom::taylor(args()) );
   elif args(0) < 2 then
      error("expecting at least two arguments")
   end_if;

   order:= ORDER;        // default
   relative:= TRUE;      // default
   furtherargs:= null(); // default
   nowarning:= FALSE;    // default
   mapCoeffs := id;      // default

   for i from 3 to args(0) do
      if has(args(i), AbsoluteOrder) then
         relative:= FALSE;
         if type(args(i)) = "_equal" then
            order:= op(args(i), 2);
         end_if;
         next;
      end_if;
      if has(args(i), RelativeOrder) then
         relative:= TRUE;
         if type(args(i)) = "_equal" then
            order:= op(args(i), 2);
         end_if;
         next;
      end_if;
      if domtype(args(i)) = DOM_INT then
           order:= args(i);
           next;
      end_if;
      if args(i) = NoWarning then
         nowarning:= TRUE;
         next;
      end_if;
      if type(args(i)) = "_equal" and
        op(args(i), 1) = Mapcoeffs then
        mapCoeffs := op(args(i), 2);
        next;
      end_if;
      if not ( has(args(i), Left) or
               has(args(i), Right) or
               has(args(i), Real)
             ) then
         error("unexpected argument: ".expr2text(args(i)));
      end_if;
      furtherargs:= furtherargs, args(i);
   end_for;

   if domtype(order) <> DOM_INT or order < 0 then
      error("3rd argument: the order of the series computation ".
            "must be a nonnegative integer");
   end_if;

   furtherargs:= subs(furtherargs, 
                      [NoWarning = null(), 
                       RelativeOrder = null(),
                       AbsoluteOrder = null()]);

/* // old code without order control

   s:= series(args(), hold(NoWarning));
   if type(s) = "series" and not nowarning then
      warning("could not compute Taylor series expansion; ".
              "try 'series' with option 'Left', 'Right', or 'Real' ".
              "for a more general expansion");
      return(procname(args()))
   elif testtype(s, Type::Series(hold(Taylor))) then
      return(s)
   else
      error("does not have a Taylor series expansion, try 'series'")
   end_if

*/ // end of of code without order control


   //-------------------------------
   // New code with order control.
   //-------------------------------

   // series does not accept indexed identifiers.
   // Fix this deficiency for taylor:
   if type(x) = "_equal" then
      x_orig:= op(x, 1); // remember the user's original name
      x0:= op(x, 2);
   elif type(x) = DOM_IDENT or
        type(x) = "_index" then
      x_orig:= x;
      x0:= 0;
   else
      error("2nd argument: illegal specification of the expansion variable"):
   end_if;
   xx:= x_orig;
   ff:= f;
   if type(xx) = "_index" then
      xx:= genident("X"):
      ff:= subs(f, x_orig = xx);
   end_if:

   //--------------------------------
   // Do the expansion. The function series has a problem with
   // cancellation. Occasionally, we have to boost the order, 
   // requesting a higher order n instead.
   //--------------------------------
   count:= 0;  // count the number of boosts
   n:= max(1, order); // first try
   val:= 0:   // with this choice, n is the absolute truncation order. If
              // RelativeOrder is specified, val will be modified below.
   g:= 1: // initialize with any dummy
   repeat
     count:= count + 1;
     if traperror((g:= series(ff, xx = x0, n, furtherargs, NoWarning))) > 0 then
        // could be caused by 'order too small'. Try again with higher order
        if traperror((g:= series(ff, xx = x0, n + 1, furtherargs, NoWarning))) > 0 then
           if not nowarning then
              warning("cannot compute a Taylor expansion of ".expr2text(f));
           end_if;
           return(procname(args()));
        end_if;
     end_if;
     if type(g) = "series" then
        if not nowarning then
           warning("cannot compute a Taylor expansion of ".expr2text(f).
                   ". Try 'series' with one of the options 'Left', 'Right', or 'Real' ".
                   "for a more general expansion");
        end_if;
        return(procname(args()));
     end_if;
     if domtype(g) <> Series::Puiseux then
        error(expr2text(f)." does not have a Taylor series expansion, ".
              "try 'series'")
     end_if;
     g := map(g, mapCoeffs);
     ord:= g::dom::order(g);     // the truncation order of the series
     if relative then 
        val:= g::dom::valuation(g); // the lowest exponent of the series
     end_if;
     // Try again with higher order n:
     if val < 0 then
          n:= n - val;
          ord:= order + val - 1; // enforce repetition
     elif ord - val < order then
           n := n + (order - ord + val);
     end_if;
   until ord - val >= order or count > 2 end_repeat;
   //---------------------
   // end of order control
   //---------------------

   if type(g) = "series" and not nowarning then
      warning("could not compute a Taylor series expansion; ".
              "try 'series' with option 'Left', 'Right', or 'Real' ".
              "for a more general expansion");
      return(procname(args()))
   elif not testtype(g, Series::Puiseux) then
        error(expr2text(f)." does not have a Taylor series expansion, ".
              "try 'series'")
   end_if;
   if not testtype(g, Type::Series(hold(Taylor))) then
      // Example:  taylor(cos(sqrt(x)), x = 0) needs condensing:
      g:= g::dom::condense(g);
      ord:= g::dom::order(g);     // the truncation order of the series
      if relative then
         val:= g::dom::valuation(g); // the lowest exponent of the series
      end_if;
   end_if;
   if not testtype(g, Type::Series(hold(Taylor))) then
        error(expr2text(f)." does not have a Taylor series expansion, ".
              "try 'series'")
   end_if;

   if xx <> x_orig then
      // there does not seem a reasonably fast interface function
      // in Series::Puiseux for changing the name of the variable,
      // so we use an explicit 'new'  (dangerous!!!!!)
      g:= new(g::dom, op(subs([extop(g)], xx = x_orig)));
   end_if;

   if ord > order + val then
      return(g::dom::truncate(g, order + val));
   end_if;

   return(g);
end_proc:

taylor:= slot(funcenv(taylor), "type", "taylor"):
