/*++
_print_userinfo -- print information for user

_print_userinfo(procedure_name, <hold(Text), hold(NoNL), > n, expr, ...)

procedure_name - procedure name
Text           - output exprsequ without commas
NoNL           - no line break after output
n              - level of information (positive integer or range)
expr           - expression to print

'_print_userinfo' prints its arguments 'expr...' to the screen if either the
"information level" of the procedure or domain calling 'userinfo' or
the "global information level" is greater or equal to n.

The information level (of a procedure or domain or the global one) is defined 
via 'setuserinfo'. The default level is 0, ie. no information is printed.
++*/

_print_userinfo :=
  proc(procedure_name)
    option hold;
    local myexpr2text, checklevel, printinfo,	// Utilities
          prefix, separator, postfix, linefeed,	// Output style
          n, arguments, domain_name, shift, flag, i;
  begin

    // This is necessary because expr2text quotes strings
    myexpr2text := proc(s) begin
      if domtype(s) = DOM_STRING then s else expr2text(s) end_if
    end_proc;

    // Set the default values for the output style

    prefix    := "Info: ";
    separator := ", ";
    postfix   := "";
    linefeed  := "\n";

    // Look for options in the arguments

    flag := FALSE;
    for shift from 2 to args(0) do
        case context(args(shift))
          of hold(Text) do
              separator := "";
              break;		// proceed to next argument
          of hold(NoNL) do
              linefeed  := "";
              prefix    := "";
              separator := "";
              break;		// proceed to next argument
          otherwise
              flag := TRUE;
        end_case;
        if flag then
          break;			// exit the loop
        end_if;
    end_for;

    // Return right now if there are no arguments to be printed
    if args(0) <= shift then
        return(null());
    end_if;

    // Store the rest of the arguments
    n  := context(args(shift));
    arguments := args(i) $ i = shift+1..args(0);

    //////////////////////////////////////////////////////////////////////////
    // Utility which returns TRUE if a user information level has been
    // set for proc_name and the level n is in range.

    checklevel :=
      proc(proc_name, n, library = FAIL) : DOM_BOOL
        local infolevel;
      begin
        
        if not contains(_userinfo_level(), proc_name) then
          return(UNKNOWN)
        else
          infolevel := _userinfo_level()[proc_name]
        end_if;

        // off (value 0)
        if infolevel = 0 then
          return(FALSE)
        end_if;

        if type(infolevel) <> DOM_INT then
            if op(infolevel, 2) = hold(Quiet) then
                prefix:= "";
                postfix:= "";
            else // Assume without checking that the option is Name
                postfix:= " " . myexpr2text([procedure_name]);
            end_if;
            infolevel:= op(infolevel, 1)
        end_if;

        case type(n)
          of DOM_INT do
              return(bool(infolevel >= n));
          of "_range" do
              return(bool(infolevel >= op(n, 1) and infolevel <= op(n, 2)));
          otherwise
              return(FALSE);
        end_case;
    end_proc;

    /////////////////////////////////////////////////////////////////////////
    // Utility to actually print the expressions passed as argument.
    // The style of the output is conditioned by variables prefix,
    // separator, postfix, and linefeed.

    printinfo:=
      proc()
        local i;
      begin
        fprint(NoNL, 0, 
              _concat(prefix, myexpr2text(args(1)),
                      (separator, myexpr2text(args(i))) $ i = 2..args(0),
                      postfix, linefeed));
      end_proc;

    // Check the information level of the calling procedure
    case checklevel(procedure_name, n)
      of TRUE do
        printinfo(context(arguments))
      of FALSE do
        return(null())
    end_case;

    // Check the information level of the calling domain
    if type(procedure_name) = "slot" then
        // Look for the name of the domain
        domain_name := procedure_name;
        repeat
            domain_name := op(domain_name, 1);
        
            case checklevel(domain_name, n)
              of TRUE do
                printinfo(context(arguments))
              of FALSE do
                return(null())
            end_case;
        until type(domain_name) <> "slot" end_repeat;
    end_if;

    // Check the global information level
    if checklevel(hold(Any), n) = TRUE then
        printinfo(context(arguments));
        return(null());
    end_if;

    null();
  end_proc:

/*
Some tests I used to compare with the standard version

>> setuserinfo(Any, 0):

>> bla:=proc() begin userinfo(4, "blonc"); end_proc: bla():

>> proc() begin userinfo(4, "blonc"); end_proc():         

>> setuserinfo(Any, 5):

>> bla:=proc() begin userinfo(4, "blonc"); end_proc: bla():
Info: blonc

>> proc() begin userinfo(4, "blonc"); end_proc():         
Info: blonc

>> setuserinfo(Any, 5, Name):

>> bla:=proc() begin userinfo(0, "blonc"); end_proc: bla():
Info: blonc [bla]

>> proc() begin userinfo(0, "blonc"); end_proc():
Info: blonc [NIL]

>> setuserinfo(bla,10):                                   

>> bla:=proc() begin userinfo(4, "blonc"); end_proc: bla()
Info: blonc

>> setuserinfo(Any,0):
>> setuserinfo(bla,10,Name):

>> bla:=proc() begin userinfo(4, "blonc"); end_proc: bla()
Info: blonc [bla]

*/

// end of file 
