/*- $Id

stringlib::contains -- tests if a substring exists

Calling sequence:

stringlib::contains(s, t)
stringlib::contains(s, t, Index)
stringlib::contains(s, t, IndexList)

Parameter:

s         - non-empty string
t         - non-empty string
Index     - an optional name
IndexList - an optional name

Summary:

stringlib::contains(s, t) returns TRUE, if the string s contains the
string t. Otherwise FALSE is returned.

stringlib::contains(s, t, Index) returns an integer i.
//a list [n, m, i]. n is the length of s and m is the length of t.
//The length of a string is the number of characters.
If i = FAIL, then the string s doesnt 
contain the string t. Otherwise i is an integer and 
s[i..i+m-1] = substring(s, i, m) = t holds (m = length(t)). 

stringlib::contains(s, t, IndexList) returns a list l
//[n, m, l].
l is a list. If l is the empty list, i.e. l = [], then the string s doesnt 
contain the string t. Otherwise s[l[i]..l[i]+m-1] = t for all
i = 1..nops(l) holds (m = length(t)).

Examples:

>> stringlib::contains("abcdabe", "ab");

           TRUE
		   
>> stringlib::contains("abcdabe", "ab", Index);

           1 // 1.4.2 yields [7, 2, 0]
		   
>> stringlib::contains("abcdabe", "ab", IndexList);

           [1, 5] // 1.4.2 yields [7, 2, [0, 4]]
		   
>> stringlib::contains("abcdabe", "abd");

           FALSE
		   
>> stringlib::contains("abcdabe", "abd", Index);

           FALSE // 1.4.2 yields [7, 3, FAIL]
		   
>> stringlib::contains("abcdabe", "abd", IndexList);

           [] // 1.4.2 yields [7, 3, []]
		   
See also:

contains, substring, strmatch

-*/

stringlib::contains:=
  proc(s, t, opt = FAIL)
    local k, m, n;
  begin
    if testargs() then
      case args(0)
        of 3 do
          if opt <> hold(Index) and opt <> hold(IndexList) then
            error("Illegal 3rd argument")
          end_if;
        of 2 do
          if not (testtype(s, DOM_STRING) /* and s <> "" */ and
                  testtype(t, DOM_STRING) and t <> "") then
            error("Non-empty string expected")
          end_if;
          break;
        otherwise
          error("Wrong number of arguments")
      end_case
    end_if;
    
    // ensure t contains no meta-characters
    t := stringlib::maskMeta(t);
    
    m := length(t); n := length(s);
    k := stringlib::FirstIndex;
    case opt
      of Index do
        return(op(strmatch(s, t, hold(Index)), 1))
      of IndexList do
	return(DOM_SET::sort(map(strmatch(s, t, Index, All),
				 op, 1)));
      otherwise
        return(strmatch(s, t))
    end_case
  end_proc:

