// Dirk Mueller, olorien, 28/10/96, v1.0 

//-------------------------------------
transform::invlaplace:=
proc(_f, var1, var2)
  name transform::invlaplace ;
  local rationalCase, _f2, dummy, constants, back, C, mycollect;
  save _X, _Y ;
begin
   if args(0) = 0 then error("no arguments given") end_if;

   // allow overloading 
   if _f::dom::invlaplace<>FAIL then
     back := _f::dom::invlaplace(args());
     if back <> FAIL then
       return(back)
     else
       return(hold(transform::invlaplace)(args()))
     end_if
   end_if;

   // do the overloading for the funcenv transform::laplace because it
   // is no longer a domain
   if type(_f) = "transform::laplace" then
     back:= transform::laplace::invlaplace(args());
     if back <> FAIL then
       return(back)
     else
       return(hold(transform::invlaplace)(args()))
     end_if
   end_if:
  
  if testargs() then
    if args(0)<>3 then error("wrong number of arguments") end_if;
 
    if type(_f) = "_equal" then
        if not testtype(_f,Type::Arithmetical=Type::Arithmetical) then
            error("invalid 1st argument")
        end_if
    elif not testtype(_f,Type::Arithmetical) then
        error("1st argument must be of 'Type::Arithmetical'")
    end_if;

    if domtype(var1)<>DOM_IDENT then
      error("second argument must be of type DOM_IDENT");
    end_if;
  end_if;

  //---------------------------
  // partial fraction expansion
  //---------------------------
  rationalCase:= FALSE;
  if traperror(poly(1, [var1])) = 0
     // otherwise partfrac produces: error("illegal indeterminate")
     and
     testtype(_f, Type::RatExpr(var1) )
     // otherwise partfrac produces: error("not a rational function")
     then
      rationalCase:= TRUE;
      // Walter: 5.4.08 (reaction to changes in partfrac):
      // partfrac((x^2 + a*x + b)/(x^2 + c*x + d), x) produces
      // 1 + (b - d + a*x - c*x)/(x^2 + c*x + d). The matcher
      // has patterns for (A*x + B)/(x^2 + C*x + D), but does 
      // not match a*x-c*x to A*x. We need to collect terms!
      mycollect:= proc(a)
                  begin
                    if not has(a, var1) then
                       return(a)
                    end_if:
                    if type(a) = "_power" and 
                       not has(op(a, 2), var1) then
                       return(mycollect(op(a, 1))^op(a, 2));
                    elif type(a) = "_mult" then
                       // do not loose factorizations, so map
                       // into the factors instead of collecting
                       // the terms of the polynomial
                       return(map(a, mycollect));
                    elif type(a) = "_plus" then
                       return(eval(collect(a, [var1])));
                    end_if:
                    return(a);
                  end_proc:
      _f:= partfrac(_f, var1, Mapcoeffs = mycollect);
  end_if:

  //-----------------------------------
  // linearity
  //-----------------------------------
  if (not rationalCase) then
     // Expand f without using expand (stuff like
     // exp, sin, cos, fact etc should not be expanded).
     _f:= transform::expand(_f, var1):
  end_if;
  if type(_f) = "_plus" then
     return(map(_f, transform::invlaplace, var1, var2));
  end_if;
  if type(_f) = "_mult" then
     // extract constants
     [_f2, constants, dummy]:= split(_f, has, var1);
     if _f2 <> _f and not iszero(constants -1) then
        return(constants*transform::invlaplace(_f2, var1, var2));
     end_if;
  end_if;

  //---------------------------
  // Prepare for the lookup (need the
  // correct pattern variable)
  //---------------------------
  _f:= combine(combine(_f), exp):
  C:=genident() ;
  delete _X;
  if var1 <>_X then _f:=subs(_f,[_X=C,var1=_X]) end_if;
  userinfo(2,"trying the lookup method for invlaplace");
  back:= transform::invlaplace::lookup(_f, _X);
  if back<>FAIL then
    back:=subs(back, [_X=var1,_Y=var2,C=_X,
     hold(transform::frozenInvlaplace) = hold(transform::invlaplace),
     hold(transform::frozenLaplace) = hold(transform::laplace)
    ]) ;
    return(back);
  else
    _f:=subs(_f, [_X=var1,C=_X]) ;
  end_if;

  //-------------------------------------
  // proceed with traditional code
  //-------------------------------------
  
  back:=transform::invlaplace::compute(_f, var1, var2);

  if back <> FAIL then
     return(back);
  end_if;

  //-------------------------------------
  // invlaplace of rational expressions
  //-------------------------------------
  if rationalCase then
     back:= transform::invlaplace::rational(_f, var1, var2):
     if back<>FAIL then
       return(back);
     end_if;
  end_if;

/*
  if back=FAIL and testtype(_f,Type::RatExpr(var1)) then
      back:=transform::invlaplace::compute(partfrac(_f, var1), var1, var2);
  end_if;
*/

  if back=FAIL then
    _f2:= expand(_f, op(select(indets(_f, RatExpr),
                               a -> contains({"sin", "cos", "sinh", "cosh"},
                                             type(a))
                               )));
    if (not (_f=_f2)) then
      back:=transform::invlaplace::compute(_f2, var1, var2);
    end_if;
  end_if;

  if back=FAIL then
    return(hold(transform::invlaplace)(_f, var1, var2)) 
  end_if;

  return(back)

end_proc:

transform::invlaplace:= funcenv(transform::invlaplace):
transform::invlaplace::type:= "transform::invlaplace":
transform::invlaplace::interface := {hold(addpattern)}:

transform::invlaplace::has:=proc() local T, v;
begin
  T:=args(1): v:=args(2):
  bool( has(extop(T,1), v) or
        has(extop(T,2), v) or
        has(extop(T,3), v) or
        v=hold(transform::invlaplace)
       )
end_proc:

transform::invlaplace::numer:= id:
transform::invlaplace::denom:= 1:
transform::invlaplace::normal:= id:

transform::invlaplace::laplace:=
proc(_f: "transform::invlaplace", var1: DOM_IDENT, var2)
  local t0;
begin
  if extop(_f, 3) = var1 then
    return(subs(extop(_f,1), extop(_f,2)=var2, EvalChanges))
  end_if;
  // The following implements
  //   invlaplace(laplace(f(x),x,t+t0),t, x) = exp(-t0*x)*f(x)
  // for t0 >= 0.
  t0 := extop(_f, 3) - var1;
  // If t0 >= 0, heaviside(t + t0) = 1 for all t >= 0,
  // hence laplace(h(t+t0)*g(t), t, s) = laplace(g(t), t, s)
  if not has(t0, var1) and is(t0 >= 0) = TRUE then
    return(exp(var2*t0)*subs(extop(_f,1), extop(_f,2)=var2, EvalChanges));
  end_if;
  return(hold(transform::laplace)(_f, var1, var2));
end_proc:


//this part computes the laplacian transform
transform::invlaplace::compute:=proc(_g, var1, var2) 
local _f, back, typ, den, dofloat,
      C, j, result, tmp,
      exps, others, dummy, processed, a, b, aa, bb, term, pow, i, r;
begin
  _f:=_g;
  if not has(_f, var1) then
    return(_f*dirac(var2));
  elif _f=var1 then
    return(dirac(var2, 1));
  end_if;

  typ:=type(_f);

  // ----------------------
  // Stephan Jan. 3. 2002
  // IL(s^k, s, x) = dirac(x, k)
  if typ = "_power" and op(_f,1) = var1 then
      pow := op(_f,2);
      if testtype(pow, DOM_INT) and pow > 1 then;
          return(dirac(var2, pow));
      end_if;
  end_if;

  if typ="_equal" then
    return(_equal(transform::invlaplace(op(_f,1), var1, var2),
                  transform::invlaplace(op(_f,2), var1, var2)))
  end_if;

  // -------------------------------------------------------------
  // Walter, July 7. 2005
  // If there is a float in the denominator of an expression,
  // do a numerical factorization of the denominator:
  // -------------------------------------------------------------
  den:= denom(_g);
  dofloat:= id;
  if testtype(den, Type::PolyExpr(var1)) and
     numeric::indets(den) = {var1} and
     degree(den, [var1]) > 1 then
        // check for floats in the expression:
        if hastype(expr(den), DOM_FLOAT, {DOM_COMPLEX}) then
           // we have checked that den is a univariate poly,
           // so it is save to call numeric::factor
           _g:= numer(_g)/numeric::factor(den);
           _g:= numeric::rationalize(_g):
           _f:= _g;
           dofloat:= () -> float(args());
        end_if;
  end_if;

  // -------------------------------------------------------------
  // the following implements
  //     invlaplace( exp(-a*s)*g(s), s, t) = 
  //         heaviside(t - a)* invlaplace(g(s), s, t - a)
  // for a >= 0
  // -------------------------------------------------------------
  if has(_g, exp) and
     (typ = "_mult" or typ = "_power" or typ = "exp")
  then
     if typ = "exp" then
        // _g = exp(a*s + b) * 1
        exps := [_g];
        others := 1;
     end_if;
     if typ = "_power" then
        if type(op(_g, 1)) = "exp" 
        then // _g = exp(a*s + b)^n * 1
             exps := [_g];
             others := 1;
        else exps:= [1];
             others:= _g;
        end_if;
     end_if;
     if typ = "_mult" then
        [exps, others, dummy] := split([op(_g)], 
           proc(x)
           begin
             if type(x) = "exp" or
               (type(x) = "_power" and type(op(x, 1)) = "exp") then
                return(TRUE):
             end_if;
             FALSE;
           end_proc);
        if dummy <> [] then
           // this should not happen
           error("failed to split off exponential factors in ".
                 expr2text(_g))
        end_if;
        others:= _mult(op(others));
     end_if;

     // all factors exp(a1*s + b1)^n1*exp(a2*s + b2)^n1* ...
     // were collected in 'exps', all other factors are
     // collected in 'others' = _g / exp(a1*s + b1)^n1 / exp(a2*s + b2)^n2 / ...
     aa:= 0;  // aa = n1*a1 + n2*a2 + ...
     bb:= 0;  // bb = n1*b1 + n2*b2 + ...
     for i from 1 to nops(exps) do
         processed:= FALSE;
         term:= op(exps, i):
         if not has(term, exp) then next end_if;
         pow := 1:
         // case: term = exp(a*s + b)^n
         if type(term) = "_power" then
            pow:= op(term, 2);
            term:= op(term, 1): // = a*s + b
         end_if;
         if type(term) = "exp" then
              a := diff(pow*op(term, 1), var1);
              if not has(a, var1) then
                 b:= expand(pow*op(term, 1) - a*var1);
                 if not has(b, var1) 
                 then // op(term, 1) = a*var1 + b
                      aa := aa + a;
                      bb := bb + b;
                      processed:= TRUE;
                 end_if;
               end_if;
         end_if;
         if not processed then
            // term is not of the form a*s + b
            // -> collect in others
            others:= others*term^pow;
         end_if;
     end_for;
     if others <> _g then
        r := exp(bb)*transform::invlaplace(others, var1, var2);
        if has(r, FAIL) then return(FAIL); end_if;
  
        if is(aa <= 0) = TRUE then
          //special case:  return dirac(x + a) rather than dirac(x + a)*heaviside(x + a)
          if hastype(r, "dirac") then
             return(dofloat(subs(r, var2 = var2 + aa,EvalChanges)));
          end_if;
  
          return(dofloat(heaviside(var2 + aa) * subs(r, var2 = var2 + aa, EvalChanges)));
        end_if;
     end_if;
  end_if;
  // nothing was changed, if no factors of the form exp(a*x + b)^n
  // were found. Proceed as usual.
  // -------------------------------------------------------------

  if typ="_mult" or typ="_plus" then
    back:=transform::IntTrans::linear(hold(transform::invlaplace)(_f, var1, var2));
    if not (back=FAIL) then
      return(dofloat(back));
    end_if;
  end_if;

  if typ="function" and type(op(_f,0))="D" then
    _f:=transform::IntTrans::D2diff(_f)
  end_if;

  back:=transform::IntTrans::builtin(hold(transform::invlaplace)(_f, var1, var2));
  if not (back=FAIL) then
    return(dofloat(back));
  end_if;

  // =====================================================================
  // Finally, try 
  //    invlaplace(s*f(s),s,x)  = 
  //         diff(invlaplace(f(s), s, x), x) + invlaplace(f(s),s,0)
  // and, consequently, 
  //    invlaplace(s^n*f(s),s,x)  = 
  //                diff(invlaplace(f(s),s,x), x $ n) 
  //      +  evalAt(diff(invlaplace(f(s),s,x), x $ n-1), 0)
  //      +  evalAt(diff(invlaplace(f(s),s,x), x $ n-2), 0)
  // + .. +  evalAt(diff(invlaplace(f(s),s,x), x $  0 ), 0)
  // Use this rule, if invlaplace(f(s), s, x) produces an explicit 
  // expression and invlaplace(f(s), s, 0) produces no error:
  // =====================================================================
  if type(_f) = "_mult" then 
    // search for a polynomial factor g(s) such that f(s) = g(s)*tmp(s):
    for i from 1 to nops(_f) do
      _g:= op(_f, i);
      if testtype(_g, Type::PolyExpr(var1)) then
         if domtype(var2) <> DOM_IDENT then
            // We need symbolic differentiation below.
            return(FAIL);
         end_if;
         tmp:= eval(subsop(_f, i = 1)):
         // Now, f(s) = g(s)*tmp(s)
         back:= normal(transform::invlaplace(tmp, var1, var2));
         if not hastype(back, "transform::invlaplace") then
            assert(domtype(var2) = DOM_IDENT);
            _g:= poly(_g, [var1]):
            assert(has(_g, FAIL) = FALSE);
            _g:= poly2list(_g):
            result:= 0:
            for C in _g do
              result:= result + C[1]*diff(back, var2 $ C[2]);
              for j from 0 to C[2] -1 do
                if traperror((tmp:= evalAt(diff(back, var2 $ j), var2 = 0))) = 0 and
                   not hastype(tmp, "evalAt") then
                  result:= result + C[1]*tmp:
                else
                  return(FAIL);
                end_if;
              end_for;
            end_for:
            return(normal(result, Expand = FALSE)):
         end_if;
      end_if;
    end_for:
  end_if;

  // Beware! No additional code down here, 
  // because the previous code used 'return'!
  return(FAIL);
end_proc:

// Necessary for the subs command after the lookup mechnism, because
// otherwise no further evaluation is done for substituted elements in a
// transform::invlaplace element
// invlaplace::subs:= e -> new(dom, subs(extop(e), args(2..args(0)))):

//----------------------------------------------------
// Walter: 5.3.06: new code for transforming
// generic rational expressions:
// invlaplace::rational(f(s), s, t)  computes invlaplace 
// for rational expressions f(s) = numer(s)/denom(s).
// Note that f(s) is not a sum, because this procedure
// is mapped to the terms of a partfrac expansion.
//----------------------------------------------------
transform::invlaplace::rational:= proc(f, s, t)
local p, q, dq, coeffs, dummy, c,
      r, roots, _s, q_s, qq, res, trans;
begin
   // The rational function is a term of a partial
   // fraction expansion with constant coefficients
   // removed:

   if not has(f, s) then
      return(f*dirac(t));
   end_if;

   // f:= p/q with polynomials p = p(s), q = q(s)
   [p, q]:= [numer(f), denom(f)];

   if degree(p, [s]) >= degree(q, [s]) then
      return(FAIL);
   end_if;

   // -----------------------------------------
   // move constant factors from denominator q
   // to the numerator p:
   // -----------------------------------------
   coeffs:= 1:
   if type(q) = "_mult" then
      [q, coeffs, dummy]:= split(q, has, s);
      p:= p/coeffs;
   end_if;

   // -----------------------------------------
   // turn denominator into the form (irreducible)^dq
   // -----------------------------------------
   if type(q) =  "_power" then
      [q, dq]:= [op(q)];
      // original_q = q^dq with some
      // **irreducible** polynomial q.
   else
      dq:= 1:
   end_if;

   assert(type(q) <> "_mult"):
   assert(type(q) <> "_power"):
   assert(irreducible(q));

   //-------------------------------------------------------
   // Result so far: f = p(s)/q(s)^dq  with irreducible q(s)
   //-------------------------------------------------------

   //--------------------------------------
   // We have to compute the residues
   // diff( exp(s*t) * p(s)/ (q(s)/(s -r))^dq , s $ dq - 1)
   // at the point s = r, where r are the roots of q(s).
   //--------------------------------------
   // Represent the roots: either explicit
   // (for degree(q) = 1, 2) or as RootOf:
   //--------------------------------------
   case degree(q, [s])
   of 0 do
       return(FAIL); // error("should not arrive here");
       break;
   of 1 do
       c[0]:= coeff(q, [s], 0);
       c[1]:= coeff(q, [s], 1);
       roots:= {-c[0]/c[1]};
       break;
   of 2 do
       c[0]:= coeff(q, [s], 0);
       c[1]:= coeff(q, [s], 1);
       c[2]:= coeff(q, [s], 2);
       roots:= {
                 (-c[1] + specfunc::Wurzelbehandlung(c[1]^2 - 4*c[0]*c[2]))/(2*c[2]),
                 (-c[1] - specfunc::Wurzelbehandlung(c[1]^2 - 4*c[0]*c[2]))/(2*c[2])
               };
       break;
   otherwise
       _s:= genident("s"):
       q_s:= subs(q, s = _s):
       roots:= RootOf(subs(q, s = _s), _s)
   end_case;

   //--------------------------------------------
   // compute qq = q(s)/(s - r) assuming q(r) = 0
   //--------------------------------------------
   qq:= (q, s, r) -> divide(q, s - r, [s])[1]:

   //-------------------------------------------
   // compute the residue of exp(s*t)*p(s)/qq(s) =
   //  1/(dq-1)!*diff(exp(s*t)*p(s)/qq(s)^d, s$dq -1)
   // at the point s = r by direct differentiation
   //-------------------------------------------

   if domtype(roots) = DOM_SET then
     trans:= 0:
     for r in roots do
       res:= exp(s*t)*p/qq(q, s, r)^dq;
       res:= diff(res, s $ dq - 1):
       res:= subs(res, s = r, EvalChanges):
       trans:= trans +  res/(dq - 1)!;
     end_for:
   else
      assert(type(roots) = RootOf);
      r:= genident("r"):
      res:= exp(s*t)*p/qq(q, s, r)^dq;
      res:= subs(diff(res, s $ dq - 1), r = s, EvalChanges):
      res:= subs(res, s = r, EvalChanges):

      /*
      // convert 1/denom(t) mod q(r) to a polynomial?
      den:= [gcdex(denom(res), subs(q, s = r), r)];
      assert(den[1] = 1): // q is irreducible!
      res:= divide(numer(res)*den[2], subs(q, s = r), [r])[2];
      */

      //-------------------------------------------
      // res is a rational expression in r. Reduce
      // numerator and denominator modulo q(r) = 0:
      //-------------------------------------------
      res:= divide(numer(res), subs(q, s = r), [r])[2] /
            divide(denom(res), subs(q, s = r), [r])[2];

      trans:= hold(sum)(res/(dq -1)! , r in roots):
   end_if;

   //beautify by rewriting exp(I*x) = cos(x) + I*sin(x) etc
   if has(trans, exp) then
      trans:= subs(trans, hold(exp) = proc(x) local i, y, re, im; begin
                            if type(x) = "_mult" and 
                               has(x, I) then
                                 // exp(x) = exp(y*(u + I*v)) = exp(y*u)*(cos(y*v) + I*sin(y*v))
                                 for i from 1 to nops(x) do
                                   if testtype(op(x, i), Type::Numeric) and 
                                      has(op(x, i), I) then
                                      y:= x/op(x, i);
                                      re:= exp(y*Re(op(x, i)));
                                      im:= y*Im(op(x, i));
                                      return(  re*cos(im) 
                                            +I*re*sin(im));
                                   end_if;
                                 end_for;
                            end_if;
                            return(hold(exp)(x));
                          end_proc):
      trans:= transform::expand(eval(trans), s);
   end_if;
   
   return(trans):
end_proc:

//------------------------------------------------
// overloading of evalAt is needed for testeq etc.
// Ignore substitutions of the second operand!
//------------------------------------------------
transform::invlaplace::evalAt:= proc(f, subst)
local g, s, ss, t, tt;
begin
   assert(type(f) = "transform::invlaplace");
   [g, s, t]:= [extop(f)];
   // ignore substitutions of the bound variable s:
   subst:= select(subst, y -> (op(y, 1) <> s) );
   // when the bound variable s is in the rhs's of the
   // substitutions, change the name of the variable:
   if contains(freeIndets(map(subst, op, 2)), s) then
      ss:= genident(expr2text(s));
      f:= subs(f, s = ss);
      g:= subs(g, s = ss);
      s:= ss;
   end_if;
   g:= g | subst;
   tt:= t | subst;
   if extop(f, 1) <> g or t <> tt then
      return(transform::invlaplace(g, s, tt))
   else
      return(f)
   end_if;
end_proc:
//------------------------------------------------

transform::invlaplace::freeIndets:= proc(f)
begin
  assert(type(f) = "transform::invlaplace");
  freeIndets([extop(f, 1), extop(f, 3)], args(2..args(0))) minus {extop(f, 2)};
end_proc:

//------------------------------------------------

transform::invlaplace::lookup:=
        loadproc(transform::invlaplace::lookup,
                 pathname("TRANS","LAPLACE"), "invlookup" ):
transform::invlaplace::patternFSA:=
        loadproc(transform::invlaplace::patternFSA,
                 pathname("TRANS","LAPLACE"), "load_patterns"):
transform::invlaplace::addpattern:=
        loadproc(transform::invlaplace::addpattern,
                 pathname("TRANS","LAPLACE"), "invaddpattern"):

transform::invlaplace::userpatterns := []:
transform::invlaplace::userpatternsFSA := FAIL:
transform::frozenInvlaplace:= () -> procname(args()):

// end of file 
