/*++
        fourier(f(x),x,y) returns the Exponential Fourier transform of f(x) 
        with respect to y,

        g_e(y) = int(f(x)*exp(I*y*x),x=-infinity..infinity).

        There are also the Fourier Cosine Transform

        g_c(y) = int(f(x)*cos(y*x),x=0..infinity)

        and the Fourier Sine Transform

        g_s(y) = int(f(x)*sin(y*x),x=0..infinity)

        If f is even, then g_e(y)=2*g_c(y); if f is odd, then g_e(y)=2*I*g_s(y),
        thus g_e(f(x)) = g_c(f(x)+f(-x))/2+I*g_s(f(x)-f(-x))/2

        f is any expression, x must be an identifier, y is any expression

        Reference: F. Oberhettinger, Tables of Fourier Transforms and Fourier 
                   Transforms of Distributions, Springer-Verlag, 1990
++*/

alias(Match=intlib::match1):
alias(Match2=intlib::matchassign):
alias(indep(a,x)=_pattern(a,Type::IndepOf(x))):
alias(_p=_pattern):

transform::fourier:= proc(f,x,y) 
   local v, yy, i, ii, tmp, _b, _c, Imba /*, mycollect */, 
        a, b, A, B, C, F, fourierExp, e, dummy;
   save _a,_b_,_aa, _bb, _c_, n,_k_,g,z,h,m, back;
   save _X, _Y ;
begin
   delete _c_,m,z,n;

   if args(0) <> 3 then error("expecting three arguments") end_if;

   if f::dom::fourier<>FAIL then 
       // allow overloading
       return(f::dom::fourier(args())) 
    end_if;

   if testargs() then
      if type(f) = "_equal" then
          if not testtype(f,Type::Arithmetical=Type::Arithmetical) then
              error("invalid 1st argument")
          end_if
      elif not testtype(f,Type::Arithmetical) then
          error("1st argument must be of 'Type::Arithmetical'")
      end_if
   end_if;

   if iszero(f) then
      return(0);
   end_if:
 
   if domtype(x)=DOM_IDENT then
     if not has(f,x) then 
         return(f*2*PI*dirac(y)) 
     end_if
   elif domtype(x/2/PI)=DOM_IDENT then 
     // frequency form:
     return(2*PI*transform::fourier(f,x/2/PI,2*PI*y))
   else
     error("invalid arguments")
   end_if;

/* Shall we preprocess the input via partfrac?
   mycollect:= proc(a)
               begin
                 if not has(a, x) then
                    return(a)
                 end_if:
                 if type(a) = "_power" and
                    not has(op(a, 2), x) then
                    return(mycollect(op(a, 1))^op(a, 2));
                 elif type(a) = "_mult" then
                    // do not loose factorizations, so map
                    // into the factors instead of collecting
                    // the terms of the polynomial
                    return(map(a, mycollect));
                 elif type(a) = "_plus" then
                    return(eval(collect(a, [x])));
                 end_if:
                 return(a);
               end_proc:
   f:= partfrac(f, x, Mapcoeffs = mycollect);
*/

   // We first use the lookup mechanism
   C:=genident() ;
   delete _X;
   if x <>_X then f:=subs(f,[_X=C,x=_X]) end_if;
   userinfo(2,"trying the lookup method for fourier");
   if has(f, sign) then
       // write x*sign(x) as abs(x)
       f:= subs(f, sign(x) = abs(x)/x,EvalChanges);
   end_if:
   back:= transform::fourier::lookup(f, _X);
   if back<>FAIL then
     back:=subs(back, [_X=x,_Y=y,C=_X],EvalChanges) ;
     return(back);
   else
     f:=subs(f, [_X=x,C=_X], EvalChanges) ;
   end_if;

   case type(f)
   of "_equal" do
   of "_plus" do return(map(f,transform::fourier,x,y))
   of DOM_IDENT do return(-2*I*PI*dirac(y,1)) // can be only x 
   //=================================================================
   of "heaviside" do // formula 3.1 p. 199 of Oberhettinger90 
      if (_k_:= transform::TypeLinear(op(f),[x]))<>FALSE then
         return( 
           (PI*dirac(y)+I*sign(_k_[1])/y ) *exp(-I*_k_[2]/_k_[1]*y)
                );
      end_if; 
      break
   //=================================================================
   of "dirac" do
      if nops([op(f)]) = 1 
        then n:= 0; // f = dirac(..)
        else n:= op(f, 2); // f = dirac(.., n)
      end_if:
      if op(f,1)=x then return( (-I*y)^n) end_if;
      if (_k_:= transform::TypeLinear(op(f,1),[x]))<>FALSE then
         return( 
           (-I*y/_k_[1])^n /abs(_k_[1]) * exp(-I*_k_[2]/_k_[1]*y)
               );
      end_if:
      break;
   //=================================================================
   of "exp" do
      if (_k_:= transform::TypeLinear(op(f),[x]))<>FALSE then
         // f = exp(a*x + b)
         _a := _k_[1]:
         _b_:= _k_[2]:
         if is(_a*I in R_) = TRUE then
           return(2*PI*exp(_b_)*dirac(y - _a*I));
         end_if:
         if not iszero(_b_) then
           return(exp(_b_)*transform::fourier(exp(_a*x), x, y));
         end_if;
      end_if; 
      if not has((_a:=op(f)/x^2),x)
        then // formula 3.17 p. 11 (even function)
        // Walter 3.5.06: added the condition Re(_a) <= 0):
        if is(Re(_a) <= 0) = TRUE then
           return(sqrt(PI)/sqrt(-_a)*exp(y^2/4/_a))
        end_if;
      end_if;
      // Walter 3.5.06: added the following pattern
      // (generalizes the pattern above)
      if testtype(op(f), Type::PolyExpr(x)) and
         degree(op(f), [x]) = 2 then
           //   exp(a*x^2 + b*x + c)
           // = exp(a*(x + b/2/a)^2 + c - b^2/4/a)
           // = exp(a*(x + b/2/a)^2) * exp(c - b^2/4/a)
           // Use fourier(f(a*(x + d), x, y)
           //   = exp(-I*d*y)*fourier(f(a*x), x, y)
           //   = exp(-I*d*y)/abs(a)*fourier(f(x), x, y/a) (a reell):
           [_c, _b, _a]:= [coeff(op(f), [x], All)];
           if is(Re(_a) <=0) = TRUE then
             return(exp(_c + (y - I*_b)^2/4/_a)* sqrt(PI)/sqrt(-_a) );
           end_if;
      end_if;
      // Walter 12.6.07: added the following pattern:
      // fourier( exp(-a*abs(x)), x, y) = 2*a/(a^2 + y^2)
      g:= op(f, 1):
      if has(g, sign) then
         // write x*sign(x) as abs(x)
         g:= subs(g, sign(x) = abs(x)/x);
      end_if:
      if has(g, abs) then
        g:= collect(g, [abs(x)]);
        delete _a, _b_;
        if Match2(g, indep(_a,x)*abs(x)+indep(_b_,x)) then
          if is(_a < 0) = TRUE then
            return(exp(_b_) *2*(-_a)/(_a^2 + y^2));
          else
            return(procname(args()));
          end_if:
        end_if:
      end_if;
      break
   //=================================================================
   of "sin" do 
      if not has((_a:=op(f)/x),x) then 
         return(I*PI*(dirac(y-_a)-dirac(y+_a))) 
      end_if; 
      // Walter, 20.1.09: added the following pattern
      // generalizing the pattern above (for which _b_ = 0)
      if (_k_:= transform::TypeLinear(op(f),[x]))<>FALSE then
         // f = sin(a*x + b)
         _a := _k_[1]:
         _b_:= _k_[2]:
        return(PI*I*(exp(-_b_*I)*dirac(y - _a)
                    -exp( _b_*I)*dirac(y + _a)));
      end_if;
      // Walter 3.5.06: added the following pattern
      if not has((_a:= op(f)/x^2),x) then
        if is(_a in R_) = TRUE then
           return(sqrt(PI)/sqrt(abs(_a))/sqrt(2)*( sign(_a)*cos(y^2/4/_a) - sin(y^2/4/_a) ));
        end_if;
      end_if;
      // Walter 3.5.06: added the following pattern
      // (generalizes the pattern above)
      if testtype(op(f), Type::PolyExpr(x)) and
         degree(op(f), [x]) = 2 then
           //   sin(a*x^2 + b*x + c)
           // = sin(a*(x + b/2/a)^2 + c - b^2/4/a)
           // = sin(a*(x + b/2/a)^2) * cos(c - b^2/4/a)
           //  +cos(a*(x + b/2/a)^2) * sin(c - b^2/4/a)
           // Use fourier(f(a*(x + d), x, y)
           //   = exp(-I*d*y)*fourier(f(a*x), x, y)
           //   = exp(-I*d*y)/abs(a)*fourier(f(x), x, y/a) (a reell):
           [_c, _b, _a]:= [coeff(op(f), [x], All)];
           if is(_a in R_) = TRUE then
             return(exp(-I*_b/2/_a*y)*cos(_c-_b^2/4/_a)*
                        sqrt(PI)/sqrt(abs(_a))/sqrt(2)*(sign(_a)*cos(y^2/4/_a)-sin(y^2/4/_a) )
                   +exp(-I*_b/2/_a*y)*sin(_c-_b^2/4/_a)*
                        sqrt(PI)/sqrt(abs(_a))/sqrt(2)*(cos(y^2/4/_a)+sin(y^2/4/abs(_a)) )
                    );
           end_if;
      end_if;
      break
   //=================================================================
   of "cos" do
      if not has((_a:=op(f)/x),x)
        then return(PI*(dirac(y+_a)+dirac(y-_a))) 
      end_if; 
      // Walter, 20.1.09: added the following pattern
      // generalizing the pattern above (for which _b_ = 0)
      if (_k_:= transform::TypeLinear(op(f),[x]))<>FALSE then
         // f = cos(a*x + b)
         _a := _k_[1]:
         _b_:= _k_[2]:
         return(PI*(exp(-_b_*I)*dirac(y - _a)
                   +exp( _b_*I)*dirac(y + _a)));
      end_if;
      // Walter 3.5.06: added the following pattern
      if not has((_a:= op(f)/x^2),x) then
        if is(_a in R_) = TRUE then
           return(sqrt(PI)/sqrt(abs(_a))/sqrt(2)*( cos(y^2/4/_a) + sin(y^2/4/abs(_a)) ));
        end_if;
      end_if;
      // Walter 3.5.06: added the following pattern
      // (generalizes the pattern above)
      if testtype(op(f), Type::PolyExpr(x)) and
         degree(op(f), [x]) = 2 then
           //   cos(a*x^2 + b*x + c)
           // = cos(a*(x + b/2/a)^2 + c - b^2/4/a)
           // = cos(a*(x + b/2/a)^2) * cos(c - b^2/4/a)
           //  -sin(a*(x + b/2/a)^2) * sin(c - b^2/4/a)
           // Use fourier(f(a*(x + d), x, y)
           //   = exp(-I*d*y)*fourier(f(a*x), x, y)
           //   = exp(-I*d*y)/abs(a)*fourier(f(x), x, y/a) (a reell):
           [_c, _b, _a]:= [coeff(op(f), [x], All)];
           if is(_a in R_) = TRUE then
             return(-exp(-I*_b/2/_a*y)*sin(_c-_b^2/4/_a)*
                         sqrt(PI)/sqrt(abs(_a))/sqrt(2)*(sign(_a)*cos(y^2/4/_a)-sin(y^2/4/_a) )
                    +exp(-I*_b/2/_a*y)*cos(_c-_b^2/4/_a)*
                         sqrt(PI)/sqrt(abs(_a))/sqrt(2)*(cos(y^2/4/_a)+sin(y^2/4/abs(_a)) )
                    );
           end_if;
      end_if;
      break
   //=================================================================
   of "_power" do // expr^k 
      _k_:=op(f,2);
      if op(f,1)=x and testtype(_k_,Type::PosInt) then
         return(2*PI*(-I)^_k_*dirac(y,_k_))
      elif _k_=-1 then
         delete _a, _b_, _c_;
         if Match2(expand(op(f,1)),indep(_a,x)*x+indep(_b_,x)) then
            // ===========================
            // fourier(1/(a*x + b) , x, y)
            // ===========================
            Imba:= Im(_b_/_a);
            if is(Imba < 0) = TRUE then
               return( I*2*PI*exp(-I*y*_b_/_a)/_a*heaviside(y));
            elif is(Imba > 0) = TRUE then
               return(-I*2*PI*exp(-I*y*_b_/_a)/_a*heaviside(-y))
            elif is(Imba = 0) = TRUE then
               return(I*PI*exp(-I*y*_b_/_a)/_a*(2*heaviside(y) - 1));
            else 
               // Beware! The shift rule
               // fourier(f(a*x + b), x, y) = exp(-I*y*b/a)/abs(a)*fourier(f(x), x, y/a)
               // if only valid for real shifts. With f(x) = 1/x and
               // fourier(1/x, x, y) = PI*I*(2*heaviside(y)-1) the shift rule
               // yields (exp(-I*y*_b_/_a)/_a*  PI*I*(2*heaviside(y) - 1)) for
               // Im(b/a) = 0. Here is the general formula for arbitrary shifts:

               return(-  PI*I/_a             *exp(-_b_/_a*y*I)
                      -  PI*I/_a*sign(Imba)  *exp(-_b_/_a*y*I) 
                      +2*PI*I/_a*heaviside(y)*exp(-_b_/_a*y*I));
               /* -------------------------------
               // The following is an alternative representation:
               return(exp(-I*y*_b_/_a)/_a*PI*I*(
                          (1 - sign(abs(Imba)))* (2*heaviside(y) - 1)
                          - 2*sign(Imba) * heaviside(-sign(Imba)*y) )):
               ------------------------------- */
            end_if;
         elif Match2(expand(op(f,1)),indep(_a,x)*x^2+indep(_b_,x)*x+indep(_c_,x)) then
            // ===================================
            // fourier(1/(a*x^2 + b*x + c) , x, y)
            // ===================================
            _b_:=_b_/2/_a; _c_:=_c_/_a;

            if _b_=0 then
               if sign(_c_)=-1 then
                    _c_:=sqrt(-_c_);
                    return(-PI/_c_/_a*sin(_c_*y)*sign(_c_*y)) // formula 1.29 p. 5 
               else _c_:=sqrt(_c_); 
               //   return(PI/_c_/_a*exp(-_c_*y)) # formula 1.25 p. 5 #
                    return( PI/_c_/_a*exp(-_c_*y)*heaviside( y)
                           +PI/_c_/_a*exp( _c_*y)*heaviside(-y))
               end_if
            end_if; 
            // 1/(a*x^2+b*x+c) = 1/a*1/((x+b/2/a)^2+c/a-b^2/4/a^2) 
            _c_:=sqrt(normal(_b_^2-_c_));
            // now 1/a/((x+b)^2-c^2) = 1/2/a/c*(1/(x+b-c)-1/(x+b+c)) 
            if not iszero(_c_) then
             // return(exp(-I*_b_*y)/_a * transform::fourier(1/(x^2-_c_^2), x, y));
                return(1/(2*_a*_c_)*( transform::fourier(1/(x+_b_-_c_),x,y)
                                    - transform::fourier(1/(x+_b_+_c_),x,y)))
            else 
                return(procname(args())); // !! TODO, insert formula here
            end_if:
         elif Match2(expand(op(f,1)),x^4+indep(_a,x)) then
            // ===================================
            // fourier(1/(x^4 + a) , x, y)
            // ===================================
            _a:=sqrt(sqrt(_a)); // formula 1.51 p. 7 
             return( PI/_a^3*exp(-_a*y/2^(1/2))*sin(PI/4+_a*y/2^(1/2))*heaviside( y)
                    +PI/_a^3*exp( _a*y/2^(1/2))*sin(PI/4-_a*y/2^(1/2))*heaviside(-y)
                     )
         end_if;
      elif not has((_a:=expand(op(f,1))-x^2),x) then
          // ===================================
          // fourier((x^2 + a)^k , x, y)
          // ===================================
         if type(_a) = "_power" then
            _a:= subsop(_a, 2 = op(_a, 2)/2);
         else
            _a:=sqrt(_a); // (x^2+a^2)^_k_ 
         end_if:
          // ===================================
          // Now:  fourier((x^2 + a^2)^k , x, y)
          // ===================================
         if testtype((m:=-_k_-1),Type::PosInt) then // formula 1.57 p. 7 
           // Walter 18.4.06: formula 1.57 in Oberhettinger is only valid
           // for y >= 0 and Re(_a) > 0. Here a general result:
           if Re(_a) = 0 then // singularity on the real line
              return(procname(args())); 
           end_if;
           return(
                  2*PI*(2*_a)^(-m-1)*exp(-_a*y)*heaviside(y)/fact(m)*
                  _plus(fact(m+_k_)/fact(m-_k_)/fact(_k_)/(2*_a)^_k_*( y)^(m-_k_) $ _k_=0..m)
                + 2*PI*(2*_a)^(-m-1)*exp( _a*y)*heaviside(-y)/fact(m)*
                  _plus(fact(m+_k_)/fact(m-_k_)/fact(_k_)/(2*_a)^_k_*(-y)^(m-_k_) $ _k_=0..m)
                 );
         elif is(Re(-_k_)>0)=TRUE then // formula 2.7 p. 8, includes formula 1.32 p. 5 
            if Re(_a) = 0 then // singularity on the real line
               return(procname(args())); 
            end_if;
            v:=-_k_-1/2;
            _a:= _a^2;
            return(2*PI^(1/2)/2^v/gamma(v+1/2)*abs(y)^v*_a^(-v/2)*besselK(v,sqrt(_a)*abs(y)))
         end_if
      elif testtype((m:=-_k_-1),Type::PosInt) then // f = g(x)^(-k)
          if op(f, 1) = x then
             // ===================================
             // fourier(1/x^k , x, y)
            // ===================================
             // remark: 2*heaviside(y) - 1 = sign(y)
             return(I^(m+1)*y^m*PI/m! * (2*heaviside(y) - 1)); 
          end_if;
          delete _a, _b_;
          if Match2(expand(op(f,1)),indep(_a,x)*x+indep(_b_,x)) then
             // ===================================
             // fourier(1/(a*x + b)^k , x, y)
             // ===================================
             // For Im(b/a) = 0, use the shift rule
             // fourier(f(a*x + b), x, y) = exp(-I*y*b/a)/abs(a)*fourier(f(x), x, y/a)
             // resulting in:
             // return(I^(m+1)*y^m/_a^(m+1)*PI/m!*exp(-I*y*_b_/_a)*(2*heaviside(y)-1));
             // For general complex shifts:
             Imba:= Im(_b_/_a):
             return(-  PI*I^(m+1)*y^m/_a^(m+1)/m!             *exp(-_b_/_a*y*I)
                    -  PI*I^(m+1)*y^m/_a^(m+1)/m!*sign(Imba)  *exp(-_b_/_a*y*I) 
                    +2*PI*I^(m+1)*y^m/_a^(m+1)/m!*heaviside(y)*exp(-_b_/_a*y*I));
             /* ------------ an alternative representation -------
             return(I^(m+1)*y^m/_a^(m+1)*PI/m!*exp(-I*y*_b_/_a)*(
                          (1 - sign(abs(Imba)))* (2*heaviside(y) - 1)
                         - 2*sign(Imba) * heaviside(-sign(Imba)*y) ));
             ----------------- */
          end_if;
      end_if;
      break
   //=================================================================
   of "diff" do
    //if op(op(f,1))=x
    //  then return((-I*y)^(nops(f)-1)*procname(op(f,1),x,y))
    //end_if;
      if op(f,2) = x
        then return(-I*y*transform::fourier(diff(op(f,1),op(f,3..nops(f))),x,y))
      end_if;
      break
   of "_mult" do
      if (_k_:=select(f,has,x))<>f then
         return(f/_k_*transform::fourier(_k_,x,y))
      end_if:
      // now all factors contain x 
      delete n, _k_;
      if Match2(f,x^_p(n,Type::PosInt)*_p(_k_)) then
         // fourier(x^n*f(x),x,y)=(-I)^n*diff(fourier(f(x),x,y),y$n) 
         yy:= genident("yy");
         // return((-I)^n*eval(subs(diff(transform::fourier(_k_,x,yy),yy$n),yy=y)))
         tmp:= transform::fourier(_k_,x,yy):
         if type(tmp) <> "transform::fourier" then
            tmp:= diff(tmp, yy $ n);
            if type(tmp) <> "diff" or      // beware: if y is not an unknown,
               domtype(y) = DOM_IDENT or   // then do not substitute this into
               type(y) = "_index" then     // a symbolic diff expression!
               return( (-I)^n*subs(tmp, yy = y,EvalChanges) );
            end_if;
         end_if;
      end_if;

      //---------------------------------------------------------
      // fourier(exp(a*x)*f(x), x, y) = fourier(f(x), x, y - I*a)
      //---------------------------------------------------------
/* -----------
      // Walter 24.6.08: the following rewrite to exp would make
      // fourier(heaviside(aa*x + b)*sin(a*x + b)) etc. work. On 
      // the other hand, if fourier is not successful, the rewritten
      // expression would be returned. Activate this code?
      if has(f, heaviside) and 
         (not has(f, exp)) and
         (has(f, cos) or has(f, sin)) then
           f:= rewrite(f, exp)
      end_if:
----------- */
      if has(f, exp) then
        for i from 1 to nops(f) do
         if type(op(f,i)) = "exp" and 
            (_k_:= transform::TypeLinear(op(op(f,i)),[x]))<>FALSE then 
            // f = ff * stuff, ff = exp(_a*x + _b_)
              _a := _k_[1]:
              _b_:= _k_[2]:
              g:= eval(subsop(f, i = 1)):
              h:= transform::fourier(g, x, y - I*_a);
              // Walter 10.6.07: restrict the exp-shift-rule
              //   fourier(exp(a*x)*f(x),x,y) = fourier(f(x),y,y-I*a)
              // to proper transformations, do not apply it to distributions!
              // E.g., note that
              //   transform::fourier(heaviside(x),x,y) = I/y + PI*dirac(y),
              // whereas
              //   transform::fourier(exp(-x)*heaviside(x),x,y) = I/(y + I)
              if is(Re(_a) = 0) = TRUE or (
                   not hastype(h, "transform::fourier") and 
                   not hastype(h, "dirac") and
                   not hastype(g, "heaviside") and 
                   not hastype(g, "abs") and 
                   not hastype(g, "sign") 
                  ) then
                  return(exp(_b_)*h);
              end_if:
              // We need to make use of the pattern for
              //   fourier(exp(a*x + b)*heaviside(c*d + d) 
              // which is implemented **for all** (symbolic)
              // values of a, b, c, d. We
              
              if hastype(g, "abs") then  // abs(x) = x*sign(x) for x in R_
                 g:= subs(g, hold(abs) = proc(z) begin 
                                            if has(z, x) then
                                               return(2*z*heaviside(z) - z);
                                            end_if;
                                            hold(abs)(z)
                                         end_proc, EvalChanges);
                 g:= expand(g, ArithmeticOnly);
              end_if;
              if hastype(g, "sign") then   // sign(x) = 2*heaviside(x) - 1
                 g:= subs(g, hold(sign) = proc(z) begin 
                                            if has(z, x) then
                                               return(2*heaviside(z) - 1);
                                            end_if;
                                            hold(sign)(z)
                                          end_proc, EvalChanges);
                 g:= expand(g, ArithmeticOnly);
              end_if;
              if type(g) = "_plus" then
                 return(_plus(exp(_b_)*transform::fourier(exp(_a*x)*tmp, x, y) $ tmp in g));
              end_if;
              // The fourier transform of a heaviside contains dirac.
              // Since the exp-shift-rule above does not apply, we
              // need to produce the result explicitly:
              if type(g) = "heaviside" and 
                 (_k_:= transform::TypeLinear(op(g),[x]))<>FALSE then 
                 // f = exp(_a*x + _b_) * heaviside(_aa*x + _bb)
                 _aa:= _k_[1]:
                 _bb:= _k_[2]:
                 if not iszero(_aa) then
                   return(-sign(_aa)*exp(-_bb/_aa*(I*y + _a) + _b_)/(I*y + _a) 
                    + (1+sign(Re(_a)/_aa))/2*exp(_b_)*transform::fourier(exp(_a*x), x, y));
                 end_if;
              end_if:
          end_if:
        end_for:
      end_if;

      // Walter 2.2.10: implemented further patterns F(x)*heaviside(a*x + b):
      if hastype(f, "heaviside") then
         [h, F, dummy]:= split(f, g -> type(g) = "heaviside");
         if type(h) = "heaviside" and // h could be a product of heavisides
            (tmp:= transform::TypeLinear(op(h),[x]))<>FALSE then
            [a, b]:= tmp;
            assert(not iszero(a)); // otherwise, the constant heaviside(b) 
                                   // would have been extracted
            //-----------------------------------------------------
            // fourier(F(x)*heaviside(a*x + b), x, y).
            //-----------------------------------------------------
            if type(F) = "exp" then
               [e, F]:= [F, 1];
               if (tmp:= transform::TypeLinear(op(e),[x]))<>FALSE then
                  [A, B]:= tmp;
               else 
                  [A, B]:= [NIL, NIL];
               end_if;
            elif type(F) = "_mult" and hastype(F, "exp") then
               [e, F, dummy]:= split(F, g -> type(g) = "exp");
               if type(e) = "_mult" then
                  e:= combine(e, exp):
               end_if;
               if type(e) = "exp" and
                  (tmp:= transform::TypeLinear(op(e),[x]))<>FALSE then
                  [A, B]:= tmp;
               else 
                  [A, B]:= [NIL, NIL];
               end_if;
            else
               e:= 1;
               [A, B]:= [0, 0]
            end_if;

            if [A, B] <> [NIL, NIL] then 

              // ================================================================
              fourierExp:= proc(A, B, a, b, x, y, factor = 1) 
              // returns fourier(exp(A*x+B)*heaviside(a*x+b), x, y) 
              // for general complex A,B,y and general real a,b,x. 
              option remember;
              begin
                if is(Re(A) = 0) = TRUE then 
                   return(+factor*PI*dirac(y+Im(A))*exp(B)*exp(-b/a*A)*cos(b/a*Im(A)) 
                          +factor*PI*dirac(y+Im(A))*exp(B)*exp(-b/a*A)*sin(b/a*Im(A))*I
                          -factor/(A+y*I)*sign(a)*exp(B)*exp(-b/a*A)*cos(b/a*y)
                          +factor/(A+y*I)*sign(a)*exp(B)*exp(-b/a*A)*sin(b/a*y)*I):
                else
                   // The following formula is valid for all A, B, a, b, y. 
                   // The case above has only been treated separately because the
                   // trigonometric representation is nicer than the exponential one. 
                   return(
                          -factor*sign(a)*exp(-b/a*(I*y + A) + B)/(I*y + A)
                          +factor*(1+sign(Re(A)/a))/2*exp(B)*transform::fourier(exp(A*x), x, y)
                          );
                end_if;
              end_proc;
              // ================================================================

              // fourier(exp(A*x+B)*F(x)*heaviside(a*x+b)
              if not has(F, x) then
                 return(fourierExp(A, B, a, b, x, y, F));
              elif type(F) = "sin" then 
                 if (tmp:= transform::TypeLinear(op(F),[x]))<>FALSE then
                   [_aa, _bb]:= tmp;
                   // exp(A*x+B)*sin(aa*x+bb) = exp((A-aa*I)*x+B-bb*I)*I/2 - exp((A+aa*I)*x+B+bb*I)*I/2
                   return(fourierExp(A-_aa*I, B-_bb*I, a, b, x, y, I/2)
                         -fourierExp(A+_aa*I, B+_bb*I, a, b, x, y, I/2));
                  end_if;
              elif type(F) = "cos" then 
                 if (tmp:= transform::TypeLinear(op(F),[x]))<>FALSE then
                   [_aa, _bb]:= tmp;
                   // exp(A*x+B)*cos(aa*x+bb) = exp((A-aa*I)*x+B-bb*I) + exp((A+aa*I)*x+B+bb*I)
                   return(fourierExp(A-_aa*I, B-_bb*I, a, b, x, y, 1/2)
                         +fourierExp(A+_aa*I, B+_bb*I, a, b, x, y, 1/2));
                 end_if;
              end_if;
            end_if; // if [A, B] <> [NIL, NIL]
         end_if;  // if heaviside argument is linear in x
      end_if: // if hastype(f, "heaviside")

      // Walter 5.10.07: Beware: do not only match dirac(x - a),
      // but also dirac(b*x - a) = dirac(x - a/b) / |b|:
      delete _a, _k_, _b_;
      if  (g:=Match(f, dirac(indep(_b_, x)*x - indep(_a,x))*_p(_k_)))<>FAIL then
         if not iszero(_b_) then
           return(subs(_k_/abs(_b_)*exp(I*y*x),op(g), x=_a/_b_, op(g),EvalChanges))
         end_if:
      end_if:
      delete _a, _b_;
      if (g:=Match(f,1/(x^2+indep(_a,x)^2)*1/(x^2+indep(_b_,x)^2)))<>FAIL
        then
         _a:=subs(_a,op(g)); _b_:=subs(_b_,op(g));
         // formula 1.34 p. 5 
         return(PI/(_a^2-_b_^2)*(exp(-_b_*y)/_b_-exp(-_a*y)/_a)) 
      end_if:
      g:= expand(f, op(select(indets(f, RatExpr),
                              a -> not contains({"sin", "cos", "sinh", "cosh"},
                                                type(a))
                              ))); 
      if g<>f then
         return(transform::fourier(g,x,y))
      end_if; 
      break
   of "int" do
      if (_a:=Match(f,hold(int)(_p(g)(_p(z))*_p(h)(x-_p(z)),
                                _p(z)=-infinity..infinity)))<>FAIL
        then
         g:=subs(g,op(_a)); h:=subs(h,op(_a));
         return(transform::fourier(g(x),x,y)*transform::fourier(h(x),x,y))
      end_if; 
      break
   of "transform::fourier" do
      // special case:
      if op(f,3)=x then 
         return(2*PI*subs(op(f,1),op(f,2)=-y,EvalChanges))
      end_if; 
  
      // more general case with linear shift of Fourier variable:
      if (_k_:= transform::TypeLinear(op(f,3),[x]))<>FALSE then
         // f:= transform::fourier(g(z),z,_a*x + _b_);
         _a := _k_[1]:
         _b_:= _k_[2]:
         if not iszero(_a) then
            return(2*PI/abs(_a)*exp(-I*_b_*y/_a)*subs(op(f,1), op(f,2)=-y/_a,EvalChanges))
         end_if;
      end_if;
      break
    of "function" do
      //-----------------------------------------------------------------------
      // implement the 'shift rule':
      // fourier(f(a*x + b), x, y) = exp(-I*y*b/a)/abs(a)*fourier(f(x), x, y/a)
      // if a in R_ \ {0} and f is just a symbolic function name
      // Warning: this is restricted to functions with 1 argument
      // ToDo: extend this to functions with several arguments
      //-----------------------------------------------------------------------
      // For multi-argument function calls, find the index i of the argument
      // that contains x:
      i:= nops(f) + 1;
      for ii from 1 to nops(f) do
        if has(op(f, ii), x) then 
           if ii > i then 
              // This is a function call with several arguments.
              // We already found an argument containing x before.
              // So, there at least 2 of the arguments contain x, 
              // we cannot do anything. 
              return(procname(args()));
           else
              i:= ii;
           end_if;
        end_if;
      end_for;
      if (_k_:= transform::TypeLinear(op(f,i),[x]))<>FALSE then
         // f:= H(_a*x + _b_);
         _a := _k_[1]:
         _b_:= _k_[2]:
         if not iszero(_a) and
            // exclude the infinite recursion for f = H(x), i.e., _a=1 and _b_ = 0)
            (not iszero(_a - 1) or not iszero(_b_)) and
            // we have to assume (implicitly) that _a is real
            not domtype(_a) = DOM_COMPLEX
         then
            return(exp(-I*_b_*y/_a)/abs(_a)*transform::fourier(
                 op(f, 0)((op(f, ii) $ ii = 1..i-1), 
                           x, 
                           (op(f, ii) $ ii = i+1..nops(f)) 
                          ), x, y/_a))
         end_if;
      end_if;
      break
   end_case;



   if (g:=Match(f,x^(2*_p(m,Type::PosInt))/(x^2+indep(z,x))^(_p(n)+1)))<>FAIL
     then
      m:=subs(m,op(g)); n:=subs(n,op(g));
      if testtype(n+1-m,Type::PosInt) then
         // formula 1.55 p. 7 
         return((-1)^(m+n)*PI/fact(n)*subs(diff(z^(m-1/2)*
                                                exp(-y*z^(1/2)),z$n),op(g),EvalChanges))
      end_if
   elif (g:=Match(f,x^(2*_p(m,Type::PosInt))/
                  (indep(_a,x)+x^(2*_p(n,Type::PosInt)))))<>FAIL
     then
      m:=subs(m,op(g)); n:=subs(n,op(g)); _a:=subs(_a,op(g))^(1/2/n);
      // formula 1.56 p. 7 
      return(PI/n*_a^(2*m-2*n+1)*_plus(sin((2*m+1)*(_k_-1/2)*PI/n+_a*y
        *cos((_k_-1/2)*PI/n))*exp(-_a*y*sin((_k_-1/2)*PI/n))$ _k_=1..n))
   end_if;
   procname(args())
end_proc:

transform::fourier:=funcenv(transform::fourier):
transform::fourier::type := "transform::fourier":
transform::fourier::interface := {hold(addpattern)}:
transform::fourier::diff :=
proc(F, z)
  local f, x, y;
begin
  // F = transform::fourier(f, x, y)
  f:= op(F, 1):
  x:= op(F, 2):
  y:= op(F, 3):
  // Mathematically, F does not depend on x:
  if z = x then return(0) end_if; 
  return(transform::fourier(diff(f, z), x, y) 
         +transform::fourier(f*I*x*diff(y,z), x, y));
end_proc:

//------------------------------------------------
// overloading of evalAt is needed for testeq etc.
// Ignore substitutions of the second operand!
//------------------------------------------------
transform::fourier::evalAt:= proc(f, subst)
local g, x, xx, y, yy;
begin
   assert(type(f) = "transform::fourier");
   [g, x, y]:= [extop(f)];
   // ignore substitutions of the bound variable x:
   subst:= select(subst, y -> (op(y, 1) <> x) );
   // when the bound variable x is in the rhs's of the
   // substitutions, change the name of the variable:
   if contains(freeIndets(map(subst, op, 2)), x) then
      xx:= genident(expr2text(x));
      f:= subs(f, x = xx);
      g:= subs(g, x = xx);
      x:= xx;
   end_if;
   g:= g | subst;
   yy:= y | subst;
   if extop(f, 1) <> g or y <> yy then
      return(transform::fourier(g, x, yy))
   else
      return(f)
   end_if;
end_proc:
//------------------------------------------------

transform::fourier::freeIndets:= proc(f)
begin
   assert(type(f) = "transform::fourier");
   freeIndets([extop(f, 1), extop(f, 3)], args(2..args(0))) minus {extop(f, 2)};
end_proc:

//------------------------------------------------


autoload(transform::fourier::lookup):
transform::fourier::patternFSA:=
loadproc(transform::fourier::patternFSA,
         pathname("TRANS","FOURIER"), "load_patterns"):
autoload(transform::fourier::addpattern):

transform::fourier::userpatterns := []:
transform::fourier::userpatternsFSA := FAIL:

