/* -----------------------------------------------------
unit -- the domain of physical units

In this file, only the original functionality is provided.
This way, library calls to simple unit (as done by plot)
do not load the whole unit domain.
Whenever unit::someslot is called with someslot not being
defined in this file, the extended functionality is loaded
automatically via read(unit2.mu) (see unit::make_slot at
the very end of this file).
--------------------------------------------
Details:
*) The internal data structure is
    unit::xyz = unit("xyz") = new(dom, "xyz")
   (the slots are nothing but the new-method of 'unit'!)
*) Products, sums, powers of units are arithmetical
   expressions.
*) The domain allows multiplication with arbitrary
   expressions:
    >> x:= 12*unit::inches*unit::m
    >> y:= 34*unit::cm
    >> z:= x/y
                           6 m inches
                           ----------
                              17 cm
    >> unit::convert(z, unit::inch)
                           600 inches
                           ----------
                               17
    >> unit::simplify(z);
                           1524 [cm]
                           ---------
                               17
--------------------------------------------
Methods:

unit::simplify(expression)
              - convert all units in expression to some
                basic units found in expression, i.e.,
                all length units are expressed by the same
                length unit,
                all mass units are expressed by the same
                mass unit,
                all time units are expressed by the same
                time unit.
unit::convert(expression, targetunit):
              - convert all units in expression to the
                targetunit if a conversion is possible.
                targetunit may be an expression such as
                unit::km/sec
unit::expr    - expr(unit::xyz) = unit::xyz
unit::findUnits - unit::findunits(expression) returns a
                  set of all units found in the expression
unit::expr2text - overloading expr2text
unit::ConversionTable - to be documented
unit::newUnit  - unit::newUnit(lightyear = 9 460 500 000 000*unit::km)
                 erzeugt die neue Einheit unit::lightyear und
                 traegt lightyear in unit::ConversionTable ein
unit::basicconvert    - utility
unit::samebaseunit    - utility
--------------------------------------------------------------------*/

unit:= newDomain("unit"):
unit::info:= "the domain of physical units":
unit::new:= proc() option hold; begin  slot(unit, expr2text(args())) end:
unit::expr:= x -> x:

//--------------------------------------------------------------------
// To add a new unit such as unit::xyz, say, just add a further entry
// "xyz" = [conversion_factor, "baseunit"] to the following table. The
// entries unit::xyz of the unit domain are generated automatically
// from this table further down below.
// Beware: the following definition of the table is not the final form
// of this table. After creation of the entries unit::xyz, the indices
// and values of this table are modified by replacing the strings "mm",
// "m" etc. by the corresponding units unit::mm, unit::m etc:

unit::ConversionTable:= table(
   //----------------------------------------
   // For length, we chose the baseunit unit::m (= unit::meter)
   //----------------------------------------
   "mm"         = [10^(-3),                "m"],
   "millimeter" = [10^(-3),                "m"],
   "millimeters"= [10^(-3),                "m"],
   "cm"         = [10^(-2),                "m"],
   "centimeter" = [10^(-2),                "m"],
   "centimeters"= [10^(-2),                "m"],
   "dm"         = [10^(-1),                "m"],
   "decimeter"  = [10^(-1),                "m"],
   "decimeters" = [10^(-1),                "m"],
   "m"          = [1,                      "m"],
   "meter"      = [1,                      "m"],
   "meters"     = [1,                      "m"],
   "km"         = [10^3,                   "m"],
   "kilometers" = [10^3,                   "m"],

   "inch"       = [(127/5/10^3),           "m"],
   "inches"     = [(127/5/10^3),           "m"],
   "yd"         = [ 36   * (127/5/10^3),   "m"], // : 1 yard  = 3 foot  = 36 inch
   "yard"       = [ 36   * (127/5/10^3),   "m"], // : 1 yard  = 3 foot  = 36 inch
   "yards"      = [ 36   * (127/5/10^3),   "m"], // : 1 yard  = 3 foot  = 36 inch
   "ft"         = [ 12      * (127/5/10^3),"m"], // Foot   : 1 ft   = 12 inch
   "foot"       = [ 12      * (127/5/10^3),"m"], // Foot   : 1 ft   = 12 inch
   "feet"       = [ 12      * (127/5/10^3),"m"], // Foot   : 1 ft   = 12 inch
   "mile"       = [ 1760*36 * (127/5/10^3),"m"], // Mile   : 1 mile = 1760 yd
   "miles"      = [ 1760*36 * (127/5/10^3),"m"], // Mile   : 1 mile = 1760 yd

   "pt"         = [(1757299/5000000/10^3), "m"],
   "point"      = [(1757299/5000000/10^3), "m"],
   "points"     = [(1757299/5000000/10^3), "m"],
   //----------------------------------------
   // For mass, we chose the base unit unit::kg (= unit::kilogram)
   //----------------------------------------
   "mg"         = [10^(-6),               "kg"],
   "milligram"  = [10^(-6),               "kg"],
   "milligrams" = [10^(-6),               "kg"],
   "g"          = [10^(-3),               "kg"],
   "gram"       = [10^(-3),               "kg"],
   "grams"      = [10^(-3),               "kg"],
   "kg"         = [1,                     "kg"],
   "kilogram"   = [1,                     "kg"],
   "kilograms"  = [1,                     "kg"],
   "pound"      = [(45359237/100000000),  "kg"],
   "pounds"     = [(45359237/100000000),  "kg"],
   "ounce"      = [(45359237/1600000000), "kg"],
   "ounces"     = [(45359237/1600000000), "kg"],
   //----------------------------------------
   // For time, we chose the base unit unit::sec (= unit::second)
   //----------------------------------------
   "ns"         = [1/10^9,                "sec"],
   "nsec"       = [1/10^9,                "sec"],
   "nanosec"    = [1/10^9,                "sec"],
   "microsec"   = [1/10^6,                "sec"],
   "ms"         = [1/10^3,                "sec"],
   "msec"       = [1/10^3,                "sec"],
   "millisec"   = [1/10^3,                "sec"],
   "s"          = [1,                     "sec"],
   "sec"        = [1,                     "sec"],
   "min"        = [60,                    "sec"],
   "minute"     = [60,                    "sec"],
   "minutes"    = [60,                    "sec"],
   "h"          = [3600,                  "sec"],
   "hour"       = [3600,                  "sec"],
   "hours"      = [3600,                  "sec"],
   "day"        = [24*3600,               "sec"],
   "days"       = [24*3600,               "sec"],

   //----------------------------------------
   // For data size, we chose the base unit unit::bit
   //----------------------------------------

   "bit"        = [1,                     "bit"],
   "Bit"        = [1,                     "bit"],
   "kBit"       = [2^10,                  "bit"],
   "kbit"       = [2^10,                  "bit"],
   "MBit"       = [2^20,                  "bit"],
   "Mbit"       = [2^20,                  "bit"],
   "GBit"       = [2^30,                  "bit"],
   "Gbit"       = [2^30,                  "bit"],
   "TBit"       = [2^40,                  "bit"],
   "Tbit"       = [2^40,                  "bit"],
   "Byte"       = [8,                     "bit"],
   "byte"       = [8,                     "bit"],
   "B"          = [8,                     "bit"],
   "kByte"      = [2^10*8,                "bit"],
   "kbyte"      = [2^10*8,                "bit"],
   "kB"         = [2^10*8,                "bit"],
   "MByte"      = [2^20*8,                "bit"],
   "Mbyte"      = [2^20*8,                "bit"],
   "MB"         = [2^20*8,                "bit"],
   "GByte"      = [2^30*8,                "bit"],
   "Gbyte"      = [2^30*8,                "bit"],
   "GB"         = [2^30*8,                "bit"],
   "TByte"      = [2^40*8,                "bit"],
   "Tbyte"      = [2^40*8,                "bit"],
   "TB"         = [2^40*8,                "bit"] 
):

// --------------------------------------------------------------
// Automatic generation of the entries unit::xyz of the domain 'unit'
// from the conversion table. Internally, these slots are domain elements.
// We use the fact that the kernel processes domain elements in a way
// in a way very similar to identifiers
//  *** provided that the arithmetical functions are not overloaded! ***
// e.g.,  2*u + 3*u --> 5*u, where u = unit::xyz = new(dom, "xyz") is
// a domain element.

map([op(unit::ConversionTable)],
    u -> (
         (slot(unit, lhs(u))):= new(unit, lhs(u));
      // (slot(unit, lhs(u))):= funcenv( () -> procname(args()) );
      // (slot(unit, lhs(u)))::print := lhs(u):
      // (slot(unit, lhs(u)))::isUnit:= TRUE;
         u:
   )):

alias(CT = unit::ConversionTable):

// --------------------------------------------------------------
// rewrite the conversion table by changing the index
// from "cm" to unit::cm and changing the base units
// in the values accordingly. Afterwards
// unit::ConversionTable:= table(
//    unit::mm = [10^(-3), unit::m],
//    unit::cm = [10^(-2), unit::m],
//    etc.):

unit::ConversionTable:= table(
  slot(unit, lhs(u)) = [rhs(u)[1], slot(unit, rhs(u)[2])]
  $ u in [op(unit::ConversionTable)]
):

//-------------------------------------------
// print & Content

unit::print:= x -> extop(x, 1):

//--------------------------------------------------------------------
// Content
//
// out = output domain = {Content, MMLContent}
// data = a single object of type unit
//
// Examples calling this method:
//     Content(unit::microOhm)
//     generate::MathML(unit::Celsius)

unit::Content := proc(out, data)
local s, l;
begin
  // if new special strings are added, keep back-translation of Cunit in
  // Content.mu in sync and add an example in OUTPUT/TEST/Content.dev.tst
   s := extop(data, 1); // the internal string
   if   stringlib::contains(s, "ohm") then
        s:= stringlib::subs(s, "ohm" = "&Omega;")
   elif stringlib::contains(s, "Ohm") then
        s:= stringlib::subs(s, "Ohm" = "&Omega;")
   elif stringlib::contains(s, "celsius") then
        s:= stringlib::subs(s, "celsius" = "&deg;C")
   elif stringlib::contains(s, "Celsius") then
        s:= stringlib::subs(s, "Celsius" = "&deg;C")
   elif stringlib::contains(s, "fahrenheit") then
        s:= stringlib::subs(s, "fahrenheit" = "&deg;F")
   elif stringlib::contains(s, "Fahrenheit") then
        s:= stringlib::subs(s, "Fahrenheit" = "&deg;F")
   end_if;
   // Take care of the prefixex 'mc' = 'micro':
   l := length(s);
   if l > 2 and substring(s, 1..2) = "mc" then
        s:= hold(`&mu;`).substring(s, 3..l);
   elif l > 5 and substring(s, 1..5) = "micro" and
        not s[1..6] = "micron" then
        s:= hold(`&mu;`).substring(s, 6..l);
   else s:= hold(``).s;
   end_if;
   return(out::Cunit(s));
end_proc:

//----------------------------------
// expr2text

unit::expr2text:= x -> "unit::".extop(x, 1):

//----------------------------------
// unit::samebaseunit
//
// unit::samebaseunit(u1 : unit, u2 : unit) checks
// whether u1 and u2 are powers of the same physical
// basic 'type' (length, time, or mass). It returns
// TRUE or FALSE. E.g.,  u1 = unit::cm^2 and
// u2 = unit::inches^3 have the 'same base unit'.

unit::samebaseunit:= proc(u1, u2)
begin
  bool(CT[u1][2] = CT[u2][2])
end_proc:

//----------------------------------
// unit::findUnits(ex)
//
// A utility function: it searches the MuPAD object ex and
// returns a set of all units that are present in ex. It only
// considers first powers, i.e., the returned set containss a
// unit such as unit::cm, but not unit::cm^2.

unit::findUnits:=proc(ex)
local exunits;
begin
  // Do a complete search of the expression tree ex
  // for the units occuring in ex. Store all units
  // that are found (store in the table exunits).
  // Note that we expect typical expressions involving
  // units to be fairly simple, so this should not be
  // too expensive:
  exunits:= table():
  misc::maprec(ex,
               (x -> testtype(x, unit)) // selector criterion
               = // enter the fact that the unit x was
                 // found in ex in the table exunits
               (x -> (exunits[x]:= 1; x))
              ):
  // extract all units that were found in ex from the table;
  // return them as a set.
  map({op(exunits)}, op, 1);
end_proc:

//----------------------------------
// unit::basicconvert(a, b) (utility method)
//
// rewrite the unit a = unit::xyz as
//    (numerical factor)* b, if possible.
// E.g.
// >> unit::basicconvert(unit::inches, unit::mm)
//
//                          127 mm
//                          ------
//                             5
//

unit::basicconvert:= proc(a, b)
begin
   if a = b then
      return(a);
   end_if;
   if not unit::samebaseunit(a, b) then
        return(a)
   else return(CT[a][1]/CT[b][1]*b);
   end_if:
end_proc:

//----------------------------------
// method unit::convert(ex, target)
//
// convert all units in an expression 'ex'
// to the unit type of 'target'. E.g.,
// >> unit::convert(3*unit::mm*unit::inches, unit::cm)
//                                2
//                          381 cm
//                          -------
//                            500
//----------------------------------

unit::convert:= proc(ex, target)
local u;
begin
  if testargs() then
     if args(0) <> 2 then
        error("expecting 2 arguments");
     end_if;
  end_if;

  if has(target, FAIL) then
    return(ex)
  end_if;

  //--------------------------------
  // special cases (for speed):
  //--------------------------------
  if ex = target then return(ex) end_if;

  // Modified by O.K.
  if has(map(unit::findUnits(target), unit::DefinedInCT), FALSE) or
     has(map(unit::findUnits(ex    ), unit::DefinedInCT), FALSE) then
     return(unit::convert2(ex, target))
  end;

  //--------------------------------
  // target is a composite expression
  // E.g.: convert(ex,  unit::mm/unit::sec)
  //--------------------------------
  if domtype(target) = DOM_EXPR then
     for u in unit::findUnits(target) do
         ex:= unit::convert(ex, u)
     end_for:
     return(ex);
  end_if;

  //--------------------------------
  // target is an elementary unit.
  // E.g.: convert(ex,  unit::mm)
  //--------------------------------
  if testargs() then
    if domtype(target) <> unit then
        error("second argument: expecting a unit");
    end_if;
  end_if;

  // basic conversion unit::inch -> unit::cm etc:
  if testtype(ex, unit) then
     if unit::samebaseunit(ex, target) then
          // both ex and target are of the same
          // type and we need to convert:
          return(unit::basicconvert(ex, target));
     else // ex and target are of different type.
          // No Conversion is possible:
          return(ex);
     end_if;
  end_if;

  //------------------------------------
  // generic conversion via misc::maprec
  //------------------------------------
  return(misc::maprec(
         ex,
         (x -> testtype(x, unit)) // selector criterion
         = // replace the unit x by its conversion to target:
         (x -> if (x = target) or not unit::samebaseunit(x, target) then
                    return(x)
               else return(unit::basicconvert(x, target));
               end_if)
              )
        ):
end_proc:

//-----------------------------------------------
// unit::coerce(ex)
// A call such as coerce(something, unit) does not
// make sense. Make sure, this call does not produce
// anything.
//-----------------------------------------------
unit::coerce:= ex -> FAIL:

//-----------------------------------------------
// unit::simplify(ex)
//
// 'unifies' the units in the expression ex,, i.e.,
// convert all units to some base units inside ex;
// >> unit::cm + 3*unit::m;
//                   unit::cm + 3 unit::m
// >> unit::simplify(%);
//                       301 unit::cm
//
// Beware: the system function simplify automatically
// touches objects that have a simplify slot. There is
// no need to simplify symbolic unit calls (there
// should not be any)


unit::simplify := proc(ex)
local units, u;
begin
  if domtype(ex) = unit then
     return(ex);
  end_if;

  if domtype(ex) <> DOM_EXPR then
     ex := expr(ex);
  end_if;

  // Find all units in the expression.
  // Only the first powers of the units
  // are returned by unit::findUnits.
  units:= unit::findUnits(ex);

  if nops(units) = 0 then
     return(ex);
  end_if;

  // Modified by O.K.
  if has(map(units, unit::DefinedInCT), FALSE) then
     return(unit::simplify2(ex))
  end;

  // Sort the units according to the sysorder
  // to guarantee a unique unification scheme.
  units:= sort([op(units)]):

  // Convert all units in the expression to common
  // units. The list 'units' stores all units
  // that still need to be considered. After
  // conversion to the type of the first element
  // in 'units', delete all units that already
  // have been converted. The number of steps in
  // repeat loop cannot be higher than the number
  // of unit types (length, time, mass):
  repeat
    u:= units[1]:
    // unify all units convertible to u:
    ex:= unit::convert(ex, u):
    // delete all units from the list 'units'
    // that were unified by the conversions above
    units:= select(units, x -> not unit::samebaseunit(x, u));
  until nops(units) = 0 end_repeat;
  return(ex);
end_proc:

//-------------------------------------------
//unit::newUnit(day = 20*unit::hour) -- creates
// the new unit unit::day and inserts it into
// the ConversionTable
//-------------------------------------------
unit::newUnit := proc(eq)
local newunit, knownunit, baseunit, f;
begin
  if args(0)=0 or type(eq) <> "_equal" then
     error("expecting an equation such as 'hour = 60*unit::minute'"):
  end_if;
  newunit:= op(eq, 1);

  // Modified by O.K.
  if domtype(newunit) <> DOM_IDENT and domtype(newunit) <> DOM_STRING then
     error("the new unit must be an identifier or a string. Got: ".expr2text(newunit))
  end_if;
  if domtype(newunit) = DOM_IDENT then
     newunit:= expr2text(newunit)
  end_if;

  knownunit:= op(eq, 2):
  for baseunit in [unit::m, unit::kg, unit::sec] do
      f := unit::convert(knownunit, baseunit)/baseunit:
      if unit::findUnits(f) <> {} then
         next;
      end_if;
      sysassign(slot(unit, newunit), new(unit, newunit)):
      sysassign(unit::ConversionTable[slot(unit, newunit)], [f, baseunit]);
      return(slot(unit, newunit));
 end_for;
 // load the extended functionality and try again
 unit::conversionTable2;
 unit::newUnit(eq);
end_proc:

/*----------------------------------
// The present implementation has no
// need for overloading type

unit::type:= proc(a)
begin
  "unit::".extop(a, 1):
end_proc:
----------------------------------*/

// all units are assumed to be positive:
unit::getprop:= proc() begin return( Dom::Interval(0,infinity) ); end_proc:
unit::sign:= 1:
unit::abs := u -> u:
unit::Re  := u -> u:
unit::Im  := u -> 0:

//----------------------------------------------------
// define some methods to avoid creation via make_slot
//----------------------------------------------------
unit::simpleVersion := TRUE: // for daily tests of the unit domain
unit::_float_mult := FAIL:
unit::_float_plus := FAIL:
unit::_float_power := FAIL:
unit::_divide:= FAIL:
unit::_negate:= FAIL:
unit::_mult:= FAIL:
unit::_plus:= FAIL:
unit::_power:= FAIL:
unit::_subtract := FAIL:
unit::_invert := FAIL:
unit::allEntries := FAIL:
unit::allAutoEntries := FAIL:
// unit::Content:=FAIL:
unit::combine := FAIL:
unit::complexity := FAIL:
unit::Simplify := FAIL:
unit::create_dom_elem := FAIL:
unit::bool :=FAIL:
unit::contains:= (x,y) -> if x=y then TRUE else FALSE end_if:
unit::enableMaprec:= FALSE:
unit::eval := FAIL:
unit::evaluate:= FAIL:
unit::expand := FAIL:
unit::expose := FAIL:
// unit::expr2text:= FAIL:
unit::domtype:= FAIL:
unit::float := FAIL:
unit::has := FAIL:
unit::hastype := FAIL:
unit::indets:= FAIL:
unit::isNeg := FAIL:
unit::isInverted := FAIL:
unit::iszero:= FAIL:
unit::length := 1:
unit::new_extelement := FAIL:
unit::Name := "unit":
unit::nops := 1:
unit::normal := 
proc(x, options)
begin
  if args(0) >=2 and (type(options) = DOM_TABLE and options[List] = TRUE) or contains({args()}, List) then 
    [x, 1]
  else  
  	x 
  end_if
end_proc:
unit::maprec := FAIL:
unit::MMLContent := FAIL:
unit::op := x -> x:
unit::operandsToSimplify:= FAIL:
unit::posteval:= FAIL:
unit::rectform:= FAIL:
unit::slot:= FAIL:
unit::subs := () -> if op(args(2), 1) = args(1) then op(args(2), 2)
                    else args(1) end_if:
unit::subsop := () -> if op(args(2), 1) = 1 then op(args(2), 2)
                      else FAIL end_if:
unit::testtype:= 
proc(x, y) 
begin
  case y 
    of Type::Arithmetical do
    of DOM_DOMAIN do
       TRUE;
       break
    otherwise
       FAIL
  end_case
end_proc:
unit::type:= FAIL:
unit::undefinedEntries := FAIL:
unit::whichEntry := FAIL:
unit::evaluateIndex := FAIL:
unit::freeIndets := {}:
unit::sortSums := FAIL:
unit::CF:= loadproc(unit::CF, pathname("GENERATE"),"CF"):
unit::hasProp := FAIL:
unit::TeX := FAIL:
unit::DefinedInCT := u -> not testtype(unit::ConversionTable[u], "_index"):

//---------------------------------------------------------
// make_slot must be the very last definition in this file!
//---------------------------------------------------------

unit::make_slot:= proc(d, entry)
begin
  sysdelete(unit::make_slot):
  sysassign(unit::simpleVersion, FALSE):
  unprotect(unit):
  read(pathname("UNIT") . "unit2.mu", Plain):
  protect(unit, Error):
  slot(d, entry);
end_proc:
