/* ------------------------------------------------------------
unit -- the domain of physical units and specific units

        Physical units like m, kg, W, ...
        Specific units like bit, byte, EUR, C, F, ...

---------------------------------------------------------------
Details:
*) The internal data structure is
           unit::xyz = unit("xyz") = new(dom, "xyz")
   (the slots are nothing but the new-method of 'unit'!)
*) Products, sums, powers of units are arithmetical expressions.
*) The domain allows multiplication with arbitrary expressions:
    >> x:= 12*unit::inches*unit::m
    >> y:= 34*unit::cm
    >> z:= x/y
                           6 m inches
                           ----------
                              17 cm
    >> unit::convert(z, unit::inch)
                           600 inches
                           ----------
                               17
    >> unit::simplify(z);
                           1524 cm
                           ---------
                               17
--------------------------------------------
Methods:

unit::simplify(expression)
              - convert all units in expression to some
                basic units found in expression, i.e.,
                all length units are expressed by the same
                length unit, 
                all mass units are expressed by the same
                mass unit,
                all time units are expressed by the same
                time unit.
unit::convert(expression, targetunit):
              - convert all units in expression to the
                targetunit if a conversion is possible.
                targetunit may be an expression such as
                unit::km/sec
unit::convert2SIunits(expression)
              - convert all units in expression to SI units.
                SI units are: m, s, kg, A, K, cd, mol. Example:
                unit::convert2SIunits(unit::W) yields
                unit::kg * unit::m^2 / unit::s^3
unit::convert2baseunits(expression)
              - convert all units in expression to "base" units.
                "base" units are the units defined in the table
                 ConversionTable2. For example:
                unit::convert2baseunits(unit::W) 
                yields unit::J / unit::s
unit::expr    - expr(unit::xyz) = unit::xyz
unit::findUnits - unit::findunits(expression) returns a
                  set of all units found in the expression
unit::expr2text - overloading expr2text
unit::ConversionTable  - to be documented
unit::ConversionTable2 - to be documented
unit::newUnit  - unit::newUnit(lightyear = 9 460 500 000 000*unit::km)
                 erzeugt die neue Einheit unit::lightyear und
                 traegt lightyear in unit::ConversionTable ein
unit::basicconvert    - utility
unit::samebaseunit    - utility 
unit::convert2        - utility used by unit::convert
unit::simplify2       - utility used by unit::simplify

--------------------------------------------------------------------*/

//--------------------------------------------------------------------
// To add a new unit such as unit::xyz, say, just add a further entry 
// "xyz" = [conversion_factor, "baseunit"] to the following table. The 
// entries unit::xyz of the unit domain are generated automatically 
// from this table further down below.
// Beware: the following definition of the table is not the final form
// of this table. After creation of the entries unit::xyz, the indices
// and values of this table are modified by replacing the strings "mm", 
// "m" etc. by the corresponding units unit::mm, unit::m etc:

unit::ConversionTable:= table(
   // siehe z.B. http://www.dietmar-knoll.de/Computer/Units.html
   //----------------------------------------------------------
   // For length, we chose the baseunit unit::m (= unit::meter)
   //----------------------------------------------------------
   "nm"         = [10^(-9),                "m"],
   "nanometer"  = [10^(-9),                "m"],
   "nanometers" = [10^(-9),                "m"],
   "My"         = [10^(-6),                "m"], // Mikron
   "micron"     = [10^(-6),                "m"], // Mikron
   "microns"    = [10^(-6),                "m"], // Mikron
   "micrometer" = [10^(-6),                "m"], // Mikron
   "micrometers"= [10^(-6),                "m"], // Mikron
   "mm"         = [10^(-3),                "m"],
   "millimeter" = [10^(-3),                "m"],
   "millimeters"= [10^(-3),                "m"],
   "cm"         = [10^(-2),                "m"],
   "centimeter" = [10^(-2),                "m"],
   "centimeters"= [10^(-2),                "m"],
   "dm"         = [10^(-1),                "m"],
   "decimeter"  = [10^(-1),                "m"],
   "decimeters" = [10^(-1),                "m"],
   "m"          = [1,                      "m"],
   "meter"      = [1,                      "m"],
   "meters"     = [1,                      "m"],
   "km"         = [10^3,                   "m"],
   "kilometer"  = [10^3,                   "m"],
   "kilometers" = [10^3,                   "m"],
   "AE"         = [149597870691,           "m"], // Astronomische Einheit
   "AU"         = [149597870691,           "m"], // Astronomical units
   "pc"         = [149597870691/tan(1/648000*float(PI)), "m"], // Parsec: 1 pc ~ 3.0857 * 10^16 m
   "parsec"     = [149597870691/tan(1/648000*float(PI)), "m"], // Parsec: 1 pc ~ 3.0857 * 10^16 m
   "lj"         = [9460730472580800,       "m"], // Lichtjahr
   "Lj"         = [9460730472580800,       "m"], // Lichtjahr
   "ly"         = [9460730472580800,       "m"], // lightyear
   "lightyear"  = [9460730472580800,       "m"], // lightyear
   "lightyears" = [9460730472580800,       "m"], // lightyear
   "Ao"         = [10^(-10),               "m"], // Angstroem
   "angstroem"  = [10^(-10),               "m"], // Angstroem
   "angstrom"   = [10^(-10),               "m"], // Angstroem
   "Angstroem"  = [10^(-10),               "m"], // Angstroem
   "Angstrom"   = [10^(-10),               "m"], // Angstroem
   "XE"         = [1.00207789*10^(-13),    "m"], // X-Einheit
   "Xu"         = [1.00207789*10^(-13),    "m"], // x unit
   "XU"         = [1.00207789*10^(-13),    "m"], // x unit
   "xu"         = [1.00207789*10^(-13),    "m"], // x unit
   "f"          = [10^(-15),               "m"], // Fermi
   "Fermi"      = [10^(-15),               "m"], // Fermi
   "fermi"      = [10^(-15),               "m"], // Fermi
   "inch"       = [(127/5/10^3),           "m"],
   "inches"     = [(127/5/10^3),           "m"],
   "zoll"       = [(127/5/10^3),           "m"],
   "Zoll"       = [(127/5/10^3),           "m"],
   "yd"         = [ 36   * (127/5/10^3),   "m"], // : 1 yard  = 3 foot  = 36 inch
   "yard"       = [ 36   * (127/5/10^3),   "m"], // : 1 yard  = 3 foot  = 36 inch
   "yards"      = [ 36   * (127/5/10^3),   "m"], // : 1 yard  = 3 foot  = 36 inch
   "Elle"       = [ 36   * (127/5/10^3),   "m"], // : 1 Elle  = 1 yard
   "Ellen"      = [ 36   * (127/5/10^3),   "m"], // : 1 Elle  = 1 yard
   "pt"         = [(1757299/5000000/10^3), "m"],
   "point"      = [(1757299/5000000/10^3), "m"],
   "points"     = [(1757299/5000000/10^3), "m"],
   "ft"         = [ 12      * (127/5/10^3),"m"], // Foot   : 1 ft   = 12 inch
   "foot"       = [ 12      * (127/5/10^3),"m"], // Foot   : 1 ft   = 12 inch
   "feet"       = [ 12      * (127/5/10^3),"m"], // Foot   : 1 ft   = 12 inch
   "ft_US"      = [1200/3937,              "m"], // Foot (US): Anwendung bei US Coast und Geodetic Survey
   "foot_US"    = [1200/3937,              "m"], // Foot (US): Anwendung bei US Coast und Geodetic Survey
   "feet_US"    = [1200/3937,              "m"], // Foot (US): Anwendung bei US Coast und Geodetic Survey
   "mile"       = [ 1760*36 * (127/5/10^3),"m"], // Mile   : 1 mile = 1760 yd
   "miles"      = [ 1760*36 * (127/5/10^3),"m"], // Mile   : 1 mile = 1760 yd
   "nmile"      = [1853.2,                 "m"], // Nautical mile (UK)
   "inm"        = [1852,                   "m"], // Nautical mile (International)
   "INM"        = [1852,                   "m"], // Nautical mile (International)
   // (Gauge: US-Maeinheit fuer die Dicke von Draehten, Blechen, Folien, Fasern usw.)
   "gg"         = [ 10^(-3) * (127/5/10^3),"m"], // Gauge  : 1 gg   = 10^(-3) inch 
   "line"       = [(1/40)   * (127/5/10^3),"m"], // Line   : 1 line = 1/40 inch
   "hand"       = [ 4       * (127/5/10^3),"m"], // Hand   : 1 hand = 4 inch (height of horses)
   "span"       = [ 9       * (127/5/10^3),"m"], // Span   : 1 span = 9 inch
   "fm"         = [ 2*36    * (127/5/10^3),"m"], // Fathom : 1 fm   = 2 yd  (Anwendung in der Seeschiffahrt)
   "fathom"     = [ 2*36    * (127/5/10^3),"m"], // Fathom : 1 fm   = 2 yd  (Anwendung in der Seeschiffahrt)
   "fathoms"    = [ 2*36    * (127/5/10^3),"m"], // Fathom : 1 fm   = 2 yd  (Anwendung in der Seeschiffahrt)
   "rod"        = [(11/2)*36* (127/5/10^3),"m"], // Rod    : 1 rod  = 11/2 yd  (Anwendung in der Geodaesie)
   "perch"      = [(11/2)*36* (127/5/10^3),"m"], // Perch  : 1 perch= 1 rod    (Anwendung in der Geodsie)
   "pole"       = [(11/2)*36* (127/5/10^3),"m"], // Pole   : 1 pole = 1 rod    (Anwendung in der Geodsie)
   "ch"         = [ 22*36   * (127/5/10^3),"m"], // Chain  : 1 ch   =   22 yd  (Anwendung in der Geodsie)
   "fur"        = [ 220*36  * (127/5/10^3),"m"], // Furlong: 1 fur  =  220 yd  (Anwendung in der Geodsie)
   "furlong"    = [ 220*36  * (127/5/10^3),"m"], // Furlong: 1 fur  =  220 yd  (Anwendung in der Geodsie)
   "furlongs"   = [ 220*36  * (127/5/10^3),"m"], // Furlong: 1 fur  =  220 yd  (Anwendung in der Geodsie)
   "mil"        = [10^(-3)  * (127/5/10^3),"m"], // mil    : 1 mil  = 10^(-3) inch
   "li"         = [ 10^(-2)*22*36*(127/5/10^3),"m"],// Link: 1 li   = 10^(-2) ch 
   "link"       = [ 10^(-2)*22*36*(127/5/10^3),"m"],// Link: 1 li   = 10^(-2) ch 
   "links"      = [ 10^(-2)*22*36*(127/5/10^3),"m"],// Link: 1 li   = 10^(-2) ch 
   //------------------------------------------------------------- 
   // For mass, we chose the base unit unit::kg (= unit::kilogram)
   //-------------------------------------------------------------
   "mcgram"     = [10^(-9),               "kg"],
   "mcgrams"    = [10^(-9),               "kg"],
   "microgram"  = [10^(-9),               "kg"],
   "micrograms" = [10^(-9),               "kg"],
   "mg"         = [10^(-6),               "kg"],
   "milligram"  = [10^(-6),               "kg"],
   "milligrams" = [10^(-6),               "kg"],
   "g"          = [10^(-3),               "kg"],
   "gram"       = [10^(-3),               "kg"],
   "grams"      = [10^(-3),               "kg"],
   "kg"         = [1,                     "kg"],
   "kilogram"   = [1,                     "kg"],
   "kilograms"  = [1,                     "kg"],
   "t"          = [10^3,                  "kg"], // ton 
   "kt"         = [10^6,                  "kg"], // kilo ton 
   "Mt"         = [10^6,                  "kg"], // mega ton 
   "lb"         = [(45359237/100000000),  "kg"],
   "pound"      = [(45359237/100000000),  "kg"],
   "pounds"     = [(45359237/100000000),  "kg"],
   "oz"         = [(45359237/1600000000), "kg"],
   "ounce"      = [(45359237/1600000000), "kg"],
   "ounces"     = [(45359237/1600000000), "kg"],
   "unze"       = [(45359237/1600000000), "kg"],
   "unzen"      = [(45359237/1600000000), "kg"],
   "Unze"       = [(45359237/1600000000), "kg"],
   "Unzen"       = [(45359237/1600000000), "kg"],
   "zentner"    = [50,                    "kg"],
   "Zentner"    = [50,                    "kg"],
   "Ztr"        = [50,                    "kg"],
   "dz"         = [100,                   "kg"], // Doppelzentner
   "doppelzentner" = [100,                "kg"], // Doppelzentner
   "Doppelzentner" = [100,                "kg"], // Doppelzentner
   "pfund"      = [1/2,                   "kg"],
   "Pfund"      = [1/2,                   "kg"],
   "Pfd"        = [1/2,                   "kg"],
   "ct"         = [1/5 * 10^(-3),         "kg"],
   "carat"      = [1/5 * 10^(-3),         "kg"],
   "karat"      = [1/5 * 10^(-3),         "kg"],
   "Karat"      = [1/5 * 10^(-3),         "kg"],
   "Kt"         = [1/5 * 10^(-3),         "kg"],
   "slug"       = [8896443230521/609600000000, "kg"], 
   //------------------------------------------------------------
   // For time, we chose the base unit unit::sec (= unit::second)
   //------------------------------------------------------------
   "ns"         = [1/10^9,               "sec"],
   "nsec"       = [1/10^9,               "sec"],
   "nanosec"    = [1/10^9,               "sec"],
   "nanosecond" = [1/10^9,               "sec"],
   "nanoseconds"= [1/10^9,               "sec"],
   "mcsec"      = [1/10^6,               "sec"],
   "mcsecond"   = [1/10^6,               "sec"],
   "mcseconds"  = [1/10^6,               "sec"],
   "microsec"   = [1/10^6,               "sec"],
   "microsecond"= [1/10^6,               "sec"],
   "microseconds"=[1/10^6,               "sec"],
   "ms"         = [1/10^3,               "sec"],
   "msec"       = [1/10^3,               "sec"],
   "millisec"   = [1/10^3,               "sec"],
   "millisecond"= [1/10^3,               "sec"],
   "milliseconds"=[1/10^3,               "sec"],
   "s"          = [1,                    "sec"],
   "sec"        = [1,                    "sec"],
   "second"     = [1,                    "sec"],
   "seconds"    = [1,                    "sec"],
   "Sekunde"    = [1,                    "sec"],
   "Sekunden"   = [1,                    "sec"],
   "min"        = [60,                   "sec"],
   "minute"     = [60,                   "sec"],
   "minutes"    = [60,                   "sec"],
   "Minute"     = [60,                   "sec"],
   "Minuten"    = [60,                   "sec"],
   "h"          = [3600,                 "sec"],
   "hour"       = [3600,                 "sec"], 
   "hours"      = [3600,                 "sec"], 
   "Stunde"     = [3600,                 "sec"], 
   "Stunden"    = [3600,                 "sec"], 
   "d"          = [24*3600,              "sec"],
   "day"        = [24*3600,              "sec"],
   "days"       = [24*3600,              "sec"],
   "Tag"        = [24*3600,              "sec"],
   "Tage"       = [24*3600,              "sec"],
   "week"       = [7*24*3600,            "sec"],
   "weeks"      = [7*24*3600,            "sec"],
   "Woche"      = [7*24*3600,            "sec"],
   "Wochen"     = [7*24*3600,            "sec"],
   "month"      = [30*24*3600,           "sec"],
   "months"     = [30*24*3600,           "sec"],
   "Monat"      = [30*24*3600,           "sec"],
   "Monate"     = [30*24*3600,           "sec"],
   "year"       = [365*24*3600,          "sec"],
   "years"      = [365*24*3600,          "sec"],
   "Jahr"       = [365*24*3600,          "sec"],
   "Jahre"      = [365*24*3600,          "sec"],

   //----------------------------------------
   // For data size, we chose the base unit unit::bit
   //----------------------------------------

   "bit"        = [1,                     "bit"],
   "Bit"        = [1,                     "bit"],
   "kBit"       = [2^10,                  "bit"],
   "kbit"       = [2^10,                  "bit"],
   "MBit"       = [2^20,                  "bit"],
   "Mbit"       = [2^20,                  "bit"],
   "GBit"       = [2^30,                  "bit"],
   "Gbit"       = [2^30,                  "bit"],
   "TBit"       = [2^40,                  "bit"],
   "Tbit"       = [2^40,                  "bit"],
   "Byte"       = [8,                     "bit"],
   "byte"       = [8,                     "bit"],
   "B"          = [8,                     "bit"],
   "kByte"      = [2^10*8,                "bit"],
   "kbyte"      = [2^10*8,                "bit"],
   "kB"         = [2^10*8,                "bit"],
   "MByte"      = [2^20*8,                "bit"],
   "Mbyte"      = [2^20*8,                "bit"],
   "MB"         = [2^20*8,                "bit"],
   "GByte"      = [2^30*8,                "bit"],
   "Gbyte"      = [2^30*8,                "bit"],
   "GB"         = [2^30*8,                "bit"],
   "TByte"      = [2^40*8,                "bit"],
   "Tbyte"      = [2^40*8,                "bit"],
   "TB"         = [2^40*8,                "bit"],

   //-------------------------------------------------------------------------
   // For electrical amperage, we chose the base unit unit::A (= unit::ampere)
   //-------------------------------------------------------------------------
   "A"          = [1,                      "A"],
   "ampere"     = [1,                      "A"],
   "Ampere"     = [1,                      "A"],
   "mA"         = [10^(-3),                "A"],
   "mAmpere"    = [10^(-3),                "A"],
   "mampere"    = [10^(-3),                "A"],
   "mcA"        = [10^(-6),                "A"],
   "microA"     = [10^(-6),                "A"],
   "microAmpere"= [10^(-6),                "A"],
   "microampere"= [10^(-6),                "A"],
   "nA"         = [10^(-9),                "A"],
   "nAmpere"    = [10^(-9),                "A"],
   "nampere"    = [10^(-9),                "A"],
   "kA"         = [10^3,                   "A"],
   "kAmpere"    = [10^3,                   "A"],
   "kampere"    = [10^3,                   "A"],
   "Bi"         = [10,                     "A"], // Biot
   "Biot"       = [10,                     "A"], // Biot
   "biot"       = [10,                     "A"], // Biot
   //-----------------------------------------------------------------
   // For temperature, we chose the base unit unit::K (= unit::kelvin)
   //-----------------------------------------------------------------
   "K"          = [1,                      "K"],
   "kelvin"     = [1,                      "K"],
   "Kelvin"     = [1,                      "K"],
   "Fahrenheit" = [1,                      "Fahrenheit"],
   "fahrenheit" = [1,                      "Fahrenheit"],
   "Celsius"    = [1,                      "Celsius"],
   "celsius"    = [1,                      "Celsius"],
   "Rankine"    = [1,                      "Rankine"],
   "rankine"    = [1,                      "Rankine"],
   "Reaumur"    = [1,                      "Reaumur"],
   "reaumur"    = [1,                      "Reaumur"],
   //-----------------------------------------------------------------
   // For molecular substance (mole), we chose the base unit unit::mol
   //-----------------------------------------------------------------
   "mol"        = [1,                    "mol"],
   "Mol"        = [1,                    "mol"],
   "kmol"       = [10^3,                 "mol"],
   "kMol"       = [10^3,                 "mol"],
   "mmol"       = [10^(-3),              "mol"],
   "mMol"       = [10^(-3),              "mol"],
   "mcmol"      = [10^(-6),              "mol"],
   "mcMol"      = [10^(-6),              "mol"],
   "micromol"   = [10^(-6),              "mol"],
   "microMol"   = [10^(-6),              "mol"],
   //------------------------------------------------------------------
   // For luminosity, we chose the base unit unit::cd (= unit::candela)
   //------------------------------------------------------------------
   "cd"         = [1,                     "cd"],
   "candela"    = [1,                     "cd"],
   "Candela"    = [1,                     "cd"],
   "HK"         = [0.903,                 "cd"], //Hefner-Kerze (veraltete Einheit)
   "IK"         = [1.019,                 "cd"], //Internationale Kerze (veraltete Einheit)
   //------------------------------------------------------------------
   // Winkel Radiant. 
   //------------------------------------------------------------------
   "degree"    = [PI/180,                "rad"],
   "degrees"   = [PI/180,                "rad"],
   "rad"       = [1,                     "rad"],
   "radian"    = [1,                     "rad"],
   //------------------------------------------------------------------
   // Raumwinkel Steradiant. Wird hier wie eine SI-Einkeit behandelt. 
   //------------------------------------------------------------------
   "sr"         = [1,                     "sr"], // 1 sr = 1 m^2/m^2
   "steradian"  = [1,                     "sr"], // 1 sr = 1 m^2/m^2
   //-------------------------------------------------------------
   // For representation of data, we chose the base unit unit::bit
   //-------------------------------------------------------------
   "bit"        = [1,                    "bit"],
// "Bit"        = [1,                    "bit"],
   //----------------------------------------------------------
   // For European currency, we choose the base unit unit::Cent
   //----------------------------------------------------------
   "cent"       = [1,                   "Cent"],
   "Cent"       = [1,                   "Cent"],
   "EUR"        = [100,                 "Cent"],
   "EURO"       = [100,                 "Cent"],
   "Euro"       = [100,                 "Cent"]
):
// --------------------------------------------------------------
// Automatic generation of the entries unit::xyz of the domain 'unit' 
// from the conversion table. Internally, these slots are domain elements.
// We use the fact that the kernel processes domain elements in a way
// very similar to identifiers 
//  *** provided that the arithmetical functions are not overloaded! ***
// e.g.,  2*u + 3*u --> 5*u, where u = unit::xyz = new(dom, "xyz") is
// a domain element.
// --------------------------------------------------------------
map([op(unit::ConversionTable)], u -> ( 
         (slot(unit, lhs(u))):= new(unit, lhs(u));
      // (slot(unit, lhs(u))):= funcenv( () -> procname(args()) );
      // (slot(unit, lhs(u)))::print := lhs(u):
      // (slot(unit, lhs(u)))::isUnit:= TRUE;
         u:)):
alias(CT = unit::ConversionTable):
// --------------------------------------------------------------
// rewrite the conversion table by changing the index
// from "cm" to unit::cm and changing the base units
// in the values accordingly. Afterwards
// unit::ConversionTable:= table(
//    unit::mm = [10^(-3), unit::m],
//    unit::cm = [10^(-2), unit::m],
//    etc.):
// --------------------------------------------------------------
unit::ConversionTable:= table(
  slot(unit, lhs(u)) = [rhs(u)[1], slot(unit, rhs(u)[2])]
  $ u in [op(unit::ConversionTable)]
):
/* ------------------------------------------------------------
// Composite units
Jede zusammengesetzte Einheit muss sich aus Basiseinheiten,
die in der ersten Konvertierungstabelle definiert wurden,
aufgebaut werden.

Beispiel:
Die zusammengesetzte Einheit "newton" besteht aus den Basiseinheiten
"m", "kg" und "s". Die zusammengesetzte Einheit muss nicht aus
SI-Einheiten, wie in diesem Beispiel, bestehen. Es koennten hier
auch "cm", "g" und "ms" benutzt werden. Die Definition wuerde dann
lauten:
unit::init("newton" = [100*1000/1000^2, unit::cm * unit::g / unit::ms ^ 2])

Zusammengesetzte Einheiten, die sich wiederum aus zusammengesetzten 
Einheiten (und Basiseinheiten) aufbauen, etwa "joule", koennen erst
dann definiert werden, wenn zuvor die benoetigten zusammengesetzten
Einheiten definiert wurden. 

Beispiel:
Die zusammengesetzte Einheit "joule" benutzt "newton". "joule" kann
erst nach der Definition von "newton" definiert werden.
------------------------------------------------------------ */
// ---------------------------------------------------------------------------
// Units defined by the first conversion table must be generated before
// generating the units of the 2nd conversion table!
// ---------------------------------------------------------------------------

unit::init := proc(u)
   local lu, ru;
begin
  lu := lhs(u); ru := rhs(u);
  if hastype(ru, DOM_FAIL) then
     //str := "\nDefinition of unit " . lu . " uses an unknown unit.";
     //str := str . "\nDefinition of unit " . lu . " ignored!";
     //warning(str);
     return( null() )
  end_if;
  if slot(unit, lu) <> FAIL then
     // unit lu already exits
     //warning(lu." already exists. Ignored!");
     return( null() )
  end_if;
  slot(unit, lu):= new(unit, lu);
  unit::nonbaseunits := append(unit::nonbaseunits, slot(unit, lu));
  slot(unit, lu) = ru
end:

unit::nonbaseunits := []:
 
unit::ConversionTable2:= table(
   /**********************************
   Groessen und Einheiten der Mechanik
   ***********************************/
   // Basiseinheiten (SI-Einheiten) der Mechanik sind
   // die Laenge (Meter), die Masse (Kilogramm) und 
   // die Zeit (Sekunde)

   // Masse (Avoirdupois)
   unit::init("gr"              = [1/7000, unit::lb ]), // grain
   unit::init("dr"              = [1/16,   unit::oz ]), // dram
   unit::init("stone"           = [14,     unit::lb ]),
   unit::init("quarter"         = [28,     unit::lb ]),
   unit::init("cental"          = [100,    unit::lb ]),
   unit::init("sh_cwt"          = [100,    unit::lb ]), // short hundredweight (US)
   unit::init("cwt"             = [100,    unit::lb ]), // short hundredweight (US)
   unit::init("cwt_UK"          = [112,    unit::lb ]), // centweight, hundredweight
   unit::init("long_cwt"        = [112,    unit::lb ]), // long  hundredweight (US)
   unit::init("gross_cwt"       = [112,    unit::lb ]), // long  hundredweight (US)
   unit::init("ton_UK"          = [2240,   unit::lb ]),
   unit::init("ton"             = [2000,   unit::lb ]), // short ton           (US)
   unit::init("tn"              = [2000,   unit::lb ]), // short ton           (US)
   unit::init("short_ton"       = [2000,   unit::lb ]), // short ton           (US)
   unit::init("long_ton"        = [2240,   unit::lb ]), // long  ton           (US)
   unit::init("gross_ton"       = [2240,   unit::lb ]), // long  ton           (US)

   // Kraft: Newton
   unit::init("Newton"         = [1,    unit::m * unit::kg / unit::s^2]),
   unit::init("newton"         = [1,    unit::m * unit::kg / unit::s^2]),
   unit::init("N"              = [1,    unit::m * unit::kg / unit::s^2]),
   unit::init("pdl"            = [1,  unit::lb * unit::ft / unit::s^2 ]), // poundal
   unit::init("lbf"            = [980665/100000, unit::lb * unit::m  / unit::s^2 ]), // pound-force
   unit::init("ozf"            = [1000         , unit::lbf                       ]), // kip ounce-force
   unit::init("tonf"           = [8896443230521/1000000000, unit::N              ]), // ton-force
   unit::init("kp"             = [196133/20000, unit::N               ]),
   unit::init("kilopond"       = [196133/20000, unit::N               ]),
   unit::init("Kilopond"       = [196133/20000, unit::N               ]),
   unit::init("p"              = [196133/20000000, unit::N            ]),
   unit::init("pond"           = [196133/20000000, unit::N            ]),
   unit::init("Pond"           = [196133/20000000, unit::N            ]),
   unit::init("dyn"            = [10^(-5)  , unit::N                  ]),

   //Drehmoment: Newtonmeter
   unit::init("Newtonmeter"    = [1,    unit::N * unit::m             ]),
   unit::init("newtonmeter"    = [1,    unit::N * unit::m             ]),
   unit::init("Nm"             = [1,    unit::N * unit::m             ]),

   //Drehimpuls: Newtonmetersekunde
   unit::init("Newtonmetersec" = [1,    unit::N * unit::m * unit::s   ]),
   unit::init("newtonmetersec" = [1,    unit::N * unit::m * unit::s   ]),
   unit::init("Nms"            = [1,    unit::N * unit::m * unit::s   ]),

   // Arbeit, Energie: Joule
   // 1 J = 1 N * m = 1 W * s = 1/(3,6*10^6) * kW * h
   unit::init("joule"          = [1,    unit::N * unit::m            ]), 
   unit::init("Joule"          = [1,    unit::N * unit::m            ]),
   unit::init("J"              = [1,    unit::N * unit::m            ]), 
   unit::init("mJ"             = [10^(-3), unit::N * unit::m         ]), 
   unit::init("mJoule"         = [10^(-3), unit::N * unit::m         ]), 
   unit::init("mjoule"         = [10^(-3), unit::N * unit::m         ]), 
   unit::init("kJ"             = [10^3, unit::N * unit::m            ]), 
   unit::init("kJoule"         = [10^3, unit::N * unit::m            ]), 
   unit::init("kjoule"         = [10^3, unit::N * unit::m            ]), 
   unit::init("MJ"             = [10^6, unit::N * unit::m            ]), 
   unit::init("MJoule"         = [10^6, unit::N * unit::m            ]), 
   unit::init("Mjoule"         = [10^6, unit::N * unit::m            ]), 
   unit::init("Ws"             = [1,    unit::N * unit::m            ]),
   unit::init("microWs"        = [10^(-6), unit::N * unit::m         ]),
   unit::init("mcWs"           = [10^(-6), unit::N * unit::m         ]),
   unit::init("mWs"            = [10^(-3), unit::N * unit::m         ]),
   unit::init("Wh"             = [3600, unit::J                      ]),
   unit::init("kWh"            = [3600*10^3, unit::J                 ]),
   unit::init("MWh"            = [3600*10^6, unit::J                 ]),
   unit::init("GWh"            = [3600*10^9, unit::J                 ]),
   unit::init("kpm"            = [9.80665, unit::J                   ]), //kilopondmeter
   unit::init("PSh"            = [2.6477955*10^6, unit::J            ]), //PS-Stunde
   unit::init("kWh"            = [3600*10^3, unit::J                 ]), //kilowatthour
   unit::init("Calory"         = [4.1868,  unit::J                   ]),
   unit::init("calory"         = [4.1868,  unit::J                   ]),
   unit::init("cal"            = [4.1868,  unit::J                   ]),
   unit::init("kcal"           = [10^3,    unit::cal                 ]),
   unit::init("eV"             = [1.60217733 * 10^(-19),  unit::J    ]), // Energie: Elektronenvolt
   unit::init("Btu"            = [23722880951/22500000    , unit::J  ]), // British thermal unit
   unit::init("therm"          = [23722880951/225         , unit::J  ]), // therm
   unit::init("erg"            = [10^(-7), unit::J                   ]), // (veraltet)

   // Leistung, Energiestrom: Watt
   unit::init("Watt"           = [1,    unit::J / unit::s             ]),
   unit::init("watt"           = [1,    unit::J / unit::s             ]),
   unit::init("W"              = [1,    unit::J / unit::s             ]), 
   unit::init("mcW"            = [10^(-6), unit::J / unit::s          ]),
   unit::init("mcWatt"         = [10^(-6), unit::J / unit::s          ]),
   unit::init("mcwatt"         = [10^(-6), unit::J / unit::s          ]),
   unit::init("microW"         = [10^(-6), unit::J / unit::s          ]),
   unit::init("microWatt"      = [10^(-6), unit::J / unit::s          ]),
   unit::init("microwatt"      = [10^(-6), unit::J / unit::s          ]),
   unit::init("mW"             = [10^(-3), unit::J / unit::s          ]),
   unit::init("mWatt"          = [10^(-3), unit::J / unit::s          ]),
   unit::init("mwatt"          = [10^(-3), unit::J / unit::s          ]),
   unit::init("kW"             = [1000,    unit::J / unit::s          ]),
   unit::init("kWatt"          = [1000,    unit::J / unit::s          ]),
   unit::init("kwatt"          = [1000,    unit::J / unit::s          ]),
   unit::init("MW"             = [10^6,    unit::J / unit::s          ]),
   unit::init("MWatt"          = [10^6,    unit::J / unit::s          ]),
   unit::init("Mwatt"          = [10^6,    unit::J / unit::s          ]),
   unit::init("GW"             = [10^9,    unit::J / unit::s          ]),
   unit::init("GWatt"          = [10^9,    unit::J / unit::s          ]),
   unit::init("Gwatt"          = [10^9,    unit::J / unit::s          ]),
   unit::init("PS"             = [588399/800,  unit::W                ]), // Pferdestaerke
   unit::init("hp"             = [550, unit::ft * unit::lbf / unit::s ]), // horsepower (UK)  
   unit::init("bhp"            = [550, unit::ft * unit::lbf / unit::s ]), // horsepower (UK)  

   // Frequenz, Drehzahl: Hertz
   unit::init("Hertz"          = [1,    unit::s ^ (-1)                ]),
   unit::init("hertz"          = [1,    unit::s ^ (-1)                ]),
   unit::init("Hz"             = [1,    unit::s ^ (-1)                ]),
   unit::init("kHz"            = [10^3, unit::s ^ (-1)                ]),
   unit::init("kHertz"         = [10^3, unit::s ^ (-1)                ]),
   unit::init("khertz"         = [10^3, unit::s ^ (-1)                ]),
   unit::init("MHz"            = [10^6, unit::s ^ (-1)                ]),
   unit::init("MHertz"         = [10^6, unit::s ^ (-1)                ]),
   unit::init("Mhertz"         = [10^6, unit::s ^ (-1)                ]),
   unit::init("GHz"            = [10^9, unit::s ^ (-1)                ]),
   unit::init("GHertz"         = [10^9, unit::s ^ (-1)                ]),
   unit::init("Ghertz"         = [10^9, unit::s ^ (-1)                ]),

   // Druck: Pascal
   unit::init("Pascal"         = [1,    unit::N / unit::m ^ 2         ]),
   unit::init("pascal"         = [1,    unit::N / unit::m ^ 2         ]),
   unit::init("Pa"             = [1,    unit::N / unit::m ^ 2         ]),
   unit::init("hPa"            = [100,  unit::N / unit::m ^ 2         ]),
   unit::init("hPascal"        = [100,  unit::N / unit::m ^ 2         ]),
   unit::init("hpascal"        = [100,  unit::N / unit::m ^ 2         ]),
   unit::init("Bar"            = [10^5,    unit::Pa                   ]),
   unit::init("bar"            = [10^5,    unit::Pa                   ]),
   unit::init("mBar"           = [10^(-3), unit::bar                  ]),
   unit::init("mbar"           = [10^(-3), unit::bar                  ]),
   unit::init("mcBar"          = [10^(-6), unit::bar                  ]),
   unit::init("mcbar"          = [10^(-6), unit::bar                  ]),
   unit::init("microBar"       = [10^(-6), unit::bar                  ]),
   unit::init("microbar"       = [10^(-6), unit::bar                  ]),
   unit::init("kBar"           = [10^3 ,   unit::bar                  ]),
   unit::init("kbar"           = [10^3 ,   unit::bar                  ]),
   unit::init("psi"            = [8896443230521/1290320000,  unit::Pa ]), // pounds per square inch
   // Druck: at (technische Atmosphaere: 1 at = 0.980665 * 10^5 Pa)
   unit::init("at"             = [196133/2 , unit::Pa                 ]),
   // Die Anhaengezeichen a, u, an at werden benutzt, um einen 
   // Absolut-, Unterdruck- bzw. Ueberdruck zu kennzeichnen.
   unit::init("ata"            = [196133/2    , unit::Pa              ]),
   unit::init("atu"            = [196133/2    , unit::Pa              ]),
   // Druck: atm (physikalische Atmosphaere)
   unit::init("atm"            = [101325      , unit::Pa              ]),
   // Druck: Torr
   unit::init("Torr"           = [20265/152   , unit::Pa              ]),
   // Druck: mmWS (Millimeter Wassersaeule)
   unit::init("mmWS"           = [196133/20000, unit::Pa              ]),
   unit::init("mmH2O"          = [196133/20000, unit::Pa              ]),
   // Druck: mWS (Meter Wassersaeule)
   unit::init("mWS"            = [196133/20    , unit::Pa             ]),
   unit::init("mH2O"           = [196133/20    , unit::Pa             ]),
   // Druck: mmHg (Millimeter Quecksilbersaeule)
   unit::init("mmHg"           = [20265/152    , unit::Pa             ]),
   // Druck: mHg (Meter Quecksilbersaeule)
   unit::init("mHg"            = [2533125/19   , unit::Pa             ]),
   // Druck, mechanische Spannung
   unit::init("inH2O"      = [24908891/100000         , unit::Pa ]), // inch of water
   unit::init("ftH2O"      = [74726673/25000          , unit::Pa ]), // foot of water
   unit::init("inHg"       = [514731/152              , unit::Pa ]), // inch of mercury

   // Flaeche
   unit::init("hectare"    = [10^4,     unit::m        ^ 2      ]),
   unit::init("Hektar"     = [10^4,     unit::m        ^ 2      ]),
   unit::init("ha"         = [10^4,     unit::m        ^ 2      ]),
   unit::init("a"          = [10^2,     unit::m        ^ 2      ]),
   unit::init("are"        = [10^2,     unit::m        ^ 2      ]),
   unit::init("Ar"         = [10^2,     unit::m        ^ 2      ]),
   unit::init("barn"       = [10^(-28), unit::m        ^ 2      ]),
   unit::init("b"          = [10^(-28), unit::m        ^ 2      ]),
   unit::init("ro"         = [1210,     unit::yd ^ 2            ]), // rood
   unit::init("rood"       = [1210,     unit::yd ^ 2            ]), // rood
   unit::init("ac"         = [4,        unit::rood              ]), // acre
   unit::init("acre"       = [4,        unit::rood              ]), // acre
   unit::init("township"   = [36,       unit::mile ^ 2          ]), // township
   unit::init("circ_mil"   = [PI/4,     unit::mil ^ 2           ]), // circular mil
   unit::init("circ_inch"  = [PI/4,     unit::inch ^ 2          ]), // circular inch

   // Volumen: Liter
   unit::init("Liter"          = [10^(-3), unit::m ^ 3      ]),
   unit::init("liter"          = [10^(-3), unit::m ^ 3      ]),
   unit::init("Litre"          = [10^(-3), unit::m ^ 3      ]),
   unit::init("litre"          = [10^(-3), unit::m ^ 3      ]),
   unit::init("l"              = [10^(-3), unit::m ^ 3      ]),
   unit::init("ml"             = [10^(-3), unit::l          ]),
   unit::init("hl"             = [10^( 2), unit::l          ]),
   // Volumen: UK-Einheiten
   unit::init("gal_UK"     = [568258750/2048383, unit::inch ^ 3 ]), // gallon       (UK)
   unit::init("pk_UK"      = [2,         unit::gal_UK           ]), // peck         (UK)
   unit::init("bu_UK"      = [4,         unit::pk_UK            ]), // bushel       (UK)
   unit::init("chaldron"   = [288,       unit::gal_UK           ]), // chaldron     (UK)
   unit::init("pottle"     = [1/2,       unit::gal_UK           ]), // pottle       (UK)
   unit::init("qt_UK"      = [1/4,       unit::gal_UK           ]), // quart        (UK)
   unit::init("pint_UK"    = [1/2,       unit::qt_UK            ]), // pint         (UK)
   unit::init("gill_UK"    = [1/4,       unit::pint_UK          ]), // gill         (UK)
   unit::init("floz_UK"    = [1/5,       unit::gill_UK          ]), // fluid ounce  (UK)
   unit::init("fldr_UK"    = [1/8,       unit::floz_UK          ]), // fluid drachm (UK)
   unit::init("minim_UK"   = [1/60,      unit::fldr_UK          ]), // minim        (UK)
   // Volumen: Fluessigkeitsmasse in US-Einheiten
   unit::init("gal"        = [231,       unit::inch ^ 3         ]), // gallon       (US)
   unit::init("barrel"     = [9702,      unit::inch ^ 3         ]), // barrel       (US)
   unit::init("liq_qt"     = [1/4,       unit::gal              ]), // liquid quart (US)
   unit::init("liq_pt"     = [1/2,       unit::liq_qt           ]), // liquid pint  (US)
   unit::init("gill"       = [1/4,       unit::liq_pt           ]), // gill         (US)
   unit::init("floz"       = [1/4,       unit::gill             ]), // fluid ounce  (US)
   unit::init("fldr"       = [1/8,       unit::floz             ]), // fluid drachm (US)
   unit::init("minim"      = [1/60,      unit::fldr             ]), // minim        (US)
   // Volumen: Trockenmasse in US-Einheiten
   unit::init("dry_bu"     = [1369/2*PI, unit::inch ^ 3         ]), // dry bushel   (US)
   unit::init("dry_pk"     = [1/4,       unit::dry_bu           ]), // dry peck     (US)
   unit::init("bbl"        = [7056,      unit::inch ^ 3         ]), // dry barrel   (US)
   unit::init("dry_gal"    = [1/2,       unit::dry_pk           ]), // dry gallon   (US)
   unit::init("dry_qt"     = [1/8,       unit::dry_pk           ]), // dry quart    (US)
   unit::init("dry_pt"     = [1/2,       unit::dry_qt           ]), // dry pint     (US)  

   // Dynamische Viskositaet: Poise
   unit::init("P"              = [10^(-1), unit::Pa * unit::s        ]),
   unit::init("Poise"          = [10^(-1), unit::Pa * unit::s        ]),
   // Kinematische Viskositaet: Stokes                
   unit::init("St"             = [10^(-4), unit::m ^ 2 / unit::s     ]),
   unit::init("Stokes"         = [10^(-4), unit::m ^ 2 / unit::s     ]),
   // Beschleuniging: Gal
   unit::init("Gal"            = [10^(-2), unit::m / unit::s ^ 2     ]),
   // Beschleuniging: Freier Fall / Free Fall
   unit::init("gn"             = [196133/20000, unit::m / unit::s ^ 2      ]),

   /*******************************
   Groessen und Einheiten der Optik
   *******************************/
   // Basiseinheit (SI-Einheit) der Optik ist Candela (Lichtstaerke)
   // Brechwert: Dioptrie
   unit::init("dpt"        = [1,       unit::m ^ (-1)          ]), 
   unit::init("diopter"    = [1,       unit::m ^ (-1)          ]), 
   unit::init("dioptre"    = [1,       unit::m ^ (-1)          ]), 
   // Lichstaerke: Hefner-Kerze (veraltete Einheit)
   unit::init("HK"         = [0.903,   unit::cd                ]),
   // Lichstaerke: Internationale Kerze (veraltete Einheit)
   unit::init("IK"         = [1.019,   unit::cd                ]),
   // Leuchtdichte: Stilb
   unit::init("sb"         = [10^4,    unit::cd / unit::m ^ 2  ]),
   unit::init("stilb"      = [10^4,    unit::cd / unit::m ^ 2  ]),
   // Leuchtdichte: Apostilb        
   unit::init("asb"        = [(1/PI),  unit::cd / unit::m ^ 2  ]),
   unit::init("apostilb"   = [(1/PI),  unit::cd / unit::m ^ 2  ]),
   // Lichtstrom: Lumen
   unit::init("lm"         = [1,       unit::cd * unit::sr     ]), // 1 lm = 1 cd * sr [1] 
   unit::init("lumen"      = [1,       unit::cd * unit::sr     ]), // 1 lm = 1 cd * sr [1] 
   // Beleuchtungsstaerke: Lux
   unit::init("lx"         = [1,       unit::lm / unit::m ^ 2  ]), // 1 lx = cd / m^2 * sr [1]
   unit::init("lux"        = [1,       unit::lm / unit::m ^ 2  ]), // 1 lx = cd / m^2 * sr [1]
   // Beleuchtungsstrke, Lichtausstrahlung: Phot
   unit::init("ph"         = [10^4,    unit::lx                ]),
   unit::init("phot"       = [10^4,    unit::lx                ]),
   // Dunkelbeleuchtungsstaerke: Nox                        
   unit::init("nx"         = [10^(-3), unit::lx                ]),
   // Bestrahlung: Langley
   unit::init("langley"    = [1,       unit::cal / unit::cm ^ 2]), // 4.1868*10^4 J/m^2

   // [1] Raumwinkel
   // Steradiant: 1 sr = 1 m^2/m^2
   // Soll Lumen durch 1 lm = 1 cd * sr definiert werden, so ist sr in 
   // units::SIunits zu ergaenzen. sr ist allerdings nicht durch 
   // m^2/m^2 zu definieren. sr ist dann als "SI-Einheit" aufzufassen.

   /****************************************************************
   Groessen und Einheiten der Elektrizitaetslehre und des Magnetismus
   ****************************************************************/
   // Basiseinheit (SI-Einheit) der Elektrizitaetslehre ist Ampere 
   // (elektrische Stromstaerke)

   // Elektrische Spannung, Elektrisches Potenzial: Volt
   unit::init("Volt"       = [1,         (unit::kg * unit::m ^ 2) / (unit::s ^ 3 * unit::A)]),
   unit::init("volt"       = [1,         (unit::kg * unit::m ^ 2) / (unit::s ^ 3 * unit::A)]),
   unit::init("V"          = [1,         (unit::kg * unit::m ^ 2) / (unit::s ^ 3 * unit::A)]),
   unit::init("nanoV"      = [10^(-9),   (unit::kg * unit::m ^ 2) / (unit::s ^ 3 * unit::A)]),
   unit::init("nV"         = [10^(-9),   (unit::kg * unit::m ^ 2) / (unit::s ^ 3 * unit::A)]),
   unit::init("nVolt"      = [10^(-9),   (unit::kg * unit::m ^ 2) / (unit::s ^ 3 * unit::A)]),
   unit::init("nvolt"      = [10^(-9),   (unit::kg * unit::m ^ 2) / (unit::s ^ 3 * unit::A)]),
   unit::init("mcV"        = [10^(-6),   (unit::kg * unit::m ^ 2) / (unit::s ^ 3 * unit::A)]),
   unit::init("microV"     = [10^(-6),   (unit::kg * unit::m ^ 2) / (unit::s ^ 3 * unit::A)]),
   unit::init("mcVolt"     = [10^(-6),   (unit::kg * unit::m ^ 2) / (unit::s ^ 3 * unit::A)]),
   unit::init("mcvolt"     = [10^(-6),   (unit::kg * unit::m ^ 2) / (unit::s ^ 3 * unit::A)]),
   unit::init("mV"         = [10^(-3),   (unit::kg * unit::m ^ 2) / (unit::s ^ 3 * unit::A)]),
   unit::init("mVolt"      = [10^(-3),   (unit::kg * unit::m ^ 2) / (unit::s ^ 3 * unit::A)]),
   unit::init("mvolt"      = [10^(-3),   (unit::kg * unit::m ^ 2) / (unit::s ^ 3 * unit::A)]),
   unit::init("kV"         = [10^( 3),   (unit::kg * unit::m ^ 2) / (unit::s ^ 3 * unit::A)]),
   unit::init("kVolt"      = [10^( 3),   (unit::kg * unit::m ^ 2) / (unit::s ^ 3 * unit::A)]),
   unit::init("kvolt"      = [10^( 3),   (unit::kg * unit::m ^ 2) / (unit::s ^ 3 * unit::A)]),
   unit::init("MV"         = [10^( 6),   (unit::kg * unit::m ^ 2) / (unit::s ^ 3 * unit::A)]),
   unit::init("MVolt"      = [10^( 6),   (unit::kg * unit::m ^ 2) / (unit::s ^ 3 * unit::A)]),
   unit::init("Mvolt"      = [10^( 6),   (unit::kg * unit::m ^ 2) / (unit::s ^ 3 * unit::A)]),
   unit::init("GV"         = [10^( 9),   (unit::kg * unit::m ^ 2) / (unit::s ^ 3 * unit::A)]),
   unit::init("GVolt"      = [10^( 9),   (unit::kg * unit::m ^ 2) / (unit::s ^ 3 * unit::A)]),
   unit::init("Gvolt"      = [10^( 9),   (unit::kg * unit::m ^ 2) / (unit::s ^ 3 * unit::A)]),
   // Elektrischer Widerstand: Ohm
   unit::init("Ohm"        = [1,         unit::V / unit::A                ]),
   unit::init("ohm"        = [1,         unit::V / unit::A                ]),
   unit::init("nOhm"       = [10^(-9),   unit::V / unit::A                ]),
   unit::init("nohm"       = [10^(-9),   unit::V / unit::A                ]),
   unit::init("mcOhm"      = [10^(-6),   unit::V / unit::A                ]),
   unit::init("mcohm"      = [10^(-6),   unit::V / unit::A                ]),
   unit::init("microOhm"   = [10^(-6),   unit::V / unit::A                ]),
   unit::init("microohm"   = [10^(-6),   unit::V / unit::A                ]),
   unit::init("milliOhm"   = [10^(-3),   unit::V / unit::A                ]),
   unit::init("milliohm"   = [10^(-3),   unit::V / unit::A                ]),
   unit::init("mOhm"       = [10^(-3),   unit::V / unit::A                ]),
   unit::init("mohm"       = [10^(-3),   unit::V / unit::A                ]),
   unit::init("kOhm"       = [10^( 3),   unit::V / unit::A                ]),
   unit::init("kohm"       = [10^( 3),   unit::V / unit::A                ]),
   unit::init("MOhm"       = [10^( 6),   unit::V / unit::A                ]),
   unit::init("Mohm"       = [10^( 6),   unit::V / unit::A                ]),
   unit::init("GOhm"       = [10^( 9),   unit::V / unit::A                ]),
   unit::init("Gohm"       = [10^( 9),   unit::V / unit::A                ]),
   // Elektrische Ladung: Coulomb
   unit::init("Coulomb"    = [1,         unit::A * unit::s                ]),
   unit::init("coulomb"    = [1,         unit::A * unit::s                ]),
   unit::init("C"          = [1,         unit::A * unit::s                ]),
   // Elektrische Kapazitaet: Farad
   unit::init("Farad"      = [1,         unit::C / unit::V                ]),
   unit::init("farad"      = [1,         unit::C / unit::V                ]),
   unit::init("F"          = [1,         unit::C / unit::V                ]),
   unit::init("pF"         = [10^(-12),  unit::C / unit::V                ]),
   unit::init("pFarad"     = [10^(-12),  unit::C / unit::V                ]),
   unit::init("pfarad"     = [10^(-12),  unit::C / unit::V                ]),
   unit::init("nF"         = [10^(- 9),  unit::C / unit::V                ]),
   unit::init("nFarad"     = [10^(- 9),  unit::C / unit::V                ]),
   unit::init("nfarad"     = [10^(- 9),  unit::C / unit::V                ]),
   unit::init("mcF"        = [10^(- 6),  unit::C / unit::V                ]),
   unit::init("mcFarad"    = [10^(- 6),  unit::C / unit::V                ]),
   unit::init("mcfarad"    = [10^(- 6),  unit::C / unit::V                ]),
   unit::init("microF"     = [10^(- 6),  unit::C / unit::V                ]),
   unit::init("microFarad" = [10^(- 6),  unit::C / unit::V                ]),
   unit::init("microfarad" = [10^(- 6),  unit::C / unit::V                ]),
   unit::init("mFarad"     = [10^(- 3),  unit::C / unit::V                ]),
   unit::init("mfarad"     = [10^(- 3),  unit::C / unit::V                ]),
   unit::init("mF"         = [10^(- 3),  unit::C / unit::V                ]),
   unit::init("kFarad"     = [10^(  3),  unit::C / unit::V                ]),
   unit::init("kfarad"     = [10^(  3),  unit::C / unit::V                ]),
   unit::init("kF"         = [10^(  3),  unit::C / unit::V                ]),
   // Elektrischer Leitwert: Siemens
   unit::init("S"          = [1,         unit::A / unit::V                ]),
   unit::init("Siemens"    = [1,         unit::A / unit::V                ]),
   unit::init("siemens"    = [1,         unit::A / unit::V                ]),
   // Induktivitaet: Henry
   unit::init("Henry"      = [1,         unit::V * unit::s / unit::A      ]),
   unit::init("henry"      = [1,         unit::V * unit::s / unit::A      ]),
   unit::init("H"          = [1,         unit::V * unit::s / unit::A      ]),
   // Magnetische Flussdichte: Tesla (veraltet: Gauss)
   unit::init("Tesla"      = [1,         unit::V * unit::s / unit::m ^ 2  ]),
   unit::init("tesla"      = [1,         unit::V * unit::s / unit::m ^ 2  ]),
   unit::init("T"          = [1,         unit::V * unit::s / unit::m ^ 2  ]),
   unit::init("G"          = [10^(-4),   unit::T                          ]),
   unit::init("Gauss"      = [10^(-4),   unit::T                          ]),
   unit::init("gauss"      = [10^(-4),   unit::T                          ]),
   // Magnetischer Fluss: Weber (veraltet: Maxwell)
   unit::init("Wb"         = [1,         unit::V * unit::s                ]),
   unit::init("Weber"      = [1,         unit::V * unit::s                ]),
   unit::init("weber"      = [1,         unit::V * unit::s                ]),
   unit::init("M"          = [10^(-8),   unit::V * unit::s                ]),
   unit::init("Maxwell"    = [10^(-8),   unit::V * unit::s                ]),
   unit::init("maxwell"    = [10^(-8),   unit::V * unit::s                ]),
   // Magnetische Feldstrke: Oersted
   unit::init("Oe"         = [10^3/(4*PI), unit:: A / unit::m             ]),
   unit::init("Oersted"    = [10^3/(4*PI), unit:: A / unit::m             ]),
   unit::init("oersted"    = [10^3/(4*PI), unit:: A / unit::m             ]),
   // Magnetische Spannung: Gilbert        
   unit::init("Gb"         = [10/(4*PI), unit::A                          ]),                
   unit::init("Gilbert"    = [10/(4*PI), unit::A                          ]),                
   unit::init("gilbert"    = [10/(4*PI), unit::A                          ]),                

   /**********************************************************
   Groessen und Einheiten der Kernphysik und im Strahlenschutz
   **********************************************************/

   // Aktivitaet: Becquerel, Curie
   unit::init("Becquerel"      = [1,                unit::s ^ (-1)     ]),
   unit::init("becquerel"      = [1,                unit::s ^ (-1)     ]),
   unit::init("Bq"             = [1,                unit::s ^ (-1)     ]),
   unit::init("Ci"             = [37000000000,      unit::s ^ (-1)     ]),
   unit::init("Curie"          = [37000000000,      unit::s ^ (-1)     ]),
   // Aequivalentdosis: Sievert
   unit::init("Sievert"        = [1,                unit::J / unit::kg ]),
   unit::init("sievert"        = [1,                unit::J / unit::kg ]),
   unit::init("Sv"             = [1,                unit::J / unit::kg ]),
   unit::init("rem"            = [10^(-2),          unit::Sv           ]),
   unit::init("Rem"            = [10^(-2),          unit::Sv           ]),
   // Energiedosis: Gray, Rad
   unit::init("Gray"           = [1,                unit::J / unit::kg ]),
   unit::init("gray"           = [1,                unit::J / unit::kg ]),
   unit::init("Gy"             = [1,                unit::J / unit::kg ]),
   unit::init("rd"             = [10^(-2),          unit::J / unit::kg ]),
   // Ionendosis, Exposition: Roentgen
   unit::init("R"              = [129/500000,       unit::C / unit::kg ]),
   unit::init("Roentgen"       = [129/500000,       unit::C / unit::kg ]),
   
   // Uebertragungsgeschwindigkeit von Daten: Bits pro Sekunde
   unit::init("bps"                 = [1,                unit::bit / unit::s ]),
   // Signalrate, mit der Daten uebertragen werden: Baud
   unit::init("Bd"                  = [1,         unit::s^(-1)   ]),
   unit::init("baud"                = [1,         unit::s^(-1)   ]),
   unit::init("Baud"                = [1,         unit::s^(-1)   ]),

   // EURO-WECHSELKURSE ab 01.01.1999 
   // Deutschland: Deutsche Mark
   unit::init("DEM"                 = [1/1.95583,  unit::EUR    ]),
   unit::init("DM"                  = [1/1.95583,  unit::EUR    ]),
   // Belgien: Belgische Francs           
   unit::init("BEF"                 = [1/40.3399,  unit::EUR    ]),
   // Finnland: Finnische Mark
   unit::init("FIM"                 = [1/5.94573,  unit::EUR    ]),
   // Frankreich: Franzoesische Francs
   unit::init("FRF"                 = [1/6.55957,  unit::EUR    ]),
   // Irland: Irische Pfund
   unit::init("IEP"                 = [1/0.787564, unit::EUR    ]),
   // Italien: Italienische Lire
   unit::init("ITL"                 = [1/1936.27,  unit::EUR    ]),
   // Luxemburg: Luxemburgische Francs
   unit::init("LUF"                 = [1/40.3399,  unit::EUR    ]),
   // Niederlande: Hollaendische Gulden
   unit::init("NLG"                 = [1/2.20371,  unit::EUR    ]),
   // Oesterreich: Oesterreichische Schillinge
   unit::init("ATS"                 = [1/13.7603,  unit::EUR    ]),
   // Portugal: Portugisische Escudos      
   unit::init("PTE"                 = [1/200.482,  unit::EUR    ]),
   // Spanien: Spanische Peseten          
   unit::init("ESP"                 = [1/166.386,  unit::EUR    ]),

   /*****************
   Sonstige Einheiten
   *****************/
   // Laengenbezogene Masse von textilen Fasern und Garnen: Tex
   unit::init("tex"        = [1,         unit::g / unit::km     ]), // Tex
   unit::init("den"        = [1/9,       unit::tex              ]), // Denier
   unit::init("denier"     = [1/9,       unit::tex              ]), // Denier

   // Geschwindigkeit
   unit::init("knot_UK"    = [1,         unit::nmile / unit::h  ]), // knot         (UK)
   unit::init("knot"       = [1,         unit::INM / unit::h    ]), // knot (International)
   unit::init("mach"       = [331.46,    unit::m / unit::s      ]), // Mach

/* clashes with previous setting
   // Finanzmathematik: 1 Monat = 30 Tage, 1 Jahr = 360 Tage
   unit::init("Monat"      = [30,        unit::d                ]),
   unit::init("Monate"     = [30,        unit::d                ]),  
   unit::init("Jahr"       = [360,       unit::d                ]),
   unit::init("Jahre"      = [360,       unit::d                ]) 
*/
  null()
):

//------------------------------------------------------
// an internal table used by info(u) with u of type unit.
// It will be filled with alternative names (AlterEgos)
// for the unit u when info(u) is called
//------------------------------------------------------
unit::infoTable:= table(
   //-------------
   // length units
   //-------------
   unit::nm    = ["nanometer: a length unit"],
   unit::My    = ["micrometer, micron: a length unit"],
   unit::mm    = ["millimeter: a length unit"],
   unit::cm    = ["centimeter: a length unit"],
   unit::dm    = ["decimeter: a length unit"],
   unit::m     = ["meter: a length unit"],
   unit::km    = ["kilometer: a length unit"],
   unit::inch  = ["inch: a length unit"],
   unit::yd    = ["yard: a length unit"],
   unit::pt    = ["point: a length unit"],
   unit::AU    = ["astronmical unit: a length unit (used in astronomy)"],
   unit::pc    = ["parsec: a length unit (used in astronomy)"],
   unit::ly    = ["lightyear: a length unit (used in astronomy)"],
   unit::Ao    = ["angstrom: a length unit (used in atomic physics)"],
   unit::xu    = ["x unit: a length unit (used in physics)"],
   unit::f     = ["fermi: a length unit (used in physics)"],
   unit::mile  = ["mile: a length unit"],
   unit::ft    = ["foot: a length unit"],
   unit::ft_US = ["foot (US): a length unit (used by US Coast and Geodesic Survey)"],
   unit::fm    = ["fathom: a nautical length unit"],
   unit::nmile = ["nautical mile (UK): a length unit"],
   unit::inm   = ["nautical mile (international): a length unit"],
   unit::gg    = ["gauge: a length unit"],
   unit::line  = ["line: a length unit"],
   unit::span  = ["span: a length unit"],
   unit::hand  = ["hand: a length unit (to measure the height of horses)"],
   unit::rod   = ["rod: a length unit (used in geodesy)"],
   unit::ch    = ["chain: a length unit (used in geodesy)"],
   unit::li    = ["link: a length unit (used in geodesy)"],
   unit::furlong = ["furlong: a length unit (used in geodesy)"],
   //-------------
   // mass units
   //-------------
   unit::microgram = ["micogram: a mass unit"],
   unit::mg     = ["milligram: a mass unit"],
   unit::g      = ["gram: a mass unit"],
   unit::kg     = ["kilogram: a mass unit"],
   unit::t      = ["ton: a mass unit"],
   unit::kt     = ["kiloton: a mass unit"],
   unit::Mt     = ["megaton: a mass unit"],
   unit::ounce  = ["ounce: a mass unit"],
   unit::Ztr    = ["Zentner: a (german) mass unit"],
   unit::dz     = ["Doppelzentner: a (german) mass unit"],
   unit::Pfund  = ["Pfund: a (german) mass unit"],
   unit::pound  = ["pound: a mass unit"],
   unit::ct     = ["carat: a mass unit (for jewels)"],
   unit::gr     = ["grain: a mass unit"],
   unit::dr     = ["dram: a mass unit"],
   unit::stone  = ["stone: a mass unit"],
   unit::quarter= ["quarter: a mass unit"],
   unit::cwt    = ["short hundredweight: a mass unit"],
   unit::cwt_UK = ["hundredweight: a mass unit (UK)"],
   unit::long_cwt= ["long hundredweight: a mass unit (US)"],
   unit::ton    = ["(short) ton: a mass unit (US)"],
   unit::ton_UK = ["(short) ton: a mass unit (UK)"],
   unit::long_ton = ["(long) ton: a mass unit (US)"],
   unit::slug   = ["slug: a mass unit"],
   //-------------
   // time units
   //-------------
   unit::ns      = ["nanosecond: a time unit"],
   unit::microsec= ["microsecond: a time unit"],
   unit::ms      = ["millisecond: a time unit"],
   unit::s       = ["second: a time unit"],
   unit::min     = ["minute: a time unit"],
   unit::h       = ["hour: a time unit"],
   unit::d       = ["day: a time unit"],
   unit::week    = ["week: a time unit"],
   unit::month   = ["month: a time unit (1 month = 30 days)"],
   unit::year    = ["year: a time unit (1 year = 365 days)"],
   //-------------
   // electrical amperage
   //-------------
   unit::nA      = ["nanoampere: a unit of electrical current"],
   unit::microA  = ["microampere: a unit of electrical current"],
   unit::mA      = ["milliampere: a unit of electrical current"],
   unit::A       = ["ampere: a unit of electrical current"],
   unit::kA      = ["kiloampere: a unit of electrical current"],
   unit::Bi      = ["biot: a unit of the electrical current"],
   //-------------
   // temperature
   //-------------
   unit::K       = ["Kelvin: a unit of temperature"],
   unit::Fahrenheit = ["Fahrenheit: a unit of temperature"],
   unit::Celsius = ["Celsius: a unit of temperature"],
   unit::Rankine = ["Rankine: a unit of temperature"],
   unit::Reaumur = ["Reaumur: a unit of temperature"],
   //-----------------------------------------------------------------
   // molecular substance 
   //-----------------------------------------------------------------
   unit::mol     = ["mole: a unit of molecular substance"],
   unit::kmol    = ["kilomole: a unit of molecular substance"],
   unit::mmol    = ["millimole: a unit of molecular substance"],
   unit::mcmol   = ["micromole: a unit of molecular substance"],
   //------------------------------------------------------------------
   // luminosity
   //------------------------------------------------------------------
    unit::cd     = ["candela: a unit of luminosity"],
    unit::HK     = ["Hefner-Kerze: a unit for luminosity"],
    unit::IK     = ["Internationale Kerze: a unit for luminosity"],
   //------------------------------------------------------------------
   // Raumwinkel Steradiant. Wird hier wie eine SI-Einkeit behandelt. 
   //------------------------------------------------------------------
   unit::degree  = ["degree: a measure for plain angle"],
   unit::rad     = ["radian: a measure for plain angle"],
   //------------------------------------------------------------------
   // Raumwinkel Steradiant. Wird hier wie eine SI-Einkeit behandelt. 
   //------------------------------------------------------------------
   unit::sr      = ["steradian: a measure for solid angle"],
   //-------------------------------------------------------------
   // data size
   //-------------------------------------------------------------
   unit::bit     = ["bit: a unit of data size"],
   unit::kbit    = ["kilobit: a unit of data size"],
   unit::Mbit    = ["megabit: a unit of data size"],
   unit::Gbit    = ["gigabit: a unit of data size"],
   unit::Tbit    = ["terabit: a unit of data size"],
   unit::byte    = ["byte: a unit of data size"],
   unit::kbyte   = ["kilobyte: a unit of data size"],
   unit::Mbyte   = ["megabyte: a unit of data size"],
   unit::Gbyte   = ["gigabyte: a unit of data size"],
   unit::Tbyte   = ["terabyte: a unit of data size"],
   //-------------------------------------------------------------
   // currency
   //-------------------------------------------------------------
   unit::cent    = ["cent: European currency (1 Euro = 100 Cent)"],
   unit::Euro    = ["Euro: European currency"],
   //-------------------------------------------------------------
   // Kraft: Newton
   //-------------------------------------------------------------
   unit::N       = ["Newton: a unit of force"],
   unit::kp      = ["kilopond: a unit of force"],
   unit::pdl     = ["poundal: a unit of force"],
   unit::lbf     = ["pound force: a unit of force"],
   unit::ozf     = ["kip ounce force: a unit of force"],
   unit::tonf    = ["ton force: a unit of force"],
   unit::pond    = ["pond: a unit of force"],
   unit::dyn     = ["dyn: a unit of force"],
   //-------------------------------------------------------------
   // energy 
   //-------------------------------------------------------------
   unit::J     = ["joule: a unit of energy"],
   unit::mJ    = ["millijoule: a unit of energy"],
   unit::kJ    = ["kilojoule: a unit of energy"],
   unit::MJ    = ["megajoule: a unit of energy"],
   unit::microWs = ["microwattsecond: a unit of energy"],
   unit::erg   = ["erg: a unit of energy"],
   unit::cal   = ["calory: a unit of energy"],
   unit::kcal  = ["kilocalory: a unit of energy"],
   unit::kWh   = ["kilowatthour: a unit of energy"],
   unit::kpm   = ["kilopondmeter: a unit of energy"],
   unit::PSh   = ["horse power hour: a (german) unit of energy"],
   unit::eV    = ["electron volt: a unit of energy"],
   unit::Btu   = ["British thermal unit: a unit of energy"],
   unit::therm = ["therm: a unit of energy"],
   unit::Wh    = ["watthours: a unit of energy"],
   unit::kWh   = ["kilowatthours: a unit of energy"],
   unit::MWh   = ["megawatthours: a unit of energy"],
   unit::GWh   = ["gigawatthours: a unit of energy"],
   //-------------------------------------------------------------
   // Torque
   //-------------------------------------------------------------
   // do not enter Nm; it would be used for Joule and Wattsec!
// unit::Nm      = ["Newton meter: a unit of torque and energy"],
   //-------------------------------------------------------------
   // Angular momentum (Drehimpuls)
   //-------------------------------------------------------------
   unit::Nms     = ["Newton meter second: a unit of angular momentum"],
   //-------------------------------------------------------------
   // power 
   //-------------------------------------------------------------
   unit::W   = ["Watt: a unit of power"],
   unit::mcW = ["microwatt: a unit of power"],
   unit::mW  = ["milliwatt: a unit of power"],
   unit::kW  = ["kilowatt: a unit of power"],
   unit::MW  = ["megawatt: a unit of power"],
   unit::GW  = ["gigawatt: a unit of power"],
   unit::PS  = ["horsepower: a (german) unit of power"], // Pferdestaerke
   unit::hp  = ["horsepower: a unit of power (UK)"], // horsepower
   //-------------------------------------------------------------
   // frequency 
   //-------------------------------------------------------------
   unit::Hz = ["Hertz: a unit of frequency"],
   unit::kHz = ["kilohertz: a unit of frequency"],
   unit::MHz = ["megahertz: a unit of frequency"],
   unit::GHz = ["gigahertz: a unit of frequency"],

   unit::bps = ["bits per second: a unit of speed for data transfer"],
   unit::baud= ["baud: a unit of data transfer rate"],
   //-------------------------------------------------------------
   // pressure 
   //-------------------------------------------------------------
   unit::Pa   = ["Pascal: a unit of pressure"],
   unit::hPa  = ["hectopascal: a unit of pressure"],
   unit::bar  = ["bar: a unit of pressure"],
   unit::mbar = ["millibar: a unit of pressure"],
   unit::kbar = ["kilobar: a unit of pressure"],
   unit::psi  = ["pounds per square inch: a unit of pressure"],
   unit::at   = ["atmosphere: a unit of pressure"],
   unit::atm  = ["(physical) atmosphere: a unit of pressure"],
   unit::Torr = ["Torr, millimeters of mercury: a unit of pressure"],
   unit::mmWS  = ["millimeters of water: a (german) unit of pressure"],
   unit::mWS   = ["meters of water: a (german) unit of pressure"],
   unit::mHg   = ["meters of mercury: a unit of pressure"],
   unit::inH2O = ["inch of water: a unit of pressure"],
   unit::inHg  = ["inch of mercury: a unit of pressure"],
   unit::ftH2O = ["foot of water: a unit of pressure"],
   //-------------------------------------------------------------
   // area 
   //-------------------------------------------------------------
   unit::ha         = ["hectares: a (german) unit of area"],
   unit::a          = ["are: a unit of area"],
   unit::barn       = ["barn: a unit of area (used in atomic physics)"],
   unit::acre       = ["acre: a unit of area"],
   unit::rood       = ["rood: a unit of area"],
   unit::township   = ["township: a unit of area"],
   unit::circ_mil   = ["circular mil: a unit of area"],
   unit::circ_inch  = ["circular inch: a unit of area"],
   //-------------------------------------------------------------
   // volume: Liter
   //-------------------------------------------------------------
   unit::l       = ["liter: a unit of volume"],
   unit::ml      = ["milliliter: a unit of volume"],
   unit::hl      = ["hectoliter: a unit of volume"],
   unit::gal_UK  = ["gallon: a unit of volume (UK)"], 
   unit::pk_UK   = ["peck: a unit of volume (UK)"],
   unit::bu_UK   = ["bushel: a unit of volume (UK)"],
   unit::chaldron= ["chauldron: a unit of volume (UK)"],
   unit::pottle  = ["pottle: a unit of volume (UK)"],
   unit::qt_UK   = ["quart: a unit of volume (UK)"],
   unit::pint_UK = ["pint: a unit of volume (UK)"],
   unit::gill_UK = ["gill: a unit of volume (UK)"],
   unit::floz_UK = ["fluid ounce: a unit of volume (UK)"],
   unit::fldr_UK = ["fluid drachm: a unit of volume (UK)"],
   unit::minim_UK= ["minim: a unit of volume (UK)"],
   // Volumen: Fluessigkeitsmasse in US-Einheiten
   unit::gal     = ["gallon: a unit of volume (US)"],
   unit::barrel  = ["barrel: a unit of volume (US)"],
   unit::liq_qt  = ["liquid quart: a unit of volume (US)"],
   unit::liq_pt  = ["liquid point: a unit of volume (US)"],
   unit::gill    = ["gill: a unit of volume (US)"],
   unit::floz    = ["fluid ounce: a unit of volume (US)"],
   unit::fldr    = ["fluid drachm: a unit of volume (US)"],
   unit::minim   = ["minim: a unit of volume (US)"],
   // Volumen: Trockenmasse in US-Einheiten
   unit::dry_bu  = ["dry bushel: a unit of volume (US)"],
   unit::dry_pk  = ["dry peck: a unit of volume (US)"],
   unit::bbl     = ["dray barrel: a unit of volume (US)"],
   unit::dry_gal = ["dray gallon: a unit of volume (US)"],
   unit::dry_qt  = ["dray quart: a unit of volume (US)"],
   unit::dry_pt  = ["dry point: a unit of volume (US)"],
   //------------------------------
   // viscosity: 
   //------------------------------
   unit::P       = ["Poise: a unit of (dynamic) viscosity"],
   unit::St      = ["Stokes: a unit of (kinematic) viscosity"],
   //------------------------------
   // velocity: 
   //------------------------------
   unit::knot    = ["knot: a nautical unit of speed (International)"],
   unit::knot_UK = ["knot: a nautical unit of speed (UK)"],
   unit::mach    = ["mach (speed of sound): a unit of speed"],
   //------------------------------
   // acceleration: 
   //------------------------------
   unit::Gal     = ["Gal: a unit of acceleration"],
   unit::gn      = ["g-force, acceleration of free fall: a unit of acceleration"],
   //------------------------------
   // optics: 
   //------------------------------
   unit::dpt     = ["diopter: unit of refractive power of lenses"],
   //------------------------------
   // light flux 
   //------------------------------
   unit::lm      = ["lumen: a unit of light flux"],
   //------------------------------
   // light intensity 
   //------------------------------
   unit::lx      = ["lux: a unit of light intensity"],
   unit::ph      = ["phot: a unit of light intensity"],
   unit::nx      = ["nox: a unit of light intensity"],// Dunkelbeleuchtungsstaerke
   //----------------y-------------
   // luminosity 
   //------------------------------
   unit::sb      = ["stilb: a unit of luminosity"],    // Leuchtdichte: Stilb
   unit::asb     = ["apostilb: a unit of luminosity"], // Leuchtdichte: Apostilb        
   //----------------y-------------
   // heat density and surface tension
   //----------------y-------------
   unit::langley = ["langley: a unit of heat density and surface tension"], // Bestrahlung: Langley
   //----------------y-------------
   // electromagnetism
   //----------------y-------------
   unit::V       = ["Volt: a unit of electrical voltage"],
   unit::nV      = ["nanovolt: a unit of electrical voltage"],
   unit::microV  = ["microvolt: a unit of electrical voltage"],
   unit::mV      = ["millivolt: a unit of electrical voltage"],
   unit::kV      = ["kilovolt: a unit of electrical voltage"],
   unit::MV      = ["megavolt: a unit of electrical voltage"],
   unit::GV      = ["gigavolt: a unit of electrical voltage"],
   unit::Ohm     = ["Ohm: a unit of electrical resistance"],
   unit::nOhm    = ["nanoohm: a unit of electrical resistance"],
   unit::microOhm= ["microohm: a unit of electrical resistance"],
   unit::mOhm    = ["milliohm: a unit of electrical resistance"],
   unit::kOhm    = ["kiloohm: a unit of electrical resistance"],
   unit::MOhm    = ["megaohm: a unit of electrical resistance"],
   unit::GOhm    = ["gigaohm: a unit of electrical resistance"],
   unit::S       = ["Siemens: a unit of electrical conductance"],
   unit::C       = ["Coulomb: a unit of electrical charge"],
   unit::F       = ["Farad: a unit of electrical capacity"],
   unit::pF      = ["picofarad: a unit of electrical capacity"],
   unit::nF      = ["nanofarad: a unit of electrical capacity"],
   unit::microF  = ["microfarad: a unit of electrical capacity"],
   unit::mF      = ["millifarad: a unit of electrical capacity"],
   unit::kF      = ["kilofarad: a unit of electrical capacity"],
   unit::Wb      = ["Weber: a unit of magnetic flux"],
   unit::M       = ["Maxwell: a unit of magnetic flux"],
   unit::T       = ["Tesla: a unit of magnetic flux density (induction)"], 
   unit::G       = ["Gauss: a unit of magnetic flux density (induction)"],
   unit::Oe      = ["Oerstedt: a unit of magnetic field strength"],
   unit::H       = ["Henry: a unit of magnetic inductivity"],
   unit::Gb      = ["Gilbert: a unit of magnetic force"],
   //------------------------------
   // radiation
   //------------------------------
   unit::Bq      = ["Becquerel: a unit of (radio-)activity"],
   unit::Ci      = ["Curie: a unit of (radio-)activity"],
   unit::Gy      = ["Gray: a unit of energy dose"],
   unit::rd      = ["Rad: a unit of energy dose"],
   unit::R       = ["Roentgen: a unit of ionic dose"],
   unit::Sv      = ["Sievert: a unit of equivalent dose"],
   unit::rem     = ["Rem (Roentgen equivalent man): a unit of equivalent dose"],
   //----------------y-------------
   // currency (European exchange rates)
   // http://www.infomia.com/wiki,index,goto,Abk%C3%BCrzungen/W%C3%A4hrungen.html
   //----------------y-------------
   unit::DM      = ["Deutsche Mark: currency"],
   unit::BEF     = ["Belgian Franc: currency"],
   unit::FIM     = ["Finnish Markka: currency"],
   unit::FRF     = ["French Franc: currency"],
   unit::IEP     = ["Irisch Pound: currency"],
   unit::ITL     = ["Italian Lire: currency"],
   unit::LUF     = ["Luxembourgian Franc: currency"],
   unit::NLG     = ["Dutch Gulden: currency"],
   unit::ATS     = ["Austrian Schilling: currency"],
   unit::PTE     = ["Portugese Escudo: currency"],
   unit::ESP     = ["Spanish Peseta: currency"],
   /*****************
   Sonstige Einheiten
   *****************/
   // Laengenbozogene Masse von textilen Fasern und Garnen: Tex, Denier
   unit::tex     = ["Tex: a unit of mass per length (yarns, fibres etc.)"], 
   unit::den     = ["Denier: a unit of mass per length (yarns, fibres etc.)"],
null()
):

delete unit::infoTable[FAIL]:

//----------------------------------------------
// unit::newUnit(day = 20*unit::hour) -- creates
// the new unit unit::day and inserts it into
// the ConversionTable
//----------------------------------------------
unit::newUnit := proc(eq)
local newunit, knownunit, baseunit, f, u, v;
begin
  if args(0)=0 or type(eq) <> "_equal" then
     error("expecting an equation such as 'hour = 60*unit::minute'"):
  end_if;
  newunit:= op(eq, 1);

  // Modified by O.K.
  if domtype(newunit) <> DOM_IDENT and domtype(newunit) <> DOM_STRING then
     error("the new unit must be an identifier or a string. Got: ".expr2text(newunit))
  end_if;
  if domtype(newunit) = DOM_IDENT then
     newunit:= expr2text(newunit)
  end_if;
  
  if slot(unit, newunit) <> FAIL then
     error("the unit ".newunit." already exists.")
  end;

  knownunit:= op(eq, 2);

  if knownunit = FAIL then
     error("the right hand side of the equation contains an unknown unit.")
  end_if;

  if not (testtype(knownunit, "_mult") or
          testtype(knownunit, "_power") or
          testtype(knownunit, unit)) then
     f:= 1;
  else
     [f, knownunit]:= unit::split(knownunit);
  end_if;

 if domtype(float(f)) <> DOM_FLOAT then
    error("Conversion factor should be an arithmetical expression.")
 end_if;
 u := unit::findUnits(knownunit);
 if nops(u) = 1 then
    baseunit := op(u);
    v := unit::ConversionTable[baseunit];
    if baseunit = knownunit and type(v) = DOM_LIST then
       sysassign(slot(unit, newunit), new(unit, newunit));
       f := f * v[1]; baseunit := v[2];
       sysassign(unit::ConversionTable[slot(unit, newunit)], [f, baseunit]); 
       return(slot(unit, newunit))
    end
 end;
 for baseunit in u do
     if slot(unit, extop(baseunit, 1)) = FAIL then
        error("Definition of ".newunit."uses an unknown unit: ".expr2text(baseunit))
     end_if
  end_for;
  sysassign(slot(unit, newunit), new(unit, newunit));
  sysassign(unit::ConversionTable2[slot(unit, newunit)], [f, knownunit]); 
  sysassign(unit::nonbaseunits, append(unit::nonbaseunits, slot(unit, newunit)));
  return(slot(unit, newunit));
end_proc:
     
unit::prefixes := [
   "E"      = 10^18   , // exa
   "exa"    = 10^18   , // exa
   "P"      = 10^15   , // peta
   "peta"   = 10^15   , // peta
   "T"      = 10^12   , // tera
   "tera"   = 10^12   , // tera
   "G"      = 10^ 9   , // giga
   "giga"   = 10^ 9   , // giga
   "M"      = 10^ 6   , // mega
   "mega"   = 10^ 6   , // mega
   "k"      = 10^ 3   , // kilo
   "kilo"   = 10^ 3   , // kilo
   "h"      = 10^ 2   , // hecto
   "hecto"  = 10^ 2   , // hecto
   "da"     = 10^ 1   , // deka
   "deka"   = 10^ 1   , // deka
   "d"      = 10^(- 1), // deci
   "deci"   = 10^(- 1), // deci
   "c"      = 10^(- 2), // centi
   "centi"  = 10^(- 2), // centi
   "m"      = 10^(- 3), // milli
   "milli"  = 10^(- 3), // milli
   "mc"     = 10^(- 6), // micro
   "micro"  = 10^(- 6), // micro
   "n"      = 10^(- 9), // nano
   "nano"   = 10^(- 9), // nano
   "p"      = 10^(-12), // pico
   "pico"   = 10^(-12), // pico
   "f"      = 10^(-15), // femto
   "femto"  = 10^(-15), // femto
   "a"      = 10^(-18), // atto
   "atto"   = 10^(-18)  // atto
]:

// This list is used to generate units with prefixes such as
// Hz -> kHz, MHz, GHz, ...

unit::generate_list := [
                        unit::m,  // we have nm, micrometer, mm,cm,dm,m, km
                        unit::g,  // we have microgram, mg, g, kg, t
                        unit::s,  // we have ns, micros, ms, s
                        unit::A,  // we have nA, microA, mA, A, kA,
                        unit::K,
                        unit::mol, // we have micromol, mmol, mol, kmol
                        unit::cd, 
                        unit::W,  // we have microW, mW, W, kW, MW, GW
                        unit::Ws, // we have microWs, mWs, Ws
                        unit::ohm,// we have nOhm,microOhm,mOhm,Ohm,kOhm,MOhm,GOhm
                        unit::Ohm,// we have nOhm,microOhm,mOhm,Ohm,kOhm,MOhm,GOhm
                        unit::Sv, 
                        unit::rem, 
                        unit::N,
                        unit::Nm, 
                        unit::Nms, 
                        unit::V,  // we have nV, microV, mV, V, kV, MV, GV
                        unit::J,  // we have mJ, J, kJ, MJ
                        unit::Hz, // we have Hz, kHz, MHz, GHz
                        unit::Bq, 
                        unit::C, 
                        unit::F,  // we have pF,nF,microF,mF,F,kF
                        unit::Gy, 
                        unit::eV, 
                        unit::l,  // we have ml, l, hl
                        unit::Pa
                       ]:

unit::generate := proc()
  local a, b, eqn, f, get_first_char, s, u;
begin
  for u in unit::generate_list do
      for f in unit::prefixes do
          a := lhs(f);
          b := extop(u, 1);
          s := ``. a . b;
          if testtype(s, DOM_IDENT) then
             s := expr2text(s)
          end_if;
          if not testtype(s, DOM_STRING) then
             error("Fatal initialization error")
          end_if;
          eqn := unit::init(s = [rhs(f), u]);
          if eqn <> null() then
             unit::ConversionTable2[lhs(eqn)] := rhs(eqn);
          end_if;
      end_for
  end_for 
end_proc(): // Execute unit::generate immediately!

//------------------------------------------------------
unit::convert2baseunits := proc(x)
  local ctn, n, r, un, uni;
begin
  assert(args(0)=1);
  if domtype(float(x)) = DOM_FLOAT then
     return(x);
  end_if;
  if testtype(x, "_plus") or
     testtype(x, "_mult") or
     testtype(x, "_power") then
    return( map(x, unit::convert2baseunits) )
  end_if;
  // All units in unit::ConversionTable2
  un := unit::nonbaseunits;
  r  := x;
  if domtype(un) <> DOM_LIST and 
     domtype(un) <> DOM_SET  then
     un := {un}
  end_if;
  for uni in un do
      if has(r, uni) then
         ctn := unit::ConversionTable2[uni];
         n   := ctn[1]*ctn[2];
         r   := subs(r, uni=n);
      end_if
  end_for;
  r
end_proc:
//------------------------------------------------------
unit::SIunits := [unit::kg, unit::s, unit::m, 
                  unit::A, unit::K, unit::cd, 
                  unit::mol, unit::Cent, unit::sr]:
//------------------------------------------------------
unit::convert2SIunits := proc(x)
   local si, y, z;
begin
   if domtype(float(x)) = DOM_FLOAT then
      return(x);
   end_if;
   if testtype(x, "_plus") or
      testtype(x, "_mult") or
      testtype(x, "_power") then
      return( map(x, unit::convert2SIunits) )
   end_if;
   z := x;
   repeat
     y := z;
     z := unit::convert2baseunits(y)
   until z = y end_repeat;
   for si in unit::SIunits do
       y := unit::convert(y, si)
   end_for;
   y
end_proc:
//------------------------------------------------------
unit::convert2 := proc(x)
  local i, r, u, un, y;
begin
  if domtype(float(x)) = DOM_FLOAT then
     return(x)
  end_if;
  assert(args(0)=2);

  //------------------------------------------------
  // Further down below, only products are expected.
  // Sums must be converted term by term:
  //------------------------------------------------
  if testtype(x, "_plus") then
     r := map(x, unit::convert2, args(2));
     return( r )
  end_if;

  un := args(2);
  if un = "SI units" then
    return( unit::convert2SIunits(x) )
  elif un = "base units" then
    return( unit::convert2baseunits(x) )
  elif testtype(un, unit) then
    un := [ un ]
  elif testtype(un, DOM_LIST) then
    // Nothing to do
  else
    un := [ op(unit::findUnits(un)) ]
  end_if;

  un := split(un, unit::DefinedInCT);
  un := un[2].un[1];
            
  y := unit::convert2SIunits(x);

  for i from 1 to nops(un) do
      u := unit::convert2SIunits(un[i]);
      y := unit::convert_back(y, u, un[i]);
      if y = un[i] or 
         (type(y)= "_mult" and 
          nops(y) = 2 and 
          op(y, 1) = un[i] and 
          type(op(y,2)) <> unit) then
         // Ist das Ergebnis y bereits vom Typ c * un[i],
         // so liefere sofort dieses Ergebnis 
         break
      else
         // Mehrfaches auftreten der Einheit un[i]
         // in y herausziehen. Beispiel:
         // x = J*W*s soll in J umgeformt werden.
         // y = unit::convert2SIunits(x) -> m^4*kg^2/s^4
         // Wird nur einmal unit::convert_back angewendet,
         // dann lautet das Ergebnis J*m^2*kg/s^2
         // Das Abbruchkriterium mittels length ist 
         // eine Heuristik.
         repeat
           r := y;
           y := unit::convert_back(r, u, un[i]);
         until unit::length(y) >= unit::length(r) end;
         y := r
      end
   end_for;

   y;
end_proc:
//------------------------------------------------------
unit::convert_back := proc(y, u, uni)
  local r, r1, v;
begin
      // Pruefe zunaechst, ob u - das ist die
      // in SI-Einheiten konvertierte Einheit uni -
      // zur selben Basiseinheit gehoert. Falls ja,
      // dann wird sofort in die Einheit uni 
      // konvertiert.
      // Beispiel:
      // y = m, u = 1000*m, uni = km
      // Meter und Kilometer gehoeren zur selben
      // Basiseinheit. Es wird also 1/1000*km 
      // zurueckgegeben.
      [r, v] := unit::split(u);
      if unit::DefinedInCT(v) and unit::DefinedInCT(uni) then
         if unit::samebaseunit(v, uni) then
            return( subs(y, v=uni/r) )
         end
      end;

      r := y/u; r1 := y*u; v := uni;
      // Idee ist: un oder 1/un ist enthalten.
      // Ist x/un bzw. y/u "einfacher" als 
      // x*un bzw. y*u, dann wird y/u genommen.
      // Beispiel:
      // unit::convert2(1/unit::J, unit::N)
      // Dabei ist y/u gleich s^4 / (m^3 * kg ^ 2)
      // hingegen ist y*u gleich 1/m. Also wird y*u
      // gewaehlt.
      // Ist unit::length(r)  + 1 > unit::length(y)
      // und unit::length(r1) + 1 > unit::length(y)
      // dann spricht einiges dafuer, dass eine 
      // Konvertierung "nicht sinnvoll" ist.
      // Soll eine "zwangsweise" Konvertierung - die 
      // Konvertierung wird also in jedem Fall durchgefuehrt -
      // vermieden werden, so ist der folgende 
      // Quelltext zu aktivieren, der allerdings
      // auch "sinnvolle" Konvertierungen verhindert:
      /*
      if not unit::samebaseunit(u, y) and
         unit::length(r)  + 1 > unit::length(y) and
         unit::length(r1) + 1 > unit::length(y) then
         next
      end_if;
      */
      if unit::length(r1) < unit::length(r) then 
         r := r1; v := 1/uni
      end_if;
      r * v
end:
//-------------------------------------------------
unit::simplify2 := proc(ex)
   local u, x;
begin
   if not (testtype(ex, "_plus") or
           testtype(ex, "_mult") or
           testtype(ex, "_power")) then
      return(ex)
   end;
   x := unit::convert2SIunits(ex);
// u := unit::findUnits(op(ex,1)); // Olli's version
   u := unit::findUnits(ex);       // Walter's version
   unit::convert2(x, [op(u)])
end:
//-------------------------------------------------

//-------------------------------------------------
/*
Die Ausgaberoutine unit::display basiert auf der 
internen Ordnung der Elemente des Domains unit
und der internen Sortierung von Termen in 
Produkten.
Beispiel: unit::kW * unit::s wird als s * kW 
ausgegeben. 
Das kann ggf. unerwuenscht sein. In dieser
Ausgaberoutine wird daher eine Umsortierung
vorgenommen, die durch unit::sort implementiert
ist.
*/

unit::display:=proc(x)
  local c, y;
begin
  if not (testtype(x, "_mult")  or testtype(x, "_plus") or
          testtype(x, "_power") or testtype(x, unit)) then
     return(x);
  // error("Illegal 1st argument")
  end_if;
  if testtype(x, "_plus") then
     y := map([op(x)], unit::display);
     return( hold(_plus)(op(y)) )
  end_if;
  if not (testtype(x, "_mult") or 
          testtype(x, "_power") or 
          testtype(x, unit)) then
     return(x);
  end_if;
  [c, y] := unit::split(x);
  // Ist c = 1/n, dann wird c * y als y/c ausgegeben.
  // Das ist nicht gewuenscht. Gewuenscht wird 1/c * y
  y := unit::sort(y);
  if domtype(c) = DOM_RAT then
     // Es wird hold verwandt, damit die internen 
     // Ausgaberoutinen der Operatoren * und / 
     // (_mult und _divide in Funktionsschreibweise)
     // die gewuenschte Ausgabe nicht abaendern
     //return( hold(_mult)(hold(_divide)(op(c)), y) )
     if c < 0 then
        c := -c;
        c := hold(_negate)(hold(_divide)(op(c)))
     else
        c := hold(_divide)(op(c))
     end_if
  end_if;
  if testtype(y, "_power") and testtype(op(y,2), Type::NegInt) then
     // y = 1/a^n, n > 0
     // hold(_mult)(c, y) liefert c/a^n
     // Gewuenscht ist c * 1/a^n, daher:
     y := hold(_divide)(1, 1/y)
  end_if;
  return( hold(_mult)(c, y) )
end_proc:
//-------------------------------------------------
unit::split := proc(z)
  local c, i, n, x, y;
begin
  if not (testtype(z, "_mult") or testtype(z, "_power") or testtype(z, unit)) then
     error("Illegal 1st argument")
  end_if;
  if not testtype(z, "_mult") then
     return( [1, z] )
  end_if;
  y := select(z, hastype, unit);
  x := z/y;
  if type(x) = "_mult" then
     // Ist in einem Produkt x[1] * ... * x[n]
     // eine Konstante vorhanden, dann ist die
     // Konstante der letzte Term, also x[n].
     n := nops(x);
     c := op(x, n);
     if testtype(c, Type::Numeric) then
     // y := x/c funktioniert nicht, wenn die
     // Konstante c eine Gleitkommazahl ist.
     // Beispiel: x = 5.2/m. Dann ist c = 5.2
     // und x/c liefert 1.0/m. Das ist nicht
     // erwuenscht. Gewuenscht ist 1/m. Daher
     // wird statt x/c = (x[1] * ... * x[n])/x[n]
     // das Produkt x[1] * ... * x[n-1] berechnet.
        x := _mult(op(x,i) $ i=1..n-1)
     else
        c := 1
     end_if;
     x := hold(_mult)(c, x)
  end_if;
  [x, y]
end_proc:
//-------------------------------------------------
// unit::length := x -> nops(unit::split(x)[2]):

unit::length := proc(x)
begin

  /* Urspruengliche Fassung:
  if testtype(x, "_mult") then
     //nops(unit::split(x)[2])
     nops(x)
  else
     // Ist testtype(x, type) = TRUE 
     // fuer type = Type::Numeric, unit oder "_power"
     // dann wird die Laenge als 1 definiert
     1
  end_if
  */
  
  if testtype(x, "_mult") then
     _plus(map(op(x), unit::length))
  elif testtype(x, "_power") then
     if op(x,2) = -1 then
        1
     else
        2
     end
  else
     // Ist testtype(x, type) = TRUE 
     // - fuer type = Type::Numeric oder unit -
     // dann wird die Laenge als 1 definiert
     1
  end_if
 
end_proc:
//-------------------------------------------------
unit::sort := proc(x)
  local d, n;
begin
  if not (testtype(x, "_mult") or 
          testtype(x, "_power") or 
          testtype(x, unit)
         ) then
     error("Illegal 1st argument")
  end_if;
  if not testtype(x, "_mult") then
     return( x )
  end_if;
  n := numer(x); d := denom(x);
  n := unit::sort_mult(n); 
  d := unit::sort_mult(d);
  if d = 1 then
     n
  else 
     // n / d
     hold(_divide)(n,d)
  end_if
end_proc:
//-------------------------------------------------
unit::sort_mult := proc(x)
  local r, y, u1, u2, u3;
begin
  if not testtype(x, "_mult") then
     return( x )
  end_if;
  y := [op(x)]; 
  r := [];

  u1:= contains(y, unit::N):
  u2:= contains(y, unit::m):
  u3:= contains(y, unit::s):
  if u1 > 0 and u2 > 0 then
     y[u1]:= NIL;
     y[u2]:= NIL;
     if u3 > 0 then
       y[u3]:= NIL;
       r := append(r, unit::Nms)
     else
       r := append(r, unit::Nm)
     end_if
  end_if;

  u1:= contains(y, unit::W):
  u2:= contains(y, unit::s):
  if u1 > 0 and u2 > 0 then
     y[u1]:= NIL;
     y[u2]:= NIL;
     r := append(r, unit::W, unit::s)
  end_if;

  u1:= contains(y, unit::V):
  u2:= contains(y, unit::A):
  if u1 > 0 and u2 > 0 then
     y[u1]:= NIL;
     y[u2]:= NIL;
     r := append(r, unit::V, unit::A)
  end_if;

  u1:= contains(y, unit::kW):
  u2:= contains(y, unit::s):
  if u1 > 0 and u2 > 0 then
     y[u1]:= NIL;
     y[u2]:= NIL;
     r := append(r, unit::kW, unit::s)
  end_if;

  u1:= contains(y, unit::kW):
  u2:= contains(y, unit::h):
  if u1 > 0 and u2 > 0 then
     y[u1]:= NIL;
     y[u2]:= NIL;
     r := append(r, unit::kW, unit::h)
  end_if;

  u1:= contains(y, unit::ohm):
  u2:= contains(y, unit::m):
  if u1 > 0 and u2 > 0 then
     y[u1]:= NIL;
     y[u2]:= NIL;
     r := append(r, unit::ohm, unit::m)
  end_if;
  y:= subs(y, NIL = null());

  if y = [] then
     if nops(r) = 1 then
        op(r)
     else
        // r[1] * ... * r[n]
        hold(_mult)(op(r))
     end_if;
  else // y <> 1
     // r[1] * ... * r[n] * y[1] * ... * y[m]
     hold(_mult)(op(r), op(y))
  end_if
end_proc:
//-------------------------------------------------
unit::TemperatureList := ["Celsius", "Fahrenheit", "Rankine", "Reaumur"]:
//-------------------------------------------------
unit::TemperatureList := map(unit::TemperatureList, 
                             u -> ( slot(unit, u) := new(unit, u) )
                         ):
//-------------------------------------------------
/*

// Es muessen 3 Argumente sein.
// Das Zusammenfassen von c und x zu einem Argument, naemlich c*x
// funktioniert nicht, denn 0*x wird sofort zu 0 vereinfacht, d.h.
// die hier zwingend benoetigte Einheit x geht verloren.
 
unit::temperature := proc(c, x, y)
  local normal_unit, u, v;
begin
  normal_unit := proc(u)
  begin
    if has(unit::TemperatureList, u) then
       extop(u, 1)
    elif u = unit::K or 
         u = unit::Kelvin or 
         u = unit::kelvin then
       "Kelvin"
    else
       FAIL
    end
  end;

  if args(0) <> 3 then 
     error("3 arguments expected.")
  end;
  u := normal_unit(x);
  if u = FAIL then
     error("Illegal 1st argument.")
  end;
  v := normal_unit(y);
  if v = FAIL then
     error("Illegal 2nd argument.")
  end;

  if u = v then return(c*x) end;
  
  slot(unit, u."2".v)(c)
end:
*/

unit::Celsius2Kelvin:=     T -> T + 273.15:
unit::Celsius2Reaumur:=    T -> 4/5 * T:
unit::Celsius2Fahrenheit:= T -> T * 9/5 + 32:
unit::Celsius2Rankine:=    T -> 1.8 * T  + 491.67:

unit::Fahrenheit2Kelvin:=  T -> (T - 32) * 5/9 + 273.15:
unit::Fahrenheit2Celsius:= T -> (T - 32) * 5/9:
unit::Fahrenheit2Reaumur:= T -> 4/9 * (T - 32):
unit::Fahrenheit2Rankine:= T -> T + 459.67:

unit::Kelvin2Celsius:=     T -> T - 273.15:
unit::Kelvin2Reaumur:=     T -> 0.8 * (T - 273.15):
unit::Kelvin2Fahrenheit:=  T -> 1.8 * (T - 273.15) + 32:
unit::Kelvin2Rankine:=     T -> 1.8 * T:

unit::Rankine2Kelvin:=     T -> 5/9 * T:
unit::Rankine2Celsius:=    T -> 5/9 * (T - 491.67):
unit::Rankine2Reaumur:=    T -> 4/9 * (T - 491.67):
unit::Rankine2Fahrenheit:= T -> T - 459.67:

unit::Reaumur2Kelvin:=     T -> 1.25 * T + 273.15:
unit::Reaumur2Celsius:=    T -> 5/4 * T:
unit::Reaumur2Fahrenheit:= T -> 9/4 * T + 32:
unit::Reaumur2Rankine:=    T -> 2.25 * T + 491.67:

unit::interface_CT := proc()
  local x;
begin
  text2expr(extop(lhs(x),1)) $ x in unit::ConversionTable
//hold(``).(extop(lhs(x),1)) $ x in unit::ConversionTable
end:

unit::interface_CT2 := proc()
  local l, u, x;
begin
  l := [];
  for x in unit::ConversionTable2 do
      u := extop(lhs(x),1);
      l := append(l, text2expr(u));
  end_for;
  op(l) 
end:

unit::interface_T := proc()
  local x;
begin
  text2expr(extop(x,1)) $ x in unit::TemperatureList;
//hold(``).(extop(x,1)) $ x in unit::TemperatureList
end:
  
unit::interface:= {
//hold(ConversionTable), 
//hold(ConversionTable2),
  hold(convert), 
  hold(convert2SIunits), 
//hold(convert2baseunits),
  hold(simplify), 
  hold(findUnits), 
  hold(newUnit),
  hold(expr2text), 
  hold(display), 
  //hold(expr), 
  hold(Kelvin2Celsius), hold(Kelvin2Reaumur), 
  hold(Kelvin2Fahrenheit), hold(Kelvin2Rankine),
  hold(Celsius2Kelvin), hold(Celsius2Reaumur), 
  hold(Celsius2Fahrenheit), hold(Celsius2Rankine),
  hold(Reaumur2Kelvin), hold(Reaumur2Celsius), 
  hold(Reaumur2Fahrenheit), hold(Reaumur2Rankine),
  hold(Fahrenheit2Kelvin), hold(Fahrenheit2Celsius), 
  hold(Fahrenheit2Reaumur), hold(Fahrenheit2Rankine),
  hold(Rankine2Kelvin), hold(Rankine2Celsius), 
  hold(Rankine2Reaumur), hold(Rankine2Fahrenheit),
  unit::interface_CT(),  // units::cm, units::kg etc.
  unit::interface_CT2(), // units::N, units::J etc.
  unit::interface_T()    // unit::Celsius, unit::Fahrenheit, etc.
}: 

// --------------------------------------------------------
// utility for filling in unit::infoTable to get information 
// about units with different names but the same meaning 
// such as unit::cm = unit::centimeter = unit::centimeters:
// --------------------------------------------------------
unit::getAlterEgos:= proc(u)
local uu, pre, base, AlterEgos, U, unit_found;
begin
  if not domtype(u) = unit then
     return()
  end_if;
  //----------------------------------------------------------------
  // this part is for adding info entries for prefix-units
  // such as m (=mill), h (=hecto), k = (kilo), M (=mega) etc.
  //----------------------------------------------------------------
  if type(unit::infoTable[op(u, 1)]) = "_index" then
     pre:= "";
     uu:= extop(u, 1);
     // do "da" before "d" and "a" and "mc" before "m" and "c",
     // otherwise we will generate deciattoXYZ and millicentiXYZ
     if pre = "" and length(uu) > 5 then
        case substring(uu, 1..5) 
        of "centi" do [pre, uu]:= ["centi", substring(uu, 6..length(uu))]; break;
        of "femto" do [pre, uu]:= ["femto", substring(uu, 6..length(uu))]; break;
        of "hecto" do [pre, uu]:= ["hecto", substring(uu, 6..length(uu))]; break;
        of "micro" do [pre, uu]:= ["micro", substring(uu, 6..length(uu))]; break;
        of "milli" do [pre, uu]:= ["milli", substring(uu, 6..length(uu))]; break;
        end_case;
     end_if;
     if pre = "" and length(uu) > 4 then
        case substring(uu, 1..4) 
        of "peta" do [pre, uu]:= ["peta",  substring(uu, 5..length(uu))]; break;
        of "tera" do [pre, uu]:= ["tera",  substring(uu, 5..length(uu))]; break;
        of "giga" do [pre, uu]:= ["giga",  substring(uu, 5..length(uu))]; break;
        of "mega" do [pre, uu]:= ["mega",  substring(uu, 5..length(uu))]; break;
        of "kilo" do [pre, uu]:= ["kilo",  substring(uu, 5..length(uu))]; break;
        of "nano" do [pre, uu]:= ["nano",  substring(uu, 5..length(uu))]; break;
        of "pico" do [pre, uu]:= ["pico",  substring(uu, 5..length(uu))]; break;
        of "atto" do [pre, uu]:= ["atto",  substring(uu, 5..length(uu))]; break;
        end_case;
     end_if;
     if pre = "" and length(uu) > 3 then
        case substring(uu, 1..3) 
        of "exa" do [pre, uu]:= ["exa",   substring(uu, 4..length(uu))]; break;
        end_case;
     end_if;
     if pre = "" and length(uu) > 2 then
        case substring(uu, 1..2) 
        of "da" do [pre, uu]:= ["deka",  substring(uu, 3..length(uu))]; break;
        of "mc" do [pre, uu]:= ["micro", substring(uu, 3..length(uu))]; break;
        end_case;
     end_if;
     if pre = "" and length(uu) > 1 then
        case substring(uu, 1..1) 
        of "E" do [pre, uu]:= ["exa",   substring(uu, 2..length(uu))]; break;
        of "P" do [pre, uu]:= ["peta",  substring(uu, 2..length(uu))]; break;
        of "T" do [pre, uu]:= ["tera",  substring(uu, 2..length(uu))]; break;
        of "G" do [pre, uu]:= ["giga",  substring(uu, 2..length(uu))]; break;
        of "M" do [pre, uu]:= ["mega",  substring(uu, 2..length(uu))]; break;
        of "k" do [pre, uu]:= ["kilo",  substring(uu, 2..length(uu))]; break;
        of "h" do [pre, uu]:= ["hecto", substring(uu, 2..length(uu))]; break;
        of "d" do [pre, uu]:= ["deci",  substring(uu, 2..length(uu))]; break;
        of "c" do [pre, uu]:= ["centi", substring(uu, 2..length(uu))]; break;
        of "m" do [pre, uu]:= ["milli", substring(uu, 2..length(uu))]; break;
        of "n" do [pre, uu]:= ["nano",  substring(uu, 2..length(uu))]; break;
        of "p" do [pre, uu]:= ["pico",  substring(uu, 2..length(uu))]; break;
        of "f" do [pre, uu]:= ["femto", substring(uu, 2..length(uu))]; break;
        of "a" do [pre, uu]:= ["atto",  substring(uu, 2..length(uu))]; break;
        end_case;
     end_if;
     if type(unit::infoTable[slot(unit, uu)]) <> "_index" then
        if slot(unit, uu) <> FAIL then
          sysassign(unit::infoTable[op(u,1)], [pre.(unit::infoTable[slot(unit, uu)][1])]);
        end_if:
     end_if;
  end_if;
  //----------------------------------------------------------------
  // Now, the infoTable for Watt is filled with info
  // about mW, kW, GW etc.
  //----------------------------------------------------------------
  // Proceed to find AlterEgos
  //----------------------------------------------------------------
  if type(unit::ConversionTable[op(u, 1)]) <> "_index" then
     base:= unit::ConversionTable[op(u, 1)];
  elif type(unit::ConversionTable2[op(u, 1)]) <> "_index" then
     base:= unit::ConversionTable2[op(u, 1)];
  else
     return();
  end_if:
  AlterEgos:= {op(select(unit::ConversionTable , u -> bool(op(u,2) = base))),
               op(select(unit::ConversionTable2, u -> bool(op(u,2) = base)))};
  AlterEgos:= sort(map([op(AlterEgos)], op, 1));
  u:= op(u, 1);
  unit_found:= FALSE;
  for U in AlterEgos do
      if type(unit::infoTable[U]) <> "_index" then
         u:= U; 
         unit_found:= TRUE;
         break;
      end_if;
  end_for;
  if unit_found then
    for U in AlterEgos do
      if type(unit::infoTable[U]) = "_index" or
         nops(unit::infoTable[U]) = 1 then
         sysassign(unit::infoTable[U], unit::infoTable[u]. [AlterEgos]);
      end_if;
    end_for:
  end_if;
  null();
end_proc:

//--------------------------------------------------------------------
unit::printInfo := proc(u)
 local p;
 save PRETTYPRINT;
begin
   PRETTYPRINT := FALSE;
   p := () -> print(Unquoted, args());
   unit::getAlterEgos(u);
   if type(unit::infoTable[u]) = DOM_LIST then
      p(unit::infoTable[u][1]);
      if nops(unit::infoTable[u]) > 1 and
         nops(unit::infoTable[u][2]) > 1 then
         p("Alternative names: ".
           expr2text(op(unit::infoTable[u][2] /*minus {u}*/)));
      end_if;
   else
      p(expr2text(u). " -- an object of domain type 'unit'")
   end_if;
   null():
end_proc:
