/** MuPAD representation of XML trees
 *
 *  Elements of this domain are used to represent arbitrary XML trees.
 *  Currently this domain is used to represent MathML Content and
 *  Presentation trees.
 */

alias(XML = adt::XML):

XML:=newDomain(hold(XML)):

XML::interface := {hold(Document), hold(Element),  hold(Doctype),
                   hold(Comment), hold(ProcessingInstruction),
                   hold(CDATA), hold(ElementDeclaration),
                   hold(AttributeDeclaration), hold(EntityDeclaration),
                   hold(NotationDeclaration),
                   // IO settings
                   hold(openOutputFile), hold(closeOutputFile),
                   // Utilities
                   hold(markupType), hold(elementName), hold(attributes),
                   hold(str2cdata), hold(multiLineStr2cdata),
                   hold(setNamespace)}:


// do we need this?
// XML::new := () -> new(dom, args()):

// container for a complete XML document
XML::Document :=
proc()
  local argTypes;
begin
  if testargs() then
    argTypes := map({args()}, domtype);
    if argTypes <> {} and argTypes <> {dom} then
      error("Wrong argument type.  Only elements of ".dom." expected")
    end_if;
  end_if;
  new(dom, "document", "", table(), args())
end:

// classical XML element with or without attributes
// only interesting type at the moment.  Children may
// be other XML elements
XML::Element :=
proc(elemName : DOM_STRING)
  local attributes, operands, opTypes, i;
begin
  i := 2;
  attributes := [];
  while i <= args(0) and testtype(args(i), DOM_LIST) = TRUE do
    attributes := attributes.args(i);
    i := i+1;
  end_while;
  operands := [args(i..args(0))];
  if testargs() then
    opTypes := map({op(operands)}, domtype);
    if opTypes minus {dom, DOM_IDENT, DOM_INT, DOM_FLOAT, DOM_STRING} <> {} then
      error("Elements must be of type ".expr2text(dom))
    end_if;
  end_if;
  new(dom, "element", elemName, attributes, op(operands)):
end_proc:

// Entity reference:  &...;  
// some renderers have Problems with entity names; set UNICODE values
XML::Entity := () -> new(dom, "entity", [], args()):

// Comment:  <!--  ... -->
// trivial implementation:  stored as a plain MuPAD string
XML::Comment :=
proc(x : DOM_STRING)
begin
  new(dom, "comment", "", [], args())
end_proc:

// Ducument Type:  <!DOCTYPE   ... [...]>
// trivial implementation of attributes:  stored as a plain MuPAD string
XML::Doctype :=
proc(x : DOM_STRING)
begin
  new(dom, "DOCTYPE", "", args())
end_proc:

// Processing Instruction (PI):  <?name pidata?>
// trivial implementation:  stored as a plain MuPAD string
XML::ProcessingInstruction := 
proc(x : DOM_STRING)
begin
  new(dom, "PI", "", [], args())
end_proc:

// CDATA section:  <![CDATA[ ... ]]>
// trivial implementation:  stored as a plain MuPAD string
XML::CDATA :=
proc(x : DOM_STRING)
begin
  new(dom, "CDATA", "", [], args())
end_proc:

// Element Type Declaration: <!ELEMENT ...>
// trivial implementation:  stored as a plain MuPAD string
XML::ElementDeclaration :=
proc(x : DOM_STRING)
begin
  new(dom, "ELEMENT", "", [], args())
end_proc:

// Attribute List Declaration:  <!ATTLIST ...>
// trivial implementation:  stored as a plain MuPAD string
XML::AttributeDeclaration :=
proc(x : DOM_STRING)
begin
  new(dom, "ATTLIST", "", [], args())
end_proc:

// Entity Declaration:  <!ENTITY ...>
// trivial implementation:  stored as a plain MuPAD string
XML::EntityDeclaration :=
proc(x : DOM_STRING)
begin
  new(dom, "ENTITY", "", [], args())
end_proc:

// Notation Declaration:  <!NOTATION  ...>
// trivial implementation:  stored as a plain MuPAD string
XML::NotationDeclaration :=
proc(x : DOM_STRING)
begin
  new(dom, "NOTATION", "", [], args())
end_proc:



/** internal Methods:
  * only these methods access the internal representation
  */
// returns the markup type of the given element
XML::markupType := x -> extop(x, 1):

// elementName: a string
XML::elementName := x -> extop(x, 2):

// the attributes stored in a list
XML::attributes := x -> extop(x, 3):

XML::attributString := proc(x)
                        local tab, str, tabNops, i;
                       begin
                        assert(domtype(extop(x, 3)) = DOM_LIST);
                        
                        tab := extop(x, 3);
                        tabNops := nops(tab);
                        str := "";
                       for i from 1 to tabNops do
                          assert(domtype(op(tab, [i, 1]))=DOM_STRING);
                          assert(domtype(op(tab, [i, 2]))=DOM_STRING);
                          str := str.op(tab, [i, 1])."='".
                                 op(tab, [i, 2]);
                          if i < tabNops then
                            str := str."' ";
                          else
                            str := str."'";
                          end_if;
                        end_for;
                        str;
                      end_proc:

XML::op := x -> (if args(0) = 1 then
                   extop(x, 4..extnops(x))
                 else
                   op(extop(x, 4..extnops(x)), args(2))
                 end_if:
                 if % = FAIL then null() else % end):
XML::nops := x -> extnops(x)-3:


/** other utility methods:
 */

XML::namespace := "":
XML::setNamespace :=
  proc(ns : DOM_STRING)
    local oldNS;
  begin
    oldNS := XML::namespace;
    sysassign(XML::namespace, ns);
    oldNS
  end_proc:

// mask some special characters
XML::str2cdata := s -> 
                 (s := stringlib::subs(s, "&"="&amp;");
                  s := stringlib::subs(s, "<"="&lt;");
                  s := stringlib::subs(s, ">"="&gt;");
                  stringlib::collapseWhitespace(s)
                  ):

// mask some special characters
// do not collapse white spaces
XML::multiLineStr2cdata := s -> 
                 (s := stringlib::subs(s, "&"="&amp;");
                  s := stringlib::subs(s, "<"="&lt;");
		  s := stringlib::subs(s, ">"="&gt;");
		  s := stringlib::subs(s, "\\"="\\\\");
		  s := stringlib::subs(s, "\n"="\\n");
		  s := stringlib::subs(s, "\b"="\\n");
		  s := stringlib::subs(s, "\t"="\\t");
		  s := stringlib::subs(s, "\r"="");
                  ):

// translates into a MuPAD expression;  still to be implemented
XML::expr := ()->error("not yet available"):

// full expose;  returns a string
XML::expr2text := ()->stringlib::collapseWhitespace(op(dom::print(args()))):

/*
 *  print redefined to expose later on!
 */
// no full expose;  something short
XML::print := 
proc(x : Type::Union(XML, DOM_INT, DOM_IDENT, DOM_FLOAT, DOM_STRING))
  local result, attributes, operands, ns;
begin
  if domtype(x) <> dom then
    pr(x); return()
  end_if;
  case dom::markupType(x)
    of "document" do
      // print all document entries, but no encosure
      result := "XML::Document(...)";
      break;

    of "element" do
      if dom::namespace <> "" then
        ns := (dom::namespace, ":")
      else
        ns := ""
      end_if;
      attributes := dom::attributString(x);
      if attributes <> "" then
        attributes := " ", attributes;
      end_if;
      operands := [op(x)];
      if nops(operands) = 0 then
        // empty element
        result := _concat("<", ns, dom::elementName(x), attributes, "/>");
      else
        result := _concat("<", ns, dom::elementName(x), attributes,
                          ">\n  ...\n</", ns, dom::elementName(x), ">");
      end_if;
      break;
    of "entity" do
      assert(domtype(XML::op(x)) = DOM_STRING);
      result := dom::op(x);
      break;
      
    of "comment" do
      // print comment
      assert(domtype(XML::op(x)) = DOM_STRING);
      result := _concat("<!--", dom::op(x), "-->");
      break;
      
    of "DOCTYPE" do
      assert(domtype(XML::attributes(x)) = DOM_STRING);
      result := _concat("<!DOCTYPE ... [ ... ]>");
      break;
      
    of "PI" do
      // print Processing instruction
      assert(domtype(XML::op(x)) = DOM_STRING);
      result := _concat("<?", dom::op(x), "?>");
      break;
      
    of "CDATA" do
      // print CDATA
      assert(domtype(XML::op(x)) = DOM_STRING);
      result := _concat("<![CDATA[", dom::op(x), "]]>");
      break;

    of "ELEMENT" do
    of "ATTLIST" do
    of "ENTITY" do
    of "NOTATION" do
      // declarations: trivial case, markup is represented by a string
      assert(domtype(XML::op(x)) = DOM_STRING);
      result := _concat("<!", dom::markupType(x), " ", dom::op(x), ">");
      break;
      
    otherwise
      error("Unexpected type");
  end_case;
  result;
end_proc:


// side effect print to XML::OutputFile
XML::expose :=
proc(x : Type::Union(XML, DOM_INT, DOM_IDENT, DOM_FLOAT, DOM_STRING))
  local DOM, doExpose, indent, indentPr, noindentPr, prNoNL, str, stri, i;
begin
  DOM := dom;
  indent := 0;

  if XML::OutputFile = 0 then
    str := table();
    stri := 0;
    indentPr := () -> (str[(stri := stri +1)] := (("  " $ indent), args(), "\n"));
    noindentPr := () -> (str[(stri := stri +1)] := (args()));
    prNoNL := () -> (str[(stri := stri +1)] := (("  " $ indent), args()));
  else
    indentPr := () -> fprint(Unquoted, XML::OutputFile,
                             ("  " $ indent), args());
    noindentPr := () -> fprint(NoNL, XML::OutputFile, args());
    prNoNL := () -> fprint(NoNL, XML::OutputFile, ("  " $ indent), args());
  end_if;

  doExpose :=
  proc(x : Type::Union(XML, DOM_INT, DOM_IDENT, DOM_FLOAT, DOM_STRING),
       pr = indentPr : DOM_PROC)
    local attributes, noBreak, operands, ns;
  begin
    noBreak := {"ci", "cn", "mi", "mo", "mn", "OMI", "OMSTR"};
    if domtype(x) <> DOM then
      if domtype(x) <> DOM_STRING then
        pr(expr2text(x)); return()
      else
        pr(x); return()
      end_if;
    end_if;
    case DOM::markupType(x)
      of "document" do
        // print all document entries, but no encosure
        map([op(x)], doExpose);
        break;
        
      of "element" do
        if DOM::namespace <> "" then
          ns := (DOM::namespace, ":")
        else
          ns := ""
        end_if;
        attributes := DOM::attributString(x);
        if attributes <> "" then
          attributes := " ", attributes;
        end_if;
        operands := [op(x)];
        if nops(operands) = 0 then
          // empty element
          pr("<", ns, DOM::elementName(x), attributes, "/>");
        else
          if contains(noBreak, DOM::elementName(x)) then
            // just one line
            prNoNL("<", ns, DOM::elementName(x), attributes, ">");
            indent := indent + 1;
            map(operands, doExpose, noindentPr);
            indent := indent - 1;
            noindentPr("</", ns, DOM::elementName(x), ">\n");
          else
            pr("<", ns, DOM::elementName(x), attributes, ">");
            indent := indent + 1;
            map(operands, doExpose);
            indent := indent - 1;
            pr("</", ns, DOM::elementName(x), ">");
          end_if;
        end_if;
        break;

      of "entity" do
        assert(domtype(op(x)) = DOM_STRING);
        pr(dom::op(x));
        break;

      of "comment" do
        // print comment
        assert(domtype(XML::op(x)) = DOM_STRING);
        pr("<!--", DOM::op(x), "-->");
        break;
        
      of "DOCTYPE" do
        assert(domtype(XML::attributes(x)) = DOM_STRING);
        pr("<!DOCTYPE ", DOM::attributes(x), " [");
        indent := indent + 1;
        map([op(x)], doExpose);
        indent := indent - 1;
        pr("]>");
        break;
        
      of "PI" do
        // print Processing instruction
        assert(domtype(XML::op(x)) = DOM_STRING);
        pr("<?", DOM::op(x), "?>");
        break;
        
      of "CDATA" do
        // print CDATA
        assert(domtype(XML::op(x)) = DOM_STRING);
        pr("<![CDATA[", DOM::op(x), "]]>");
        break;

      of "ELEMENT" do
      of "ATTLIST" do
      of "ENTITY" do
      of "NOTATION" do
        // declarations: trivial case, markup is represented by a string
        assert(domtype(XML::op(x)) = DOM_STRING);
        pr("<!", DOM::markupType(x), " ", DOM::op(x), ">");
        break;
        
      otherwise
        error("Unexpected type");
    end_case;
    null()
  end_proc:
  doExpose(x);
  if XML::OutputFile = 0 then
    // concat the result string and create a stdlib::Exposed Object
    stdlib::Exposed(_concat(str[i] $ i=1..stri));
  else
    null()
  end_if;
end_proc:
XML::print := XML::expose:

// file descriptor for expose output
XML::OutputFile := 0:

/**
 *   XML::openOutputFile :  this function opens an output file.
 *
 *     All following expose calls of adt::XML elements will write
 *     into this file until it is closed with XML::closeOutputFile.
 *
 *   Argument :     file : DOM_STRING  - the name of the file 
 *                  If the string is empty an temporary file is opened.

 *   Return value : TRUE   iff the file was successfully opened
 *                  FALSE  iff the file could not be opened
 */
XML::openOutputFile :=
proc(file : DOM_STRING) : DOM_BOOL
  local fd;
begin
  if file = "" then
  	fd := fopen(Text, TempFile, Write);
	else
  	fd := fopen(Text, file, Write);
	end;
  if fd <> FAIL then
    dom::OutputFile := fd;
    return(TRUE)
  end;
  return(FALSE)
end:


/**
 *   XML::closeOutputFile :  closes the file opened by
 *                           XML::openOutputFile.
 *
 *   Argument :      -
 *
 *   Return value :  (0   -- (which is the new value of XML::OutputFile))
 *                   Testwise: the name of the just closed file.
 */
XML::closeOutputFile :=
proc()
  local a;
begin
  if dom::OutputFile <> 0 then
    a := fname(dom::OutputFile);
     fclose(dom::OutputFile)
     else
     a:=0
  end;
  dom::OutputFile := 0:
  a
end:


unalias(XML):
// end of file
