// combinat::partitions(n) returns the number of integer partitions of n 
combinat::partitions:=proc(n) local k;
begin
   if testargs() then
   if args(0) <> 1 then
      error("Wrong number of arguments") ;
   end_if ;
   if type(n) <> DOM_INT or n < 0 then
      error("Argument must be a nonnegative integer") ;
   end_if ;
   end_if ;

   if n<100 then // use Euler's pentagonal formula 
      // the following avoids stacks problems under MACOS 
      for k from n mod 25 to n step 25 do combinat::Euler_parts(k) end_for
   else // use Hardy-Ramanujan-Rademacher formula 
      combinat::HRRpartitions(n)
   end_if
end_proc:

/* the following procedure uses Euler's pentagonal formula

  p(n) + sum((-1)^k*(p(n-w(k))+p(n-w(-k))),k=1..infinity) = 0

  where w(k)=(3*k^2+k)/2. The asymptotic expansion of p(n) is:

                             1/2    /    / 2 n \1/2 \
                            3    exp| PI | --- |    |
                                    \    \  3  /    /
                     p(n) ~ -------------------------
                                      12 n

  Reference: G. Andrews, The Theory of Partitions, Addison-Wesley, 1976.
*/
combinat::Euler_parts:=proc(n) local s,t,k;
option remember;
begin
       s := 0;
       for k from 1 to n do
            t := (3*k^2-k)/2;
            if n < t then return(abs(s)) end_if;
            s := combinat::Euler_parts(n-t)-s;
            t := t+k;
            if n < t then return(abs(s)) end_if;
            s := combinat::Euler_parts(n-t)+s
        end_for;
end_proc:
combinat::Euler_parts(0):=1:

/* 
combinat::HRRpartitions(n) computes the number of partitions of n using
Hardy-Ramanujan-Rademacher formula. It is more efficient than Euler's
pentagonal formula up from n=100.
*/

combinat::HRRpartitions := proc(n) 
local s,k,t,f,x,X,C,zeroes,Akn,r,rt,ok;
save DIGITS;
begin
   // first compute an approximation to set DIGITS 
   s:=float(1/(4*n*sqrt(3))*exp(PI*sqrt(2*n/3)));
   s:=ln(s)/ln(10.0);
   DIGITS:=ceil(s+ln(s)+10);
   C:= float(1/PI/sqrt(2));
   X:= genident();
   x:= genident();
   f:= subs( diff(sinh(PI/X*(2/3*(x-1/24))^(1/2))/(x-1/24)^(1/2),x),x=n);

   s:=0; zeroes:=1; r:=0.0; ok:=0;
   k:=1; repeat
     if igcd(zeroes,k)=1 then
	Akn:=combinat::HRR_A(k,n);
	if Akn=0 then zeroes:=zeroes*k
	else
	   t:=r+C*Re(float( Akn*k^(1/2)*subs(f,X=k,EvalChanges) ));
           rt:=round(t); s:=s+rt; r:=t-rt;
	   DIGITS:=ceil(0.4343*ln(abs(t))+10);
           userinfo(1,"iteration ".k." last term=".expr2text(rt)." rest=".
		expr2text(r));
           if rt=0 and abs(r)<0.1 then ok:=ok+1 else ok:=0 end_if
        end_if
     end_if;
     k:=k+1;
   until ok=3 end_repeat;
   s
end_proc:

combinat::HRR_A := proc(k,n) local s,h;
begin
   s:=0;
   for h from 0 to k-1 do
      if igcd(h,k)=1 then
         combinat::HRR_omega(h,k)/2-n*h/k;
	 s:=s+(if type(%)=DOM_INT then 1 else
		cos(2*PI*mods(op(%))/op(%,2)) end_if)
      end_if
   end_for;
   s
end_proc:

combinat::HRR_omega:=proc(h,k) local mu;
begin
   _plus((frac(mu/k)-1/2)*(frac(h*mu/k)-1/2)$mu=1..k-1)
end_proc:

