/*
    Cat::Matrix  --  represents the category of matrices

    Cat::Matrix(R)

    R - the coefficient domain (a Cat::Ring)

    The category Cat::Matrix(R) represents all matrices over 
    the Cat::Ring R.

    The method "dimen" has to return the dimension of a matrix 
    in form of a list [nrows,ncols].
    The method "new" of a respected domain of these category has 
    to define a m times n matrix by "this::new( m,n )". This is 
    necessary for method "_mult" of this category to define a 
    new matrix of a certain dimension.

    The methods "_index", "set_index" and "matdim" need the internal
    representation of a matrix. Therefore these methods have to be 
    implemented in a respected domain.
*/


category Cat::Matrix( R )
    category Cat::BaseCategory;

    // basis methods:
    new; _index; set_index; matdim;

    // entries: 
    coeffRing := R;


    // Default implementation of standard methods

    nrows :=
    proc(M: dom) : Type::NonNegInt
    begin
	dom::matdim(M)[1];
    end_proc;

    ncols :=
    proc(M: dom) : Type::NonNegInt
    begin
	dom::matdim(M)[2];
    end_proc;

    _plus:= proc(x)
        local y, i, j, d;
    begin
        case args(0)
        of 0 do return( 0 )
        of 1 do return( x )
        otherwise
            d:= x::dom::matdim( x );
            for y in [args(2..args(0))] do
                if y::dom <> dom then
                    return( FAIL )
                elif y::dom::matdim( y ) <> d then
                    error("matrix dimensions don't match")
                else
                    ((x[i,j]:= R::_plus( x[i,j],y[i,j] )) $ j=1..d[2]) $ i=1..d[1];
                end_if
            end_for;

            return( x )
        end_case
    end_proc;

    _negate:= proc(x) // FIXME: Use multcoeffs?
        local d, i, j;
    begin
        d:= x::dom::matdim( x );
        ((x[i,j]:= R::_negate( x[i,j] )) $ j=1..d[2]) $ i=1..d[1];

        return( x )
    end_proc;

    _subtract:= (x,y) -> dom::_plus( x, _negate(y) );

    iszero:= proc(x)
        local i, j, d;
    begin
        d:= x::dom::matdim( x );
        for i from 1 to d[1] do
            for j from 1 to d[2] do
                if not R::iszero( x[i,j] ) then return( FALSE ) end_if
            end_for
        end_for;

        return( TRUE )
    end_proc;

    equal:= proc(x,y)
    begin
        if dom::matdim(x) <> dom::matdim(y) then 
            return( FALSE )
        else
            return( dom::iszero(dom::_plus( x,dom::_negate(y) )) )
        end_if
     end_proc;

    row:= proc(x,p)
        local d, r, j;
    begin
        d:= x::dom::matdim( x );
        r:= x::dom::new( 1,d[2] );
        (r[1,j]:= x[p,j]) $ j=1..d[2];

        return( r )
    end_proc;

    col:= proc(x,p)
        local d, c, i;
    begin
        d:= x::dom::matdim( x );
        c:= x::dom::new( d[1],1 );
        (c[i,1]:= x[i,p]) $ i=1..d[1];

        return( c )
    end_proc;

    delRow:= proc(x,p)
        local i, j, ii, d, rx;
    begin
        d:= x::dom::matdim( x );
        if d[1] = 1 then return( NIL ) end_if;

        rx:= x::dom::new( d[1]-1,d[2] );
        ii:= 1;
        for i from 1 to d[1] do
            if i <> p then
                (rx[ii,j]:= x[i,j]) $ j=1..d[2];
                ii:= ii + 1
            end_if
        end_for;

        return( rx )
    end_proc;

    delCol:= proc(x,p)
        local i, j, jj, d, cx;
    begin
        d:= x::dom::matdim( x );
        if d[2] = 1 then return( NIL ) end_if;

        cx:= x::dom::new( d[1],d[2]-1 );
        jj:= 1;
        for j from 1 to d[2] do
            if j <> p then
                (cx[i,jj]:= x[i,j]) $ i=1..d[1];
                jj:= jj + 1
            end_if
        end_for;

        return( cx )
    end_proc;

    swapRow:= proc(x,k,l)
        local d, j, t;
    begin
        d:= x::dom::matdim( x );
        for j from 1 to d[2] do
            t:= x[k,j]; x[k,j]:= x[l,j]; x[l,j]:= t
        end_for;

        return( x )
    end_proc;

    swapCol:= proc(x,k,l)
        local d, i, t;
    begin
        d:= x::dom::matdim( x );
        for i from 1 to d[1] do
            t:= x[i,k]; x[i,k]:= x[i,l]; x[i,l]:= t
        end_for;

        return( x )
    end_proc;

    setRow:= proc(x,p,y)
        local d, j;
    begin
        d:= x::dom::matdim(x);
        (x[p,j]:= y[1,j]) $ j=1..d[2];

        return( x )
    end_proc;

    setCol:= proc(x,p,y)
        local d, i;
    begin
        d:= x::dom::matdim(x);
        (x[i,p]:= y[i,1]) $ i=1..d[1];
        
        return( x )
    end_proc;

    transpose:= proc(x)
        local d, i, j, t;
    begin
        d:= x::dom::matdim( x );
        t:= x::dom::new( d[2],d[1] );
        ((t[j,i]:= x[i,j]) $ j=1..d[2]) $ i=1..d[1];

        return( t )
    end_proc;

    concatMatrix:= proc(x,y)
        local dx, dy, r, cm, i, j;
    begin
        dx:= x::dom::matdim( x );
        dy:= y::dom::matdim( y );
        if dx[1] <> dy[1] then
            error("incompatible operands")
        end_if;

        r:= dx[1];
        cm:= x::dom::new( r,dx[2]+dy[2] );
        ((cm[i,j]:= x[i,j]) $ j=1..dx[2]) $ i=1..r;
        ((cm[i,j+dx[2]]:= y[i,j]) $ j=1..dy[2]) $ i=1..r;

        return( cm )
    end_proc;

    stackMatrix:= proc(x,y)
        local dx, dy, c, cm, i, j;
    begin
        dx:= x::dom::matdim( x );
        dy:= y::dom::matdim( y );
        if dx[2] <> dy[2] then
            error("incompatible operands")
        end_if;

        c:= dx[2]; 
        cm:= x::dom::new( dx[1]+dy[1],c );
        ((cm[i,j]:= x[i,j]) $ j=1..c) $ i=1..dx[1];
        ((cm[i+dx[1],j]:= y[i,j]) $ j=1..c) $ i=1..dy[1];

        return( cm )
    end_proc;

    identity:= 
    proc(n)
        local x, i;
    begin
        if args(0) <> 1 then 
            error("expecting one argument")
        else
            if not testtype( n,Type::PosInt ) then
                error("expecting a positive integer")
            end_if
        end_if;

        x:= dom::new( n,n );
        (x[i,i]:= R::one) $ i=1..n;
        return( x )
    end_proc;


      
    /*
        solving a matrix (a_{i, j}) means solving the system a_{i, j} = 0
    */
    solve:=
    proc(A)
      local i, j, dim;
    begin
      dim:= dom::matdim(A);
      solve({A[i, j] $i=1..dim[1] $j=1..dim[2]}, args(2..args(0)))
    end_proc;
    
Content_ := DOM_ARRAY::Content_(expr2text(dom));
Content := proc(Out, data)
             save PRETTYPRINT;
           begin
             PRETTYPRINT := TRUE;
             dom::Content_(Out, dom::print(data));
           end;

   
    
// -----------------------------------------
//          body of the category
// -----------------------------------------
begin
    if args(0) <> 1 then error("wrong no of args") end_if;
    if R::dom <> DOM_DOMAIN or R::hasProp(Cat::Ring) <> TRUE then
        error("expecting a coefficient domain of category 'Cat::Ring'")
    end_if
end_category:

