//    
/*++
Cat::Polynomial -- the category of polynomials

Cat::Polynomial(R)

R - coefficient ring

Methods:
indets(p)        - returns list of indeterminates of p
mainvar(p)	 - returns the main variable of p
degree(p)        - returns total degree of p
degree(p,x)      - returns degree of variable x
degreevec(p)     - returns exponent list of leading term
evalp(p,x=v...)  - evaluates p at x = v
lcoeff(p)        - returns leading coefficient of p
lmonomial(p)     - returns leading monomial of p
lterm(p)         - returns leading term of p
nterms(p)        - returns number of non-zero terms of p
nthterm(p,n)     - returns term number n of p, the order may be not represent
		   the term ordering given by 'lcoeff'.
nthcoeff(p,n)    - returns coefficient number n of p
nthmonomial(p,n) - returns monomial number n of p
tcoeff(p)        - returns the trailing coefficient of p
coeff(p)         - returns a list of all non-zero coefficients
coeff(p,x,n)     - returns the coefficient of x^n
coeff(p,n)       - returns the coefficient of x^n where x is the main variable
multcoeffs(p,c)  - multiplies the coefficients of p by c
mapcoeffs(p,f,a...) - replaces the coefficients ci of p by f(ci,a,...)
content(p)       - returns the content of p
primpart(p)      - returns the primitive part of p
solve(p,x[,...]) - finds roots of p

Entries:
coeffRing	 - the coefficient ring
++*/

category Cat::Polynomial( R )
    category 
        if R::hasProp(Cat::FactorialDomain) then
	    Cat::FactorialDomain
        elif R::hasProp(Cat::GcdDomain) then
	    Cat::GcdDomain
        elif R::hasProp(Cat::IntegralDomain) then
	    Cat::IntegralDomain
	end_if,
      Cat::PartialDifferentialRing,
      Cat::Algebra(R) ;
    axiom  if R::hasProp(Ax::canonicalUnitNormal) then
	     Ax::canonicalUnitNormal
	   end_if ,
           if R::hasProp(Ax::closedUnitNormals) then
	     Ax::closedUnitNormals
    	   end_if ;

    indets; mainvar; degree; degreevec; lcoeff; lmonomial;
    lterm; nterms; nthterm; nthcoeff; nthmonomial; tcoeff;
    coeff; multcoeffs; mapcoeffs; evalp;

    coeffRing := R;

    characteristic := R::characteristic;
    
    content := if R::hasProp(Cat::Field) then
	R::one
    elif R::hasProp(Cat::GcdDomain) then
	R::gcd @ dom::coeff
    end_if;

    primpart := if R::hasProp(Cat::Field) then
	dom::unitNormal
    elif R::hasProp(Cat::GcdDomain) then
	proc(x) begin
	    dom::unitNormal(mapcoeffs(x, R::_divide, dom::content(x)))
	end_proc
    end_if;

    unitNormalRep := if R::hasProp(Ax::canonicalUnitNormal) then
	proc(x) local n; begin
	    n:= R::unitNormalRep(dom::lcoeff(x));
	    [ dom::multcoeffs(x, n[2]), dom::new(n[2]), dom::new(n[3]) ]
	end_proc
    end_if;

    unitNormal := if R::hasProp(Ax::canonicalUnitNormal) then
	proc(x) begin
	    dom::multcoeffs(x, (R::unitNormalRep(dom::lcoeff(x)))[2])
	end_proc
    end_if;

    isUnit := proc(x) begin
        if dom::degree(x) > 0 then FALSE
        else R::isUnit(dom::lcoeff(x))
        end_if
    end_proc;


    poly2list:=
    proc(x)
    begin
      // this always works, since poly2list(FAIL) = FAIL
      poly2list(poly(x))
    end_proc;
      
/*--
    solve( p, x[=T] [,options] )

    p      : polynomial of category this
    x      : (optional) DOM_IDENT
    T      : (optional) domain for x
    options: (optional) options for 'solve' (stdlib) command
--*/
   solve := proc(p,x)
           local T, o;
       begin
           if p::dom::hasProp( dom ) <> TRUE then 
               error("expecting polynomial of category Cat::PolyCat")
           elif args(0) = 1 then
               x:= p::dom::indets(p);
               if nops(x)=0 then
                   error("no variable given")
               elif nops(x) > 1 then
                   error("cannot solve one polynomial for several variables")
               else
                   x:= x[1];
                   o:= [args(2..args(0))]
               end_if
           else
               o:= [args(3..args(0))]
           end_if;

           if type(x) = "_equal" then
               if domtype(op(x,1)) <> DOM_IDENT then error("invalid unknown") end_if;
               if domtype(op(x,2)) <> DOM_DOMAIN then error("invalid computation domain") end_if;
               T:= op(x,2); x:= op(x,1)
           else
               if domtype(x) <> DOM_IDENT then error("invalid unknown") end_if;
               T:= R
           end_if;

           x:= solvelib::solve_poly( poly(p::dom::expr(p),[x]),x,op(o) );
           select( map(x,T::coerce),_not@_equal,FAIL )
       end_proc;

begin
      if args(0) <> 1 then error("wrong no of args") end_if;
      if R::hasProp(Cat::CommutativeRing) <> TRUE then
	error("no commutative ring")
      end_if;
end_category:
