//     

/*++
Dom::ExpressionField -- domain of arithmetical MuPAD expressions
                        supposed to be a field. 

Dom::ExpressionField( [ Normal [, IsZero ]] )

Normal - function used to normalize intermediate expressions,
         must be of type Expression -> Expression
IsZero - function used to detect 0, must be of type Expression -> Boolean
         and return TRUE iff the argument is zero

This domain assumes that the expressions occuring are valid representations
of field elements; the field is assumed to be of characteristic 0. The
function 'Normal' is used to normalize an expression, the function 'IsZero'
is used to test an expression for 0.

The call 'Dom::ExpressionField()' creates a domain which is similar to 
the domain 'Dom::ArithmeticalExpression'. In this case only the standard 
simplifications are done. Some methods are defined additionaly (see
below).

If 'IsZero' is missing then 'iszero @ Normal' is used instead. 

If a normalizing function other than 'id' is given it is assumed that
this functions returns a normal form where the zero element is uniquely
represented by the constant 0.

The elements of this domain are created by 'new' and have one operand:
the expression representing the element of the field.

Examples:

Dom::ExpressionField( normal ) creates a field of rational expressions over
the rationals where the intermediate expressions occuring are normalized
by 'normal'.

In Dom::ExpressionField( id, iszero @ normal ) the intermediate expressions
are not normalized, but 0 is detected correctly.


The following methods are defined additionaly:

conjugate(x)	-- computes the complex conjugate of x
Re(x)		-- computes the real part of x
Im(x)		-- computes the imaginary part of x
norm(x)		-- computes the norm of x 
numer(x)	-- return the numerator of x
denom(x)	-- return the denominator of x
sqrfree(x)	-- returns a sqrfree representation of x
factor(x)	-- factorize x
diff(...)	-- computes the derivative of an expression
max(...)	-- returns the maximum of the arguments
min(...)	-- returns the minimum of the arguments
++*/



domain Dom::ExpressionField(NORMAL, IsZero)
  inherits (if NORMAL = id and IsZero = iszero then Dom::ArithmeticalExpression
           else Dom::BaseDomain 
           end_if);
  category Cat::Field, Cat::DifferentialRing;
  axiom ( if NORMAL = id then
              Ax::efficientOperation("_divide"), 
              Ax::efficientOperation("_mult"),
              Ax::efficientOperation("_invert"),
              (if IsZero = iszero then Ax::systemRep end_if)
          else Ax::normalRep 
          end_if ), Ax::indetElements;

    Name:= if NORMAL = id and IsZero = iszero then hold(Dom::ExpressionField)() else FAIL end;

    
    new:= proc(x)
    begin
        if args(0) <> 1 then error("wrong no of args") end_if;
        x:= dom::coerce(x);
        if x = FAIL then 
            error("illegal argument")
        else
            x
        end_if
    end_proc;

    characteristic:= 0;
    
    convert:= if not dom::hasProp(Ax::systemRep) then
        proc(x)
        begin
            if x::dom = dom then 
                return( x )
            elif x::dom::constructor = Dom::ExpressionField then
                return( new(dom,NORMAL(extop(x,1))) )
            else
                x:= Dom::ArithmeticalExpression::convert(x);
               // x:= eval(expr(x)); // use eval here! E.g., expr(sin(R(0))) = sin(0)
                if x = FAIL or traperror((x:= NORMAL(x))) <> 0 then
                    return( FAIL )
                else
                    return( new(dom,x) )
                end_if
            end_if
        end_proc
    else 
        Dom::ArithmeticalExpression::convert
    end_if;

    convert_to:= if not dom::hasProp(Ax::systemRep) then
        proc(x,T)
        begin
            if domtype(T) <> DOM_DOMAIN then T:= domtype(T) end;

            if x::dom = T then 
                return(x)
            elif T::constructor = Dom::ExpressionField then
                return( T::new(extop(x,1)) )
            else
                x:= extop(x,1);
                if domtype(x) = T then 
                    return( x )
                else
                    return( T::convert(x) )
                end_if
            end_if
        end_proc 
    end_if;

    length:= if not dom::hasProp(Ax::systemRep) then
        x -> length(extop(x,1))
    else length end_if;

    indets:= if not dom::hasProp(Ax::systemRep) then
       x -> indets(extop(x,1),args(2..args(0)))
    else indets end_if;

    freeIndets:= if not dom::hasProp(Ax::systemRep) then
       x -> freeIndets(extop(x,1),args(2..args(0)))
    else freeIndets end_if;

    expr:= if not dom::hasProp(Ax::systemRep) then
       x -> expr(extop(x,1))
           else expr end_if;
      
    exprForPrint := if not dom::hasProp(Ax::systemRep) then
		      x -> extop(x, 1)
		    else id end_if;
      
    print:= generate::sortSums@dom::exprForPrint;
    Content := (Out, data) -> Out(generate::sortSums(dom::exprForPrint(data)));

    conjugate:= if not dom::hasProp(Ax::systemRep) then
        x -> new( dom,NORMAL(conjugate(extop(x,1))) )
    else conjugate end_if;

    Re:= if not dom::hasProp(Ax::systemRep) then
        x -> new( dom,NORMAL(Re(extop(x,1))) )
    else Re end_if;

    Im:= if not dom::hasProp(Ax::systemRep) then
        x -> new( dom,NORMAL(Im(extop(x,1))) )
    else Im end_if;

    norm:= if not dom::hasProp(Ax::systemRep) then 
        x -> new( dom,NORMAL(abs(extop(x,1))) )
    else abs end_if;

    abs:= if not dom::hasProp(Ax::systemRep) then
        x -> new( dom,NORMAL(abs(extop(x,1))) )
    else abs end_if;

    zero:= if not dom::hasProp(Ax::systemRep) then new(dom, 0) else 0 end_if;

    one:= if not dom::hasProp(Ax::systemRep) then new(dom, 1) else 1 end_if;

    iszero:= if not dom::hasProp(Ax::systemRep) then
        x-> IsZero(extop(x,1))
    else iszero end_if;

    equal:= if not dom::hasProp(Ax::systemRep) then
        proc(x,y)
        begin
            x:= NORMAL(extop(x,1)-extop(y,1));
            if testtype(x,Type::Numeric) then
                return(iszero(x))
            else
                return(is(x,Type::Zero))
            end_if
        end_proc
    else
        proc(x,y)
        begin
            x:= x - y;
            if testtype(x,Type::Numeric) then
                return(iszero(x))
            else
                return(is(x,Type::Zero))
            end_if
        end_proc
    end_if;

    _less:= if not dom::hasProp(Ax::systemRep) then
        (x,y) -> bool(_less( extop(x,1),extop(y,1) ))
    else _less end_if;

    _leequal:= if not dom::hasProp(Ax::systemRep) then
       (x,y) -> bool(_leequal( extop(x,1),extop(y,1) )) 
    else _leequal end_if;

    sign:= if not dom::hasProp(Ax::systemRep) then
        x -> new( dom,sign(extop(x,1)) )
    else sign end_if;

    _plus:= if not dom::hasProp(Ax::systemRep) then
        proc() local p, l;
        begin
            if map({args()}, domtype) = {dom} then
                new(dom, NORMAL(_plus(op(map([args()], extop, 1)))))
            else
                l:= map([args()], dom::coerce);
                if (p:= contains(l, FAIL)) = 0 then
                    new(dom, NORMAL(_plus(op(map(l, extop, 1)))))
                else
                    (args(p))::dom::_plus(
                        new(dom, NORMAL(_plus(map(op(l,1..p-1),extop,1))) ),args(p..args(0)) )
                end_if
             end_if
         end_proc
    else _plus end_if;

    _negate:= if not dom::hasProp(Ax::systemRep) then
        x -> new(dom, NORMAL(-extop(x, 1)))
    else _negate end_if;

    max:= if not dom::hasProp(Ax::systemRep) then
        (()->( new(dom,max(op(map([args()],extop,1)))) ))
    end_if;

    min:= if not dom::hasProp(Ax::systemRep) then
        (()->( new(dom,min(op(map([args()],extop,1)))) ))
    end_if;

    _subtract:= if not dom::hasProp(Ax::systemRep) then
        (x,y) -> new(dom, NORMAL(extop(x,1)-extop(y,1)) )
    else
        (x,y) -> x-y
    end_if;

    _mult:=
    if not dom::hasProp(Ax::systemRep) then
      proc()
        local p, l, coeffRings;
      begin
        if map({args()}, domtype) = {dom} then
          new(dom, NORMAL(_mult(op(map([args()], extop, 1)))))
        else
          l:= map([args()], dom::coerce);
          coeffRings:= map([args()], x -> x::dom::coeffRing);
          if (p:= contains(coeffRings, dom)) > 0 then
            (args(p))::dom::_mult(args(p), args(1..p-1),args(p+1..args(0)))
          elif (p:= contains(l, FAIL)) = 0 then
            new(dom, NORMAL(_mult(op(map(l, extop, 1)))))
          else
            (args(p))::dom::_mult
            (
             new(dom, NORMAL(_mult(map(op(l,1..p-1),extop,1))) ),
             args(p..args(0))
             )
          end_if
        end_if
      end_proc
    else
      _mult
    end_if;

    _invert:= if not dom::hasProp(Ax::systemRep) then
        x -> new(dom,NORMAL(1/extop(x,1)))
    else _invert end_if;

    _divide:= if not dom::hasProp(Ax::systemRep) then 
        (x,y) -> new(dom, NORMAL(extop(x,1) / extop(y,1)))
    else
        (x,y) -> x / y
    end_if;

    intmult:= if not dom::hasProp(Ax::systemRep) then
        (x,n) -> new(dom, NORMAL(extop(x,1) * n))
    else _mult end_if;

    _power:= (if not dom::hasProp(Ax::systemRep) then
    	proc(x,n) begin
    	    if domtype(x) <> dom then 
    	        x:= dom::coerce(x);
    	        if x = FAIL then error("Illegal power") end_if
     	    elif domtype(n) <> dom then 
    	        n:= dom::coerce(n);
    	        if n = FAIL then error("Illegal power") end_if
    	    end_if;
    	    new(dom, NORMAL(extop(x,1) ^ extop(n,1)))
    	end_proc
    else _power end_if);

    D:= if not dom::hasProp(Ax::systemRep) then
    	(()->((
    	    case args(0)
    	    of 1 do
    		new(dom, NORMAL(D(extop(args(1),1)))); break;
    	    of 2 do
    		new(dom, NORMAL(D(args(1), extop(args(2),1)))); break;
    	    otherwise error("wrong no of args");
    	    end_case
    	)))
    else D end_if;

    diff:= if not dom::hasProp(Ax::systemRep) then
        x -> new(dom, NORMAL(diff(extop(x,1), op(map([args(2..args(0))],expr)) )))
    else diff end_if;

    has:= if not dom::hasProp(Ax::systemRep) then
        x -> has(extop(x,1), args(2..args(0)))
    else has end_if;

    int:= if not dom::hasProp(Ax::systemRep) then
        proc(f,x)
            local r;
        begin 
            f:= extop(f,1);
            if type(x) = "_equal" then
                r:= op(x,2); x:= op(x,1);
                if domtype(x) = dom then x:= extop(x,1) end;
                if domtype(op(r,1)) = dom then r:= subsop(r,1=extop(op(r,1),1)) end;
                if domtype(op(r,2)) = dom then r:= subsop(r,2=extop(op(r,2),1)) end;
                x:= x = r;
            else
                if domtype(x) = dom then x:= extop(x,1) end;
            end;
            f:= int(f,x,args(3..args(0)));
            if (f:= dom::convert(f)) = FAIL then
                error("can't perform computation over this domain")
            else
                return(f)
            end_if
        end_proc
    else int end_if;

    limit:= if not dom::hasProp(Ax::systemRep) then
        proc(f,x)
            local r;
        begin 
            f:= extop(f,1);
            if type(x) = "_equal" then
                r:= op(x,2); x:= op(x,1);
                if domtype(x) = dom then x:= extop(x,1) end;
                if domtype(r) = dom then r:= extop(r,1) end;
                x:= x = r;
            else
                if domtype(x) = dom then x:= extop(x,1) end;
            end;
            f:= limit(f,x,args(3..args(0)));
            if (f:= dom::convert(f)) = FAIL then
                error("can't perform computation over this domain")
            else
                return(f)
            end_if
        end_proc
    else limit end_if;

    normal:= if not dom::hasProp(Ax::systemRep) then 
        x->extsubsop( x,1=NORMAL(extop(x,1)) )
    else 
        x->normal(x, Expand = FALSE);
    end_if;

    simplify:= if not dom::hasProp(Ax::systemRep) then
        x -> new(dom, simplify(extop(x,1),args(2..args(0))) )
    else simplify end_if;

    expand:= if not dom::hasProp(Ax::systemRep) then
        x -> new(dom, expand( extop(x,1) ) )
    else expand end_if;

    combine:= if not dom::hasProp(Ax::systemRep) then
        x -> new(dom, combine(extop(x,1),args(2..args(0))))
    else combine end_if;

    radsimp:= if not dom::hasProp(Ax::systemRep) then
        x -> new(dom, radsimp(extop(x,1)))
    else radsimp end_if;

    numer:= if not dom::hasProp(Ax::systemRep) then
        x -> new(dom, NORMAL(numer(extop(x,1))))
    else numer end_if;

    denom:= if not dom::hasProp(Ax::systemRep) then
        x -> new(dom, NORMAL(denom(extop(x,1))))
    else denom end_if;

    gcd:= if not dom::hasProp(Ax::systemRep) then
            proc()
            begin
              if select([args()], not iszero) = [] then
                dom::zero
              else
                dom::one
              end_if
            end_proc
          else
            gcd
          end_if;

    lcm:= (()->((if select([args()], iszero) <> [] then dom::zero else dom::one end_if)));

    factor:=
    if not dom::hasProp(Ax::systemRep) then
      proc(x)
        local i;
      begin
        x:= factor( extop(x,1) );
       // Anpassung an Umstellung von Factored::_index (Walter 29.12.04) 
        x:= coerce(x, DOM_LIST):
        x[1]:= new(dom, NORMAL(x[1]));
        ( x[2*i]:= new(dom, NORMAL(x[2*i])) ) $ i=1..nops(x) div 2;
        x:= Factored::create(x):
        x:= x::dom::setRing(x,dom);
        return(x)
      end_proc
    else
      factor
    end_if;

    sqrfree:= if not dom::hasProp(Ax::systemRep) then
        proc(x) local i;
        begin
            if dom::iszero(x) then return(dom::zero) end_if;

            x:= polylib::sqrfree( extop(x,1) );
            ( x[2*i]:= new(dom, NORMAL(x[2*i])) ) $ i=1..nops(x) div 2;
            x:= x::dom::setRing(x,dom);
            return(x)
        end_proc
    else
        polylib::sqrfree
    end_if;

    randomIdent:= genident("R");

    random:= if not dom::hasProp(Ax::systemRep) then
        proc() local t;
        begin
            t:= polylib::randpoly([dom::randomIdent], hold(Expr));
            if iszero(t) then t:= 0 else
                t:= expr(polylib::randpoly([dom::randomIdent], hold(Expr))) / expr(t)
            end_if;
            return(new(dom, NORMAL(t)))
        end_proc
    end_if;

    float:= (if not dom::hasProp(Ax::systemRep) then
        x -> new(dom, NORMAL(float(extop(x,1))))
    else float end_if);

    // overload numeric::rationalize
    numeric_rationalize:= (if not dom::hasProp(Ax::systemRep) then
                     x -> new(dom, NORMAL
                              (numeric::rationalize(extop(x,1),
                                                    args(2..args(0)))))
                   else
                     numeric::rationalize
                   end_if);
                     
    
    nops:= if not dom::hasProp(Ax::systemRep) then
        x -> nops(extop(x,1))
    else
        nops
    end_if;

    map:= if not dom::hasProp(Ax::systemRep) then
      proc(x)
      begin
        dom::new(map(extop(x,1), args(2..args(0))))
      end_proc;
    else
      map
    end_if;

    subs:= if not dom::hasProp(Ax::systemRep) then
        x -> new(dom, NORMAL(
                 subs(extop(x,1), args(2..args(0)))
             ))
    else subs end_if;

    op:= if not dom::hasProp(Ax::systemRep) then
      proc()
        local t;
        name ExpressionField::op;
      begin
        if args(0)=1 then
          map(op(extop(args(1),1)), dom::new)
        elif args(0) = 2 and has(args(2),0) then
          op(extop(args(1),1), args(2..args(0)))
        else
          t:= op(extop(args(1),1), args(2..args(0)));
          if t <> FAIL then
            new( dom, NORMAL(t) )
          else
            FAIL
          end_if
        end_if
      end_proc
    else
      op
    end_if;
    

     
    subsop:= if not dom::hasProp(Ax::systemRep) 
      then proc(x)
           begin
             // op is overloaded to convert operands to dom.
             // In calls such as subsop(.., [1, 1, 1] = newValue
             // the kernel subsop calls the overloaded op and
             // calls dom::subsop(..., 1 = something), where
             // 'something' is not an DOM_EXPR, but of type dom.
             // This is not what we want (the internal representation
             // consists of basic objects of type DOM_EXPR).
             // Use maprec to reconvert dom to DOM_EXPR
             x:=  misc::maprec(
                            subsop(extop(x,1), args(2..args(0))),
                            {dom} = (x -> extop(x, 1))
                              );
              new(dom, NORMAL(x)) 
            end_proc;
       else subsop 
       end_if;

    subsex:= if not dom::hasProp(Ax::systemRep) 
      then proc(x)
           begin
             // See subsop for an explanation
             // of the maprec construct
             x:=  misc::maprec(
                            subsex(extop(x,1), args(2..args(0))),
                            {dom} = (x -> extop(x, 1))
                              );
              new(dom, NORMAL(x)) 
            end_proc;
     else subsex 
     end_if;

    solve:=
    proc(x:dom)
    begin
        solve(expr(x), args(2..args(0)))
    end_proc;

    domsolve:=
    proc(eqs, var, options)
      local result, d, conv;
    begin
      d:=dom;
      // local method conv
      conv:= proc(a)
      begin
        case type(a)
          of DOM_LIST do
          of DOM_SET do
          of Dom::Multiset do
            return(map(a, conv))
          of "_equal" do
            return(lhs(a)=conv(rhs(a)))
          otherwise
            d(a)
        end_case;
      end_proc;
      
      options[Domain]:=hold(Expr);
      eqs:= misc::maprec(eqs, {DOM_POLY}=expr);
      result:= solve(eqs, var, options);
      if type(result)=DOM_SET or type(result)=Dom::Multiset then
        result:=map(result, conv)
      end_if;
      result
    end_proc;
      

    pivotSize:= if not dom::hasProp(Ax::systemRep) then
        x-> length(extop(x,1))
                else length end_if;

    // default_slot method for creating slots not defined elsewhere
      
    default_slot :=
    proc(d:DOM_DOMAIN, slotname)
    begin
      // the first argument is always the domain
      // Dom::ExpressionField(...) whose slot is to be created

      // although this is syntactically legal, we do not use
      // slots that are not strings
      if type(slotname) <> DOM_STRING then
        return(FAIL)
      end_if;

      // We only create slots for special functions
      // note that it would be very dangerous to remove
      // this restriction !!!
      
      if contains(specfunc, slotname) and
        not contains({"interface", "info", "exported"}, slotname)
        then
        (d::new)@eval(text2expr(slotname))@extop
      else
        FAIL
      end_if;  
    end_proc;
      
    TeXrep := x -> "\\mathbb{E}";

begin
  case args(0)
    of 0 do
      NORMAL:= id;
      IsZero:= iszero;
      break;
    of 1 do
      if NORMAL = id then
        IsZero:= iszero
      else 
       IsZero:= iszero @ NORMAL
      end_if;
      break;
    of 2 do
      break;
    otherwise
      error("wrong no of args");
  end_case;

end_domain:

